/**
 * @brief       HTS221.cpp
 * @details     Capacitive digital sensor for relative humidity and temperature.
 *              Function file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ). All rights reserved.
 */

#include "HTS221.h"


HTS221::HTS221 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq )
    : _i2c           ( sda, scl )
    , _HTS221_Addr ( addr )
{
    _i2c.frequency( freq );
}


HTS221::~HTS221()
{
}



/**
 * @brief       HTS221_GetDeviceID ( HTS221_data_t* )
 *
 * @details     It gets the device identification.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myDeviceID:    Device ID.
 *
 *
 * @return       Status of HTS221_GetDeviceID.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetDeviceID ( HTS221_data_t* myDeviceID )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_WHO_AM_I;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Mask it and update it with the new value  */
    myDeviceID->deviceID   =   cmd;



    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetResolution ( HTS221_data_t )
 *
 * @details     It set humidity and temperature resolution mode.
 *
 * @param[in]    myTempHumResolution: Humidity and Temperature resolution mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HTS221_SetResolution.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetResolution ( HTS221_data_t myTempHumResolution )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_AV_CONF;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Mask it and update it with the new value  */
    cmd[1]  &=  ~( AV_CONF_AVGT_MASK | AV_CONF_AVGH_MASK );
    cmd[1]  |=   ( myTempHumResolution.temperatureResolution | myTempHumResolution.humidityResolution );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_GetResolution ( HTS221_data_t* )
 *
 * @details     It get humidity and temperature resolution mode.
 *
 * @param[in]    N/A
 *
 * @param[out]   myTempHumResolution: Humidity and Temperature resolution mode
 *
 *
 * @return       Status of HTS221_GetResolution.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetResolution ( HTS221_data_t* myTempHumResolution )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_AV_CONF;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data  */
    myTempHumResolution->temperatureResolution   =   (HTS221_av_config_avgt_t)( cmd & AV_CONF_AVGT_MASK );
    myTempHumResolution->humidityResolution      =   (HTS221_av_config_avgh_t)( cmd & AV_CONF_AVGH_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetPowerDown ( HTS221_ctrl_reg1_pd_t )
 *
 * @details     It sets the device into power-down ( low-power mode ) or active mode.
 *
 * @param[in]    myPowerMode:     Power-Down/Active mode.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetPowerDown.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetPowerDown ( HTS221_ctrl_reg1_pd_t myPowerMode )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG1;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG1_PD_MASK ) | myPowerMode );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetBlockDataUpdate ( HTS221_data_t )
 *
 * @details     It sets the block data update.
 *
 * @param[in]    myBDU:           Block data update mode.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetBlockDataUpdate.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetBlockDataUpdate ( HTS221_data_t myBDU )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG1;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG1_BDU_MASK ) | myBDU.bdu );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_GetBlockDataUpdate ( HTS221_data_t* )
 *
 * @details     It gets the block data update.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myBDU:           Block data update mode
 *
 *
 * @return       Status of HTS221_GetBlockDataUpdate.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetBlockDataUpdate ( HTS221_data_t* myBDU )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_CTRL_REG1;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data  */
    myBDU->bdu   =   (HTS221_ctrl_reg1_bdu_t)( cmd & CTRL_REG1_BDU_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetOutputDataRate ( HTS221_data_t )
 *
 * @details     It sets the output data rate ( ODR ).
 *
 * @param[in]    N/A.
 * @param[in]    myODR:           Output data rate.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetOutputDataRate.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetOutputDataRate  ( HTS221_data_t myODR )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG1;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG1_ODR_MASK ) | myODR.odr );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_GetOutputDataRate ( HTS221_data_t* )
 *
 * @details     It gets the output data rate ( ODR ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myODR:           Output data rate
 *
 *
 * @return       Status of HTS221_GetOutputDataRate.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetOutputDataRate  ( HTS221_data_t* myODR )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_CTRL_REG1;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data   */
    myODR->odr   =   (HTS221_ctrl_reg1_odr_t)( cmd & CTRL_REG1_ODR_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetBoot ( void )
 *
 * @details     It sets reboot memory content.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetBoot.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         At the end of the boot process, the BOOT bit is set again to '0'.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetBoot  ( void )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG2;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG2_BOOT_MASK ) | CTRL_REG2_BOOT_REBOOT_MEMORY_CONTENT );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_GetBoot ( HTS221_data_t* )
 *
 * @details     It gets reboot memory content.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myBOOT:   Reboot memory content flag
 *
 *
 * @return       Status of HTS221_GetBoot.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetBoot ( HTS221_data_t* myBOOT )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_CTRL_REG2;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data   */
    myBOOT->boot   =   (HTS221_ctrl_reg2_boot_t)( cmd & CTRL_REG2_BOOT_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetHeater ( HTS221_data_t* )
 *
 * @details     It sets heater mode: Enabled/Disabled.
 *
 * @param[in]    N/A.
 * @param[in]    myHeater:   Heater mode.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetHeater.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetHeater ( HTS221_data_t myHeater )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG2;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG2_HEATER_MASK ) | myHeater.heater );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_GetHeater ( HTS221_data_t* )
 *
 * @details     It gets heater mode.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myHeater         Heater mode
 *
 *
 * @return       Status of HTS221_GetHeater.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetHeater ( HTS221_data_t* myHeater )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_CTRL_REG2;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data   */
    myHeater->heater   =   (HTS221_ctrl_reg2_heater_t)( cmd & CTRL_REG2_HEATER_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetOneShot ( void )
 *
 * @details     It sets one-shot, new data set.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetOneShot.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         ONE_SHOT bit comes back to '0' by hardware
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetOneShot ( void )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG2;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG2_ONE_SHOT_MASK ) | CTRL_REG2_ONE_SHOT_START );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_GetOneShot ( HTS221_data_t* )
 *
 * @details     It gets one-shot flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myOneShot:  One-shot flag
 *
 *
 * @return       Status of HTS221_GetOneShot.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetOneShot ( HTS221_data_t* myOneShot )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_CTRL_REG2;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data   */
    myOneShot->one_shot   =   (HTS221_ctrl_reg2_one_shot_t)( cmd & CTRL_REG2_ONE_SHOT_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetDataReadyOuput ( HTS221_ctrl_reg3_drdy_h_l_t )
 *
 * @details     It sets data ready output signal active high/low.
 *
 * @param[in]    N/A.
 * @param[in]    myDRDY_H_L:   Data ready output signal high/low.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetDataReadyOuput.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetDataReadyOuput ( HTS221_ctrl_reg3_drdy_h_l_t myDRDY_H_L )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG3;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG3_DRDY_H_L_MASK ) | myDRDY_H_L );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetSelectionOnPin3 ( HTS221_ctrl_reg3_pp_od_t )
 *
 * @details     It sets Push-pull/Open Drain selection on pin 3 ( DRDY ).
 *
 * @param[in]    N/A.
 * @param[in]    myDRDY:     Push-pull/Open Drain selection on pin 3 (DRDY).
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetDataReadyOuput.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetSelectionOnPin3 ( HTS221_ctrl_reg3_pp_od_t myDRDY )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG3;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG3_PP_OD_MASK ) | myDRDY );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}



/**
 * @brief       HTS221_SetDataReadyEnable ( HTS221_ctrl_reg3_drdy_en_t )
 *
 * @details     It sets data ready enable.
 *
 * @param[in]    N/A.
 * @param[in]    myDRDY_EN:    Data Ready enable.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HTS221_SetDataReadyEnable.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_SetDataReadyEnable ( HTS221_ctrl_reg3_drdy_en_t myDRDY_EN )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   HTS221_CTRL_REG3;
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]   =   ( ( cmd[1] & ~CTRL_REG3_DRDY_EN_MASK ) | myDRDY_EN );
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}


/**
 * @brief       HTS221_GetHumidityDataAvailable ( HTS221_data_t* )
 *
 * @details     It gets humidity data available flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myHumidityFlag:  Humidity data available flag
 *
 *
 * @return       Status of HTS221_GetHumidityDataAvailable.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetHumidityDataAvailable ( HTS221_data_t* myHumidityFlag )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_STATUS_REG;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data   */
    myHumidityFlag->h_da   =   (HTS221_status_reg_h_da_t)( cmd & STATUS_REGISTER_H_DA_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}


/**
 * @brief       HTS221_GetTemperatureDataAvailable ( HTS221_data_t* )
 *
 * @details     It gets temperature data available flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTemperatureFlag: Temperature data available flag
 *
 *
 * @return       Status of HTS221_GetTemperatureDataAvailable.
 *
 *
 * @author      Manuel Caballero
 * @date        24/May/2019
 * @version     24/May/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetTemperatureDataAvailable ( HTS221_data_t* myTemperatureFlag )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register   */
    cmd   =   HTS221_STATUS_REG;
    aux   =   _i2c.write ( _HTS221_Addr, &cmd, 1U, true );
    aux   =   _i2c.read  ( _HTS221_Addr, &cmd, 1U );

    /* Parse the data   */
    myTemperatureFlag->t_da   =   (HTS221_status_reg_t_da_t)( cmd & STATUS_REGISTER_T_DA_MASK );




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}


/**
 * @brief       HTS221_GetRawHumidity ( HTS221_data_t* )
 *
 * @details     It gets raw humidity.
 *
 * @param[in]    N/A
 *
 * @param[out]   myRawHumidity:     Raw humidity
 *
 *
 * @return       Status of HTS221_GetRawHumidity.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         This function uses autoincrementing for reading the registers.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetRawHumidity ( HTS221_data_t* myRawHumidity )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   ( HTS221_HUMIDITY_OUT_L | 0x80 );                                        // Autoincrementing
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

    /* Parse the data   */
    myRawHumidity->rawHumidity   =   cmd[1];
    myRawHumidity->rawHumidity <<=   8U;
    myRawHumidity->rawHumidity  |=   cmd[0];




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}


/**
 * @brief       HTS221_GetRawTemperature ( HTS221_data_t* )
 *
 * @details     It gets raw temperature.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRawTemperature:  Raw temperature
 *
 *
 * @return       Status of HTS221_GetRawTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         This function uses autoincrementing for reading the registers.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetRawTemperature ( HTS221_data_t* myRawTemperature )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   ( HTS221_TEMP_OUT_L | 0x80 );                                             // Autoincrementing
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );


    /* Parse the data   */
    myRawTemperature->rawTemperature   =   cmd[1];
    myRawTemperature->rawTemperature <<=   8U;
    myRawTemperature->rawTemperature  |=   cmd[0];




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}


/**
 * @brief       HTS221_GetCalibrationCoefficients ( HTS221_data_t* )
 *
 * @details     It gets calibration coefficients.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myCoeff:  Calibration coefficients
 *
 *
 * @return       Status of HTS221_GetCalibrationCoefficients.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         This function uses autoincrementing for reading the registers.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetCalibrationCoefficients ( HTS221_data_t* myCoeff )
{
    char     cmd[16]  = { 0U };
    uint32_t aux;

    /* Read the register   */
    cmd[0]   =   ( HTS221_CALIB_0 | 0x80 );                                             // Autoincrementing
    aux      =   _i2c.write ( _HTS221_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HTS221_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

    /* Parse the data   */
    myCoeff->h0_rH_x2    =   cmd[0];
    myCoeff->h1_rH_x2    =   cmd[1];
    myCoeff->t0_degC_x8  =   cmd[2];
    myCoeff->t1_degC_x8  =   cmd[3];
    myCoeff->t1_T0_msb   =   ( cmd[5] & 0x0F );
    myCoeff->h0_T0_OUT   =   cmd[7];
    myCoeff->h0_T0_OUT <<=   8U;
    myCoeff->h0_T0_OUT  |=   cmd[6];
    myCoeff->h1_T0_OUT   =   cmd[11];
    myCoeff->h1_T0_OUT <<=   8U;
    myCoeff->h1_T0_OUT  |=   cmd[10];
    myCoeff->t0_OUT      =   cmd[13];
    myCoeff->t0_OUT    <<=   8U;
    myCoeff->t0_OUT     |=   cmd[12];
    myCoeff->t1_OUT      =   cmd[15];
    myCoeff->t1_OUT    <<=   8U;
    myCoeff->t1_OUT     |=   cmd[14];

    /* Coefficient result for temperature & humidity  */
    myCoeff->t0_degC   =  ( myCoeff->t1_T0_msb & 0x03 );
    myCoeff->t0_degC <<=  8U;
    myCoeff->t0_degC  |=  myCoeff->t0_degC_x8;
    myCoeff->t0_degC >>=  3U;

    myCoeff->t1_degC   =   ( myCoeff->t1_T0_msb & 0x0C );
    myCoeff->t1_degC <<=   6U;
    myCoeff->t1_degC  |=   myCoeff->t1_degC_x8;
    myCoeff->t1_degC >>=  3U;

    myCoeff->h0_RH     =   myCoeff->h0_rH_x2;
    myCoeff->h0_RH   >>=  1U;

    myCoeff->h1_RH     =   myCoeff->h1_rH_x2;
    myCoeff->h1_RH   >>=  1U;




    if ( aux == I2C_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}


/**
 * @brief       HTS221_GetTemperature ( HTS221_data_t* )
 *
 * @details     It gets the current temperature in Celsius degrees.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTemperature:     Current temperature in Celsius degress
 *
 *
 * @return       Status of HTS221_GetTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetTemperature ( HTS221_data_t* myTemperature )
{
    HTS221::HTS221_status_t aux;

    /* Get temperature  */
    aux  =   HTS221::HTS221_GetRawTemperature ( &(*myTemperature) );

    /* Parse the data   */
    myTemperature->temperature   =   (int32_t)( 10.0*( myTemperature->rawTemperature - myTemperature->t0_OUT )*( myTemperature->t1_degC - myTemperature->t0_degC )/( myTemperature->t1_OUT - myTemperature->t0_OUT ) );
    myTemperature->temperature  +=   10.0*myTemperature->t0_degC;
    myTemperature->temperature  /=   10.0;



    if ( aux == HTS221_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}


/**
 * @brief       HTS221_GetHumidity ( HTS221_data_t* )
 *
 * @details     It gets the current humidity value.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myHumidity:        Current humidity value
 *
 *
 * @return       Status of HTS221_GetHumidity.
 *
 *
 * @author      Manuel Caballero
 * @date        31/May/2019
 * @version     31/May/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HTS221::HTS221_status_t HTS221::HTS221_GetHumidity ( HTS221_data_t* myHumidity )
{
    HTS221::HTS221_status_t aux;

    /* Get humidity  */
    aux  =   HTS221::HTS221_GetRawHumidity ( &(*myHumidity) );

    /* Parse the data   */
    myHumidity->humidity   =   (int32_t)( 10.0*( myHumidity->rawHumidity - myHumidity->h0_T0_OUT )*( myHumidity->h1_RH - myHumidity->h0_RH )/( myHumidity->h1_T0_OUT - myHumidity->h0_T0_OUT ) );
    myHumidity->humidity  +=   10.0*myHumidity->h0_RH;
    myHumidity->humidity  /=   10.0;

    /* Check if it is saturated  */
    if ( myHumidity->humidity > 100 ) {
        myHumidity->humidity   =   100;
    }



    if ( aux == HTS221_SUCCESS ) {
        return   HTS221_SUCCESS;
    } else {
        return   HTS221_FAILURE;
    }
}