/**
 * @brief       HDC2080.cpp
 * @details     Low-Power Humidity and Temperature Digital Sensor.
 *              Functions file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to Nimbus Centre ( https://www.nimbus.cit.ie ).
 */

#include "HDC2080.h"


HDC2080::HDC2080 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq )
    : _i2c          ( sda, scl )
    , _HDC2080_Addr ( addr )
{
    _i2c.frequency( freq );
}


HDC2080::~HDC2080()
{
}


/**
 * @brief       HDC2080_GetRawTemperature ( HDC2080_data_t* )
 *
 * @details     It gets the raw temperature.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRawTemperature:  Raw temperature.
 *
 *
 * @return       Status of HDC2080_GetRawTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetRawTemperature  ( HDC2080_data_t* myRawTemperature )
{
    char     cmd[2] = { 0 };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_TEMPERATURE_LOW;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );


    /* Parse data   */
    myRawTemperature->rawTemperature     =   cmd[1];
    myRawTemperature->rawTemperature   <<=   8U;
    myRawTemperature->rawTemperature    |=   cmd[0];



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetTemperature ( HDC2080_data_t* )
 *
 * @details     It gets the current temperature.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTemperature:     Current temperature.
 *
 *
 * @return       Status of HDC2080_GetTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetTemperature  ( HDC2080_data_t* myTemperature )
{
    HDC2080_status_t aux;

    /* Read the register */
    aux      =   HDC2080_GetRawTemperature  ( &(*myTemperature) );

    /* Parse data   */
    myTemperature->temperature  =   ( ( myTemperature->rawTemperature ) / 65536.0 ) * 165.0 - 40.0;


    return   aux;
}



/**
 * @brief       HDC2080_GetRawHumidity ( HDC2080_data_t* )
 *
 * @details     It gets the raw humidity.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRawTemperature:  Raw humidity.
 *
 *
 * @return       Status of HDC2080_GetRawHumidity.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetRawHumidity  ( HDC2080_data_t* myRawHumidity )
{
    char     cmd[2] = { 0 };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_HUMIDITY_LOW;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );


    /* Parse data   */
    myRawHumidity->rawHumidity   =   cmd[1];
    myRawHumidity->rawHumidity <<=   8U;
    myRawHumidity->rawHumidity  |=   cmd[0];



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetHumidity ( HDC2080_data_t* )
 *
 * @details     It gets the current humidity.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTemperature:     Current humidity.
 *
 *
 * @return       Status of HDC2080_GetTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetHumidity  ( HDC2080_data_t* myHumidity )
{
    HDC2080_status_t aux;

    /* Read the register */
    aux      =   HDC2080_GetRawHumidity  ( &(*myHumidity) );

    /* Parse data   */
    myHumidity->humidity    =   ( ( myHumidity->rawHumidity ) / 65536.0 ) * 100.0;


    return   aux;
}



/**
 * @brief       HDC2080_GetInterrupt_DRDY_Status ( HDC2080_data_t* )
 *
 * @details     It gets the interrupt DRDY status.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myStatus:          Interrupt DRDY status.
 *
 *
 * @return       Status of HDC2080_GetInterrupt_DRDY_Status.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetInterrupt_DRDY_Status ( HDC2080_data_t* myStatus )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_INTERRUPT_DRDY;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );


    /* Parse data   */
    myStatus->interrupt_drdy_status  =   (HDC2080_interrupt_drdy_drdy_status_t)( INTERRUPT_DRDY_DRDY_STATUS_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_ConfTemperatureMax ( HDC2080_data_t )
 *
 * @details     It configures the temperature MAX register.
 *
 * @param[in]    myTemperatureMax:  Temperature max raw value.
 *
 * @param[out]   N/A
 *
 *
 * @return       Status of HDC2080_ConfTemperatureMax.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_ConfTemperatureMax ( HDC2080_data_t myTemperatureMax )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_TEMPERATURE_MAX;
    cmd[1]   =   myTemperatureMax.rawTemperature_max;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );


    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_ConfHumidityMax ( HDC2080_data_t )
 *
 * @details     It configures the humidity MAX register.
 *
 * @param[in]    myTemperatureMax:  Humidity max raw value.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_ConfHumidityMax.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_ConfHumidityMax ( HDC2080_data_t myHumidityMax )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_HUMIDITY_MAX;
    cmd[1]   =   myHumidityMax.rawHumidity_max;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );


    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetInterruptConfiguration ( HDC2080_data_t* )
 *
 * @details     It gets the interrupt configuration bits.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myIntConfiguration:    DRDY_ENABLE, TH_ENABLE, TL_ENABLE, HH_ENABLE and HL_ENABLE status.
 *
 *
 * @return       Status of HDC2080_GetInterruptConfiguration.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetInterruptConfiguration  ( HDC2080_data_t* myIntConfiguration )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_INTERRUPT_ENABLE;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myIntConfiguration->drdy_enable  =   (HDC2080_interrupt_enable_drdy_enable_t)( cmd & INTERRUPT_DRDY_DRDY_ENABLE_MASK );
    myIntConfiguration->th_enable    =   (HDC2080_interrupt_enable_th_enable_t)( cmd & INTERRUPT_DRDY_TH_ENABLE_MASK );
    myIntConfiguration->tl_enable    =   (HDC2080_interrupt_enable_tl_enable_t)( cmd & INTERRUPT_DRDY_TL_ENABLE_MASK );
    myIntConfiguration->hh_enable    =   (HDC2080_interrupt_enable_hh_enable_t)( cmd & INTERRUPT_DRDY_HH_ENABLE_MASK );
    myIntConfiguration->hl_enable    =   (HDC2080_interrupt_enable_hl_enable_t)( cmd & INTERRUPT_DRDY_HL_ENABLE_MASK );


    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetInterruptConfiguration ( HDC2080_data_t )
 *
 * @details     It configures the humidity MAX register.
 *
 * @param[in]    myIntConfiguration:    DRDY_ENABLE, TH_ENABLE, TL_ENABLE, HH_ENABLE and HL_ENABLE.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetInterruptConfiguration.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetInterruptConfiguration  ( HDC2080_data_t myIntConfiguration )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_INTERRUPT_ENABLE;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  &=  ~( INTERRUPT_DRDY_DRDY_ENABLE_MASK | INTERRUPT_DRDY_TH_ENABLE_MASK | INTERRUPT_DRDY_TL_ENABLE_MASK | INTERRUPT_DRDY_HH_ENABLE_MASK | INTERRUPT_DRDY_HL_ENABLE_MASK );
    cmd[1]  |=   ( myIntConfiguration.drdy_enable | myIntConfiguration.th_enable | myIntConfiguration.tl_enable | myIntConfiguration.hh_enable | myIntConfiguration.hl_enable );
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetTemperatureOffset ( HDC2080_data_t )
 *
 * @details     It sets the temperature offset.
 *
 * @param[in]    myTemperatureOffset:   Temperature offset value.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetTemperatureOffset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetTemperatureOffset ( HDC2080_data_t myTemperatureOffset )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Write the register */
    cmd[0]   =   HDC2080_TEMP_OFFSET_ADJUST;
    cmd[1]   =   myTemperatureOffset.temp_offset_adjust;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetTemperatureOffset ( HDC2080_data_t* )
 *
 * @details     It gets the temperature offset ( raw value ).
 *
 * @param[in]    N/A
 *
 * @param[out]   myTemperatureOffset:   Temperature offset value ( raw value ).
 *
 *
 * @return       Status of HDC2080_GetTemperatureOffset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetTemperatureOffset ( HDC2080_data_t* myTemperatureOffset )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_TEMP_OFFSET_ADJUST;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myTemperatureOffset->temp_offset_adjust  =   cmd;



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetHumidityOffset ( HDC2080_data_t )
 *
 * @details     It sets the humidity offset.
 *
 * @param[in]    myHumidityOffset:  Humidity offset value.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetHumidityOffset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetHumidityOffset ( HDC2080_data_t myHumidityOffset )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Write the register */
    cmd[0]   =   HDC2080_HUM_OFFSET_ADJUST;
    cmd[1]   =   myHumidityOffset.hum_offset_adjust;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetHumidityOffset ( HDC2080_data_t* )
 *
 * @details     It gets the humidity offset ( raw value ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myHumidityOffset:  Humidity offset value ( raw value ).
 *
 *
 * @return       Status of HDC2080_GetHumidityOffset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetHumidityOffset ( HDC2080_data_t* myHumidityOffset )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_HUM_OFFSET_ADJUST;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myHumidityOffset->hum_offset_adjust  =   cmd;



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetTemperatureThresLow ( HDC2080_data_t )
 *
 * @details     It sets the temperature threshold low value ( raw value ).
 *
 * @param[in]    myTempThresLow:  Temperature threshold low value ( raw value ).
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetTemperatureThresLow.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetTemperatureThresLow ( HDC2080_data_t myTempThresLow )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Write the register */
    cmd[0]   =   HDC2080_TEMP_THR_L;
    cmd[1]   =   myTempThresLow.temp_thres_low;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetTemperatureThresLow ( HDC2080_data_t* )
 *
 * @details     It gets the temperature threshold low value ( raw value ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTempThresLow:  Temperature threshold low value ( raw value ).
 *
 *
 * @return       Status of HDC2080_GetTemperatureThresLow.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetTemperatureThresLow ( HDC2080_data_t* myTempThresLow )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_TEMP_THR_L;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myTempThresLow->temp_thres_low   =   cmd;



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetTemperatureThresHigh ( HDC2080_data_t )
 *
 * @details     It sets the temperature threshold high value ( raw value ).
 *
 * @param[in]    myTempThresHigh: Temperature threshold high value ( raw value ).
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetTemperatureThresHigh.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetTemperatureThresHigh ( HDC2080_data_t myTempThresHigh )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Write the register */
    cmd[0]   =   HDC2080_TEMP_THR_H;
    cmd[1]   =   myTempThresHigh.temp_thres_high;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetTemperatureThresHigh ( HDC2080_data_t* )
 *
 * @details     It gets the temperature threshold high value ( raw value ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTempThresHigh: Temperature threshold high value ( raw value ).
 *
 *
 * @return       Status of HDC2080_GetTemperatureThresHigh.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetTemperatureThresHigh ( HDC2080_data_t* myTempThresHigh )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_TEMP_THR_H;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myTempThresHigh->temp_thres_high     =   cmd;



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetHumidityThresLow ( HDC2080_data_t )
 *
 * @details     It sets the humidity threshold low value ( raw value ).
 *
 * @param[in]    myTempThresLow:  Humidity threshold low value ( raw value ).
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetHumidityThresLow.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetHumidityThresLow  ( HDC2080_data_t myHumThresLow )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Write the register */
    cmd[0]   =   HDC2080_RH_THR_L;
    cmd[1]   =   myHumThresLow.hum_thres_low;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetHumidityThresLow ( HDC2080_data_t* )
 *
 * @details     It gets the humidity threshold low value ( raw value ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myHumThresLow:   Humidity threshold low value ( raw value ).
 *
 *
 * @return       Status of HDC2080_GetHumidityThresLow.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetHumidityThresLow ( HDC2080_data_t* myHumThresLow )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RH_THR_L;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myHumThresLow->hum_thres_low     =   cmd;



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetHumidityThresHigh ( HDC2080_data_t )
 *
 * @details     It sets the humidity threshold high value ( raw value ).
 *
 * @param[in]    myHumThresHigh:  Humidity threshold high value ( raw value ).
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetHumidityThresHigh.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetHumidityThresHigh ( HDC2080_data_t myHumThresHigh )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Write the register */
    cmd[0]   =   HDC2080_RH_THR_H;
    cmd[1]   =   myHumThresHigh.hum_thres_high;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetHumidityThresHigh ( HDC2080_data_t* )
 *
 * @details     It gets the humidity threshold high value ( raw value ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myHumThresHigh:  Humidity threshold high value ( raw value ).
 *
 *
 * @return       Status of HDC2080_GetHumidityThresHigh.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetHumidityThresHigh ( HDC2080_data_t* myHumThresHigh )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RH_THR_H;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myHumThresHigh->hum_thres_high   =   cmd;



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetSoftReset ( void )
 *
 * @details     It makes a soft reset.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetSoftReset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         This bit is self-clear.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetSoftReset ( void )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_RESET_DRDY_INT_CONF;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  |=   RESET_DRDY_INT_CONF_SOFT_RES_RESET;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetSoftReset ( HDC2080_data_t* )
 *
 * @details     It checks the soft reset flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   mySoftResetFlag: Soft reset flag.
 *
 *
 * @return       Status of HDC2080_SetSoftReset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetSoftReset ( HDC2080_data_t* mySoftResetFlag )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RESET_DRDY_INT_CONF;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    mySoftResetFlag->soft_res    =   (HDC2080_reset_drdy_int_conf_soft_res_t)( RESET_DRDY_INT_CONF_SOFT_RES_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetAutoMeasurementMode ( HDC2080_data_t )
 *
 * @details     It sets auto measurement mode ( AMM ).
 *
 * @param[in]    myAMM:           Auto measurement mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetAutoMeasurementMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetAutoMeasurementMode ( HDC2080_data_t myAMM )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_RESET_DRDY_INT_CONF;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  &=  ~( RESET_DRDY_INT_CONF_AMM_MASK );
    cmd[1]  |=   ( myAMM.amm );
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetAutoMeasurementMode ( HDC2080_data_t* )
 *
 * @details     It gets auto measurement mode ( AMM ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myAMM:           Auto measurement mode.
 *
 *
 * @return       Status of HDC2080_GetAutoMeasurementMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetAutoMeasurementMode ( HDC2080_data_t* myAMM )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RESET_DRDY_INT_CONF;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myAMM->amm   =  (HDC2080_reset_drdy_int_conf_amm_t)( RESET_DRDY_INT_CONF_AMM_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetHeaterMode ( HDC2080_data_t )
 *
 * @details     It sets the heater mode.
 *
 * @param[in]    myHeatEn:        Heater mode: On/Off.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetHeaterMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetHeaterMode ( HDC2080_data_t myHeatEn )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_RESET_DRDY_INT_CONF;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  &=  ~( RESET_DRDY_INT_CONF_HEAT_EN_MASK );
    cmd[1]  |=   ( myHeatEn.heater_en );
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetHeaterMode ( HDC2080_data_t* )
 *
 * @details     It gets the heater mode.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myHeatEn:        Heater mode: On/Off..
 *
 *
 * @return       Status of HDC2080_GetHeaterMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetHeaterMode ( HDC2080_data_t* myHeatEn )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RESET_DRDY_INT_CONF;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myHeatEn->heater_en  =  (HDC2080_reset_drdy_int_conf_heat_en_t)( RESET_DRDY_INT_CONF_HEAT_EN_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetPinConfifuration ( HDC2080_data_t )
 *
 * @details     It sets the DRDY/INT_EN pin configuration.
 *
 * @param[in]    myPinConfiguration: DRDY/INT_EN pin configuration.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetPinConfifuration.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetPinConfifuration ( HDC2080_data_t myPinConfiguration )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_RESET_DRDY_INT_CONF;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  &=  ~( RESET_DRDY_INT_CONF_DRDY_INT_EN_MASK );
    cmd[1]  |=   ( myPinConfiguration.drdy_intEn );
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetPinConfiguration ( HDC2080_data_t* )
 *
 * @details     It gets the DRDY/INT_EN pin configuration.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myPinConfiguration: DRDY/INT_EN pin configuration..
 *
 *
 * @return       Status of HDC2080_GetPinConfiguration.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetPinConfiguration ( HDC2080_data_t* myPinConfiguration )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RESET_DRDY_INT_CONF;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Update the register   */
    myPinConfiguration->drdy_intEn   =  (HDC2080_reset_drdy_int_conf_drdr_int_en_t)( RESET_DRDY_INT_CONF_DRDY_INT_EN_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetInterruptPolarity ( HDC2080_data_t )
 *
 * @details     It sets the interrupt polarity
 *
 * @param[in]    myIntPol:          Interrupt polarity: Active Low/High.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetInterruptPolarity.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetInterruptPolarity ( HDC2080_data_t myIntPol )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_RESET_DRDY_INT_CONF;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  &=  ~( RESET_DRDY_INT_CONF_INT_POL_MASK );
    cmd[1]  |=   ( myIntPol.int_pol );
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetInterruptPolarity ( HDC2080_data_t* )
 *
 * @details     It gets the interrupt polarity
 *
 * @param[in]    N/A.
 *
 * @param[out]   myIntPol:          Interrupt polarity: Active Low/High.
 *
 *
 * @return       Status of HDC2080_GetInterruptPolarity.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetInterruptPolarity ( HDC2080_data_t* myIntPol )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RESET_DRDY_INT_CONF;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myIntPol->int_pol    =  (HDC2080_reset_drdy_int_conf_int_pol_t)( RESET_DRDY_INT_CONF_INT_POL_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetInterruptMode ( HDC2080_data_t )
 *
 * @details     It sets the interrupt mode
 *
 * @param[in]    myIntMode:         Interrupt mode: Level sensitive/Comparator mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetInterruptMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetInterruptMode ( HDC2080_data_t myIntMode )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_RESET_DRDY_INT_CONF;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  &=  ~( RESET_DRDY_INT_CONF_INT_MODE_MASK );
    cmd[1]  |=   ( myIntMode.int_mode );
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetInterruptMode ( HDC2080_data_t* )
 *
 * @details     It gets the interrupt mode
 *
 * @param[in]    N/A.
 *
 * @param[out]   myIntMode:         Interrupt mode: Level sensitive/Comparator mode.
 *
 *
 * @return       Status of HDC2080_GetInterruptMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetInterruptMode ( HDC2080_data_t* myIntMode )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_RESET_DRDY_INT_CONF;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myIntMode->int_mode  =  (HDC2080_reset_drdy_int_conf_int_mode_t)( RESET_DRDY_INT_CONF_INT_MODE_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_SetMeasurementConf ( HDC2080_data_t )
 *
 * @details     It sets the measurement configuration.
 *
 * @param[in]    myMeasConf:        Temperature resolution, Humidity resolution and Measurement configuration.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_SetMeasurementConf.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_SetMeasurementConf ( HDC2080_data_t myMeasConf )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_MEASUREMENT_CONFIGURATION;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  &=  ~( MEASUREMENT_CONF_TRES_MASK | MEASUREMENT_CONF_HRES_MASK | MEASUREMENT_CONF_MEAS_CONF_MASK );
    cmd[1]  |=   ( myMeasConf.tres | myMeasConf.hres | myMeasConf.meas_conf );
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetMeasurementConf ( HDC2080_data_t* )
 *
 * @details     It gets the measurement configuration.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myMeasConf:        Temperature resolution, Humidity resolution and Measurement configuration.
 *
 *
 * @return       Status of HDC2080_GetMeasurementConf.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetMeasurementConf ( HDC2080_data_t* myMeasConf )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_MEASUREMENT_CONFIGURATION;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myMeasConf->tres         =  (HDC2080_measurement_configuration_tres_t)( MEASUREMENT_CONF_TRES_MASK & cmd );
    myMeasConf->hres         =  (HDC2080_measurement_configuration_hres_t)( MEASUREMENT_CONF_HRES_MASK & cmd );
    myMeasConf->meas_conf    =  (HDC2080_measurement_configuration_meas_conf_t)( MEASUREMENT_CONF_MEAS_CONF_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_StartMeasurementTrigger ( void )
 *
 * @details     It triggers a new measurement.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of HDC2080_StartMeasurementTrigger.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_StartMeasurementTrigger ( void )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_MEASUREMENT_CONFIGURATION;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[1], 1U );

    /* Update the register   */
    cmd[1]  |=   MEASUREMENT_CONF_MEAS_TRIG_START_MEASUREMENT;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetMeasurementTrigger ( HDC2080_data_t* )
 *
 * @details     It gets the measurement trigger flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myMeasTrig:        Measurement trigger flag.
 *
 *
 * @return       Status of HDC2080_GetMeasurementTrigger.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetMeasurementTrigger ( HDC2080_data_t* myMeasTrig )
{
    char     cmd  = 0U;
    uint32_t aux;

    /* Read the register */
    cmd  =   HDC2080_MEASUREMENT_CONFIGURATION;
    aux  =   _i2c.write ( _HDC2080_Addr, &cmd, 1U, true );
    aux  =   _i2c.read  ( _HDC2080_Addr, &cmd, 1U );

    /* Parse the data    */
    myMeasTrig->meas_trig    =  (HDC2080_measurement_configuration_meas_trig_t)( MEASUREMENT_CONF_MEAS_TRIG_MASK & cmd );



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetManufacturerID ( HDC2080_data_t* )
 *
 * @details     It gets the manufacturer ID.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myManufacturerID:  Manufacturer ID.
 *
 *
 * @return       Status of HDC2080_GetManufacturerID.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetManufacturerID ( HDC2080_data_t* myManufacturerID )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_MANUFACTURER_ID_LOW;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

    /* Parse the data    */
    myManufacturerID->manufacturer_id    =   cmd[1];
    myManufacturerID->manufacturer_id  <<=   8U;
    myManufacturerID->manufacturer_id   |=   cmd[0];



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}



/**
 * @brief       HDC2080_GetDeviceID ( HDC2080_data_t* )
 *
 * @details     It gets the device ID.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myDeviceID:        Device ID.
 *
 *
 * @return       Status of HDC2080_GetDeviceID.
 *
 *
 * @author      Manuel Caballero
 * @date        19/November/2019
 * @version     19/November/2019   The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
HDC2080::HDC2080_status_t HDC2080::HDC2080_GetDeviceID ( HDC2080_data_t* myDeviceID )
{
    char     cmd[2]  = { 0U };
    uint32_t aux;

    /* Read the register */
    cmd[0]   =   HDC2080_DEVICE_ID_LOW;
    aux      =   _i2c.write ( _HDC2080_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _HDC2080_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

    /* Parse the data    */
    myDeviceID->device_id    =   cmd[1];
    myDeviceID->device_id  <<=   8U;
    myDeviceID->device_id   |=   cmd[0];



    if ( aux == I2C_SUCCESS ) {
        return   HDC2080_SUCCESS;
    } else {
        return   HDC2080_FAILURE;
    }
}