/**
 * @brief       DS1624.h
 * @details     Digital Thermometer and Memory.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        31/January/2018
 * @version     31/January/2018    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef DS1624_H
#define DS1624_H

#include "mbed.h"


/**
    Example:

#include "mbed.h"
#include "DS1624.h"

DS1624 myDS1624   ( I2C_SDA, I2C_SCL, DS1624::DS1624_ADDRESS_0, 400000 );
Serial pc         ( USBTX, USBRX );

Ticker     newReading;
DigitalOut myled(LED1);

DS1624::DS1624_status_t             aux;
DS1624::DS1624_vector_data_t        myDS1624Data;
DS1624::DS1624_access_config_done_t myDS1624_TempConversionStatus;
uint32_t                            myState = 0;


void changeDATA ( void )
{
    myState = 1;
}


int main()
{
    char     myDataInEEPROM[] =   { 0x23, 0x23, 0x23 };
    uint32_t myTimeoOut       =   0;

    pc.baud ( 115200 );

    // Configure 1SHOT mode
    aux  =   myDS1624.DS1624_SetConversionMode ( DS1624::ACCESS_CONFIG_1SHOT_ONE_TEMPERATURE_CONVERSION );
    wait_ms ( 50 );

    // Write myDataInEEPROM into EEPROM memory ( address: 0x33 )
    aux  =   myDS1624.DS1624_WriteBytesEEPROM  ( 0x33, myDataInEEPROM, sizeof( myDataInEEPROM )/sizeof( myDataInEEPROM[0] ) );
    wait_ms ( 50 );

    // Read EEPROM memory ( address: 0x33 ), just to check if the data was stored correctly
    myDataInEEPROM[0]    =   0;
    myDataInEEPROM[1]    =   0;
    myDataInEEPROM[2]    =   0;

    aux  =   myDS1624.DS1624_ReadBytesEEPROM   ( 0x33, &myDataInEEPROM[0], sizeof( myDataInEEPROM )/sizeof( myDataInEEPROM[0] ) );

    pc.printf( "DataEEPROM[0]: %x\nDataEEPROM[1]: %x\nDataEEPROM[2]: %x\n\r\n", myDataInEEPROM[0], myDataInEEPROM[1], myDataInEEPROM[2] );



    newReading.attach( &changeDATA, 1 );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 1s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        myled = 1;

        if ( myState == 1 ) {
            // Trigger a new temperature conversion
            aux  =   myDS1624.DS1624_StartConvertTemperature ();

            // Wait until the temperature conversion is completed or timeout
            myTimeoOut   =   232323;
            do {
                aux  =   myDS1624.DS1624_IsTemperatureConversionDone ( &myDS1624_TempConversionStatus );
                myTimeoOut--;
            } while ( ( ( myDS1624_TempConversionStatus & DS1624::ACCESS_CONFIG_DONE_MASK ) != DS1624::ACCESS_CONFIG_DONE_CONVERSION_COMPLETE ) && ( myTimeoOut > 0 ) );


            // Check if TimeOut, if so, there was an error ( send: Error! ), send the data otherwise
            if ( myTimeoOut <= 0 ) {
                pc.printf( "Error!\r\n" );
            } else {
                aux  =   myDS1624.DS1624_ReadTemperature    ( &myDS1624Data );
                pc.printf( "Temperature: %0.4fC\r\n", myDS1624Data.Temperature );
            }

            myState  =   0;                                                     // Reset the variable
        }

        myled = 0;
    }
}
*/


/*!
 Library for the DS1624 Digital Thermometer and Memory.
*/
class DS1624
{
public:
    /**
    * @brief   DEFAULT ADDRESSES. [ 0 : GND | 1 : VDD ]
    */
    typedef enum {
        DS1624_ADDRESS_0     =   ( ( ( 0b1001 << 3 ) | 0b000 ) << 1 ),          /*!<   Address A2 A1 A0: 000                */
        DS1624_ADDRESS_1     =   ( ( ( 0b1001 << 3 ) | 0b001 ) << 1 ),          /*!<   Address A2 A1 A0: 001                */
        DS1624_ADDRESS_2     =   ( ( ( 0b1001 << 3 ) | 0b010 ) << 1 ),          /*!<   Address A2 A1 A0: 010                */
        DS1624_ADDRESS_3     =   ( ( ( 0b1001 << 3 ) | 0b011 ) << 1 ),          /*!<   Address A2 A1 A0: 011                */
        DS1624_ADDRESS_4     =   ( ( ( 0b1001 << 3 ) | 0b100 ) << 1 ),          /*!<   Address A2 A1 A0: 100                */
        DS1624_ADDRESS_5     =   ( ( ( 0b1001 << 3 ) | 0b101 ) << 1 ),          /*!<   Address A2 A1 A0: 101                */
        DS1624_ADDRESS_6     =   ( ( ( 0b1001 << 3 ) | 0b110 ) << 1 ),          /*!<   Address A2 A1 A0: 110                */
        DS1624_ADDRESS_7     =   ( ( ( 0b1001 << 3 ) | 0b111 ) << 1 )           /*!<   Address A2 A1 A0: 111                */
    } DS1624_address_t;


// DS1624 Command Set
    /**
      * @brief   TEMPERATURE CONVERSION COMMANDS
      */
    typedef enum {
        DS1624_READ_TEMPERATURE     =   0xAA,               /*!<  Reads last converted temperature value from temperature register  */
        DS1624_START_CONVERT_T      =   0xEE,               /*!<  Initiates temperature conversion                                  */
        DS1624_STOP_CONVERT_T       =   0x22                /*!<  Halts temperature conversion                                      */
    } DS1624_temperature_conversion_commands_t;


    /**
      * @brief   THERMOSTAT COMMANDS. NOTE: Writing to the EEPROM typically requires 10ms at room temperature. After issuing a write
      *                                     command, no further reads or writes should be requested for at least 10ms.
      */
    typedef enum {
        DS1624_ACCESS_MEMORY        =   0x17,               /*!<  Reads or writes to 256-byte EEPROM memory                         */
        DS1624_ACCESS_CONFIG        =   0xAC                /*!<  Reads or writes configuration data to configuration register      */
    } DS1624_thermostat_commands_t;



// ACCESS CONFIG ( CONFIGURATION/STATUS REGISTER )
    /**
      * @brief   BIT7: DONE
      */
    typedef enum {
        ACCESS_CONFIG_DONE_MASK                     =   ( 1 << 7 ),                 /*!<  DONE Mask                                 */
        ACCESS_CONFIG_DONE_CONVERSION_COMPLETE      =   ( 1 << 7 ),                 /*!<  Conversion complete                       */
        ACCESS_CONFIG_DONE_CONVERSION_IN_PROGRESS   =   ( 0 << 7 )                  /*!<  Conversion in progress                    */
    } DS1624_access_config_done_t;


    /**
      * @brief   BIT0: 1SHOT
      */
    typedef enum {
        ACCESS_CONFIG_1SHOT_MASK                                =   ( 1 << 0 ),     /*!<  1SHOT Mask                                                                        */
        ACCESS_CONFIG_1SHOT_ONE_TEMPERATURE_CONVERSION          =   ( 1 << 0 ),     /*!<  Perform one temperature conversion upon receipt of the Start Convert T protocol   */
        ACCESS_CONFIG_1SHOT_CONTINUOUSLY_TEMPERATURE_CONVERSION =   ( 0 << 0 )      /*!<  Perform continuously temperature conversions                                      */
    } DS1624_access_config_1shot_t;



// TEMPERATURE REGISTER
    /**
      * @brief   MSB BIT15: SIGN
      */
    typedef enum {
        MSB_TEMPERATURE_SIGN_BIT_MASK               =   ( 1 << 7 ),                 /*!<  Temperature sign Mask                     */
        MSB_TEMPERATURE_SIGN_BIT_POSITIVE           =   ( 0 << 7 ),                 /*!<  Temperature is POSITIVE                   */
        MSB_TEMPERATURE_SIGN_BIT_NEGATIVE           =   ( 1 << 7 )                  /*!<  Temperature is NEGATIVE                   */
    } DS1624_msb_temperature_sign_bit_t;




#ifndef DS1624_VECTOR_STRUCT_H
#define DS1624_VECTOR_STRUCT_H
    typedef struct {
        uint8_t MSBTemperature;
        uint8_t LSBTemperature;
        char    Control_Status_Register;

        float   Temperature;
    } DS1624_vector_data_t;
#endif



    /**
      * @brief   INTERNAL CONSTANTS
      */
#define DS1624_TEMPERATURE_RESOLUTION   0.0625

    typedef enum {
        DS1624_SUCCESS     =       0,
        DS1624_FAILURE     =       1,

        I2C_SUCCESS        =       0,                                           /*!<   I2C communication was fine        */
        I2C_FAILURE        =       1
    } DS1624_status_t;




    /** Create an DS1624 object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency in Hz.
      */
    DS1624 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete DS1624 object.
     */
    ~DS1624();

    /** It triggers a new temperature measurement.
      */
    DS1624_status_t DS1624_StartConvertTemperature     ( void );

    /** It stops the current temperature conversion.
          */
    DS1624_status_t DS1624_StopConvertTemperature      ( void );

    /** It reads the last raw temperature conversion result.
          */
    DS1624_status_t DS1624_ReadRawTemperature          ( DS1624_vector_data_t* myRawTemperature );

    /** It reads the last temperature conversion result.
          */
    DS1624_status_t DS1624_ReadTemperature             ( DS1624_vector_data_t* myTemperature );

    /** It reads the CONFIGURATION/STATUS register.
          */
    DS1624_status_t DS1624_GetStatusRegister           ( DS1624_vector_data_t* myStatusRegister );

    /** It sets 1SHOT/Continuous temperature conversion mode.
          */
    DS1624_status_t DS1624_SetConversionMode           ( DS1624_access_config_1shot_t myConversionMode );

    /** It checks if a temperature conversion is done.
          */
    DS1624_status_t DS1624_IsTemperatureConversionDone ( DS1624_access_config_done_t* myTemperatureConversionStatus );

    /** It reads a certain number of bytes from EEPROM memory.
          */
    DS1624_status_t DS1624_ReadBytesEEPROM             ( char myStartingAddress, char* myReadBytesEEPROM, uint8_t myLength );

    /** It writes a certain number of bytes to EEPROM memory.
          */
    DS1624_status_t DS1624_WriteBytesEEPROM            ( char myStartingAddress, char myWriteBytesEEPROM[], uint8_t myLength );




private:
    I2C         _i2c;
    uint32_t    _DS1624_Addr;
};

#endif
