/**
 * @brief       BMP180.h
 * @details     Digital Pressure Sensor.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        11/September/2018
 * @version     11/September/2018    The ORIGIN
 * @pre         N/A
 * @warning     N/A
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */
#ifndef BMP180_H_
#define BMP180_H_

#include "mbed.h"


/**
    Example:

@code
#include "mbed.h"
#include "BMP180.h"

BMP180 myBMP180 ( I2C_SDA, I2C_SCL, BMP180::BMP180_ADDRESS, 400000 );
Serial pc       ( USBTX, USBRX );                                               // tx, rx

DigitalOut  myled       ( LED1 );
Ticker      newAction;


// @brief Constants.
//


// @brief Variables.
//
volatile uint32_t myState;                                                      //   State that indicates when to perform an ADC sample


//
// @brief   FUNCTION PROTOTYPES
//
void    changeDATA     ( void );


// @brief FUNCTION FOR APPLICATION MAIN ENTRY.
//
int main()
{
    uint32_t                            aux;
    BMP180::BMP180_chip_id_data_t       myBMP180_ChipID;
    BMP180::BMP180_temperature_data_t   myBMP180_TemperatureData;
    BMP180::BMP180_pressure_data_t      myBMP180_PressureData;
    BMP180::BMP180_calibration_data_t   myBMP180_Calibration_Data;
    BMP180::BMP180_uncompensated_data_t myBMP180_Uncompensated_Data;

    pc.baud ( 115200 );


    myled   =   1;
    wait(3);
    myled   =   0;

    // Get the chip-id
    aux  =   myBMP180.BMP180_GetID ( &myBMP180_ChipID );
    pc.printf( "Chip-ID: %d\r\n", myBMP180_ChipID.id );

    // Get calibration data
    aux  =   myBMP180.BMP180_Get_Cal_Param ( &myBMP180_Calibration_Data );
    pc.printf( "AC1: %d, AC2: %d, AC3: %d, AC4: %d, AC5: %d, AC6: %d, B1: %d, B2: %d, MB: %d, MC: %d, MD: %d\r\n", myBMP180_Calibration_Data.ac1, myBMP180_Calibration_Data.ac2,
               myBMP180_Calibration_Data.ac3, myBMP180_Calibration_Data.ac4, myBMP180_Calibration_Data.ac5, myBMP180_Calibration_Data.ac6, myBMP180_Calibration_Data.b1, myBMP180_Calibration_Data.b2,
               myBMP180_Calibration_Data.mb, myBMP180_Calibration_Data.mc, myBMP180_Calibration_Data.md );



    myState  =   0UL;                                                           // Reset the variable
    newAction.attach( &changeDATA, 2 );                                         // the address of the function to be attached ( changeDATA ) and the interval ( 2s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        if ( myState == 1UL ) {
            myled = 1U;

            // Get uncompensated temperature
            aux  =   myBMP180.BMP180_Get_UT ( &myBMP180_Uncompensated_Data );

            // Get uncompensated pressure, Resolution: Standard
            aux  =   myBMP180.BMP180_Get_UP ( BMP180::PRESSURE_STANDARD_MODE, &myBMP180_Uncompensated_Data );

            // Calculate the true temperature
            myBMP180_TemperatureData  =   myBMP180.BMP180_Get_Temperature ( myBMP180_Calibration_Data, myBMP180_Uncompensated_Data );

            // Calculate the true pressure
            myBMP180_PressureData  =   myBMP180.BMP180_Get_CalPressure ( myBMP180_Calibration_Data, myBMP180_TemperatureData, BMP180::PRESSURE_STANDARD_MODE, myBMP180_Uncompensated_Data );

            // Transmit result through the UART
            pc.printf ( "Temperature: %0.1f C | Pressure: %ld Pa\r\n", ( 0.1f * myBMP180_TemperatureData.temp ), myBMP180_PressureData.press );


            // Reset the variables
            myState  =   0UL;
            myled    =   0U;
        }
    }
}


//
 // @brief       changeDATA ( void  )
 //
 // @details     It changes myState variable
 //
 // @param[in]    N/A
 //
 // @param[out]   N/A.
 //
 //
 // @return       N/A..
 //
 //
 // @author      Manuel Caballero
 // @date        11/September/2018
 // @version     11/September/2018   The ORIGIN
 // @pre         N/A
 // @warning     N/A.
 //
void changeDATA ( void )
{
    myState  =   1UL;
}
@endcode
*/


/*!
 Library for the BMP180 Digital Pressure Sensor.
*/
class BMP180
{
public:
    /**
    * @brief   DEFAULT ADDRESS
    */
    typedef enum {
        BMP180_ADDRESS     =   ( 0b1110111 << 1U )  /*!<   BMP180 I2C Address           */
    } BMP180_address_t;



    /**
      * @brief   CALIBRATION COEFFICIENTS
      */
    typedef enum {
        BMP180_AC1_MSB      =   0xAA,               /*!<  MSB AC1 coefficient           */
        BMP180_AC1_LSB      =   0xAB,               /*!<  LSB AC1 coefficient           */
        BMP180_AC2_MSB      =   0xAC,               /*!<  MSB AC2 coefficient           */
        BMP180_AC2_LSB      =   0xAD,               /*!<  LSB AC2 coefficient           */
        BMP180_AC3_MSB      =   0xAE,               /*!<  MSB AC3 coefficient           */
        BMP180_AC3_LSB      =   0xAF,               /*!<  LSB AC3 coefficient           */
        BMP180_AC4_MSB      =   0xB0,               /*!<  MSB AC4 coefficient           */
        BMP180_AC4_LSB      =   0xB1,               /*!<  LSB AC4 coefficient           */
        BMP180_AC5_MSB      =   0xB2,               /*!<  MSB AC5 coefficient           */
        BMP180_AC5_LSB      =   0xB3,               /*!<  LSB AC5 coefficient           */
        BMP180_AC6_MSB      =   0xB4,               /*!<  MSB AC6 coefficient           */
        BMP180_AC6_LSB      =   0xB5,               /*!<  LSB AC6 coefficient           */
        BMP180_B1_MSB       =   0xB6,               /*!<  MSB B1 coefficient            */
        BMP180_B1_LSB       =   0xB7,               /*!<  LSB B1 coefficient            */
        BMP180_B2_MSB       =   0xB8,               /*!<  MSB B2 coefficient            */
        BMP180_B2_LSB       =   0xB9,               /*!<  LSB B2 coefficient            */
        BMP180_MB_MSB       =   0xBA,               /*!<  MSB MB coefficient            */
        BMP180_MB_LSB       =   0xBB,               /*!<  LSB MB coefficient            */
        BMP180_MC_MSB       =   0xBC,               /*!<  MSB MC coefficient            */
        BMP180_MC_LSB       =   0xBD,               /*!<  LSB MC coefficient            */
        BMP180_MD_MSB       =   0xBE,               /*!<  MSB MD coefficient            */
        BMP180_MD_LSB       =   0xBF                /*!<  LSB MD coefficient            */
    } BMP180_calibration_coefficients_t;



    /**
      * @brief   REGISTERS MAP
      */
    typedef enum {
        BMP180_OUT_XLSB     =   0xF8,               /*!<  ADC OUT XLSB                              | Reset: 0x00   */
        BMP180_OUT_LSB      =   0xF7,               /*!<  ADC OUT MSB                               | Reset: 0x00   */
        BMP180_OUT_MSB      =   0xF6,               /*!<  ADC OUT LSB                               | Reset: 0x80   */
        BMP180_CTRL_MEAS    =   0xF4,               /*!<  Measurement control ( Control register )  | Reset: 0x00   */
        BMP180_SOFT         =   0xE0,               /*!<  Software reset                            | Reset: 0x00   */
        BMP180_ID           =   0xD0                /*!<  Chip-ID                                   | Reset: 0x55   */
    } BMP180_registers_map_t;



    /* MEASUREMENT CONTROL  */
    /**
      * @brief   START OF CONVERSION ( SCO, BIT<5> )
      */
    /*
        NOTE: The value of this bit stays '1' during conversion and is reset to '0' after conversion is complete ( data registers
              are filled ).
    */
    typedef enum {
        CTRL_MEAS_SCO_MASK                      =   ( 1U << 5U ),       /*!<  SCO Mask                         */
        CTRL_MEAS_SCO_CONVERSION_IN_PROGRESS    =   ( 1U << 5U ),       /*!<  SCO conversion in progress       */
        CTRL_MEAS_SCO_CONVERSION_COMPLETED      =   ( 0U << 5U )        /*!<  SCO conversion is complete       */
    } BMP180_sco_t;



    /**
      * @brief   CONTROLS THE OVERSAMPLING RATIO OF THE PRESSURE MEASUREMENT ( OSS, BIT<7:6> )
      */
    typedef enum {
        CTRL_MEAS_OSS_MASK                      =   ( 0b11 << 6U ),       /*!<  OSS Mask                         */
        CTRL_MEAS_OSS_OVERSAMPLING_SINGLE       =   ( 0b00 << 6U ),       /*!<  OSS oversampling ratio: single   */
        CTRL_MEAS_OSS_OVERSAMPLING_2_TIMES      =   ( 0b01 << 6U ),       /*!<  OSS oversampling ratio: 2 times  */
        CTRL_MEAS_OSS_OVERSAMPLING_4_TIMES      =   ( 0b10 << 6U ),       /*!<  OSS oversampling ratio: 4 times  */
        CTRL_MEAS_OSS_OVERSAMPLING_8_TIMES      =   ( 0b11 << 6U )        /*!<  OSS oversampling ratio: 8 times  */
    } BMP180_oss_t;



    /* SOFT RESET   */
    /**
      * @brief   SOFT RESET
      */
    typedef enum {
        SOFT_SOFT_RESET         =   0xB6                  /*!<  It will perform the same sequence as power on reset   */
    } BMP180_soft_reset_t;



    /* ID   */
    /**
      * @brief   CHIP-ID
      */
    typedef enum {
        ID_CHIP_ID              =   0x55                  /*!<  Fixed value                         */
    } BMP180_chip_id_t;



    /*
        AUXILIAR REGISTERS

        NOTE: These definitions are for better understanding in order to use the driver
    */
    /**
      * @brief   COMMANDS
      */
    typedef enum {
        BMP180_TRIGGER_TEMPERATURE      =   0x2E,           /*!<  It triggers a new temperature measurement  */
        BMP180_TRIGGER_PRESSURE         =   0x34            /*!<  It triggers a new pressure measurement     */
    } BMP180_commands_t;



    /**
      * @brief   PRESSURE
      */
    typedef enum {
        PRESSURE_ULTRA_LOW_POWER_MODE   =   CTRL_MEAS_OSS_OVERSAMPLING_SINGLE,      /*!<  OSS oversampling ratio: single   */
        PRESSURE_STANDARD_MODE          =   CTRL_MEAS_OSS_OVERSAMPLING_2_TIMES,     /*!<  OSS oversampling ratio: 2 times  */
        PRESSURE_HIGH_RESOLUTION_MODE   =   CTRL_MEAS_OSS_OVERSAMPLING_4_TIMES,     /*!<  OSS oversampling ratio: 4 times  */
        PRESSURE_ULTRA_HIGH_RES_MODE    =   CTRL_MEAS_OSS_OVERSAMPLING_8_TIMES      /*!<  OSS oversampling ratio: 8 times  */
    } BMP180_pressure_resolution_t;





#ifndef BMP180_VECTOR_STRUCT_H
#define BMP180_VECTOR_STRUCT_H
    /* Calibration data  */
    typedef struct {
        int16_t  ac1;
        int16_t  ac2;
        int16_t  ac3;
        uint16_t ac4;
        uint16_t ac5;
        uint16_t ac6;

        int16_t  b1;
        int16_t  b2;

        int16_t  mb;
        int16_t  mc;
        int16_t  md;
    } BMP180_calibration_data_t;


    /* Uncompensated data  */
    typedef struct {
        int32_t  ut;
        int32_t  up;
    } BMP180_uncompensated_data_t;


    /* Data  */
    typedef struct {
        int32_t  b5;

        int32_t  temp;
    } BMP180_temperature_data_t;

    typedef struct {
        int32_t  press;
    } BMP180_pressure_data_t;

    typedef struct {
        uint8_t  id;
    } BMP180_chip_id_data_t;
#endif


    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        BMP180_SUCCESS    =       0,
        BMP180_FAILURE    =       1,
        I2C_SUCCESS       =       0                                           /*!<   I2C communication was fine     */
    } BMP180_status_t;




    /** Create an BMP180 object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency in Hz.
      */
    BMP180 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete BMP180 object.
     */
    ~BMP180();

    /** It performs a soft reset.
    */
    BMP180_status_t             BMP180_SoftReset           ( void                                                                                                                                                                       );

    /** It gets the chip-ID.
      */
    BMP180_status_t             BMP180_GetID               ( BMP180_chip_id_data_t* myID                                                                                                                                         );

    /** It reads the calibration data.
      */
    BMP180_status_t             BMP180_Get_Cal_Param       ( BMP180_calibration_data_t* myCalibrationData                                                                                                                         );

    /** It reads uncompensated temperature value.
      */
    BMP180_status_t             BMP180_Get_UT              ( BMP180_uncompensated_data_t* myUT                                                                                                                                    );

    /** It reads uncompensated pressure value.
      */
    BMP180_status_t             BMP180_Get_UP              ( BMP180_pressure_resolution_t myPressureResolutionMode, BMP180_uncompensated_data_t* myUP                                                                             );

    /** It calculates true temperature.
      */
    BMP180_temperature_data_t   BMP180_Get_Temperature     ( BMP180_calibration_data_t myCalibrationData, BMP180_uncompensated_data_t myUT                                                                                        );

    /** It calculates true pressure.
      */
    BMP180_pressure_data_t      BMP180_Get_CalPressure     ( BMP180_calibration_data_t myCalibrationData, BMP180_temperature_data_t myB5, BMP180_pressure_resolution_t myPressureResolutionMode, BMP180_uncompensated_data_t myUP );


private:
    I2C      _i2c;
    uint32_t _BMP180_Addr;
};

#endif /* BMP180_H */
