/**
 * @brief       BME280.h
 * @details     Combined humidity and pressure sensor.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        03/September/2019
 * @version     03/September/2019    The ORIGIN
 * @pre         This is just a port from Bosh driver to mBed ( c++ )
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
/**
 * Copyright (C) 2018 - 2019 Bosch Sensortec GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * Neither the name of the copyright holder nor the names of the
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDER
 * OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES(INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 * The information provided is believed to be accurate and reliable.
 * The copyright holder assumes no responsibility
 * for the consequences of use
 * of such information nor for any infringement of patents or
 * other rights of third parties which may result from its use.
 * No license is granted by implication or otherwise under any patent or
 * patent rights of the copyright holder.
 *
 * @file    bme280.h
 * @date    08 Mar 2019
 * @version 3.3.6
 * @brief
 *
 */

/*! @file bme280.h
 * @brief Sensor driver for BME280 sensor
 */

/*!
 * @defgroup BME280 SENSOR API
 */
#ifndef BME280_H_
#define BME280_H_

#include "mbed.h"
#include "bme280_defs.h"


/**
    Example:
@code
#include "mbed.h"
#include "BME280.h"

BME280 myBME280     ( I2C_SDA, I2C_SCL, 400000 );
Serial pc           ( USBTX, USBRX );

DigitalOut  myled   ( LED1 );
Ticker      newReading;

uint32_t    myState = 0;


//@brief   FUNCTION PROTOTYPES
void    changeDATA     ( void );
void    user_delay_ms  ( uint32_t  period );
int8_t  user_i2c_read  ( uint8_t dev_id, uint8_t reg_addr, uint8_t *reg_data, uint16_t len );
int8_t  user_i2c_write ( uint8_t dev_id, uint8_t reg_addr, uint8_t *reg_data, uint16_t len );


//@brief FUNCTION FOR APPLICATION MAIN ENTRY.
int main()
{
    uint8_t             settings_sel;
    struct bme280_dev   dev;
    struct bme280_data  comp_data;
    int8_t              rslt = BME280_OK;

    pc.baud ( 115200 );

    myled   =   1;
    wait(3);
    myled   =   0;

    // Configure the device I2C interface
    dev.dev_id    =   BME280_I2C_ADDR_PRIM;
    dev.intf      =   BME280_I2C_INTF;
    dev.read      =   user_i2c_read;
    dev.write     =   user_i2c_write;
    dev.delay_ms  =   user_delay_ms;

    rslt  =  myBME280.bme280_init(&dev);

    // Configure the device in Force mode
    dev.settings.osr_h  = BME280_OVERSAMPLING_1X;
    dev.settings.osr_p  = BME280_OVERSAMPLING_16X;
    dev.settings.osr_t  = BME280_OVERSAMPLING_2X;
    dev.settings.filter = BME280_FILTER_COEFF_16;

    settings_sel = BME280_OSR_PRESS_SEL | BME280_OSR_TEMP_SEL | BME280_OSR_HUM_SEL | BME280_FILTER_SEL;

    rslt  =  myBME280.bme280_set_sensor_settings(settings_sel, &dev);



    newReading.attach( &changeDATA, 1 );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 1s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        myled = 1;

        if ( myState == 1 ) {
            // Trigger a new sample
            rslt = myBME280.bme280_set_sensor_mode(BME280_FORCED_MODE, &dev);

            // Wait for the measurement to complete and print data @25Hz
            dev.delay_ms(40);

            // Get the data
            rslt = myBME280.bme280_get_sensor_data(BME280_ALL, &comp_data, &dev);

            // Transmit result over the UART
            pc.printf( "T: %ld C | P: %ld Pa | RH: %ld %%\r\n", ( comp_data.temperature / 100 ), comp_data.pressure, ( comp_data.humidity / 1024 ) );

            myState  =   0;                                                     // Reset the variable
        }

        myled = 0;
    }
}


// @brief       changeDATA ( void  )
//
// @details     It changes myState variable
//
// @param[in]    N/A
//
// @param[out]   N/A.
//
//
// @return       N/A..
//
//
// @author      Manuel Caballero
// @date        03/September/2019
// @version     03/September/2019   The ORIGIN
// @pre         N/A
// @warning     N/A.
void changeDATA ( void )
{
    myState = 1;
}


// @brief       user_delay_ms ( uint32_t  )
//
// @details     Return control or wait, for a period amount of milliseconds
//
// @param[in]    period:       Delay in milliseconds.
//
// @param[out]   N/A.
//
//
// @return       N/A..
//
//
// @author      Manuel Caballero
// @date        03/September/2019
// @version     03/September/2019   The ORIGIN
// @pre         This is a Bosh pointer function adapted to our system.
// @warning     N/A.
void user_delay_ms ( uint32_t period )
{
    // Return control or wait,
    // for a period amount of milliseconds

    wait_ms ( period );
}


// @brief       user_i2c_read ( uint8_t , uint8_t reg_addr, uint8_t *reg_data, uint16_t len )
//
// @details     It adapts I2C reading functionality.
//
// @param[in]    dev_id:    I2C address.
// @param[in]    reg_addr:  Register to be read.
// @param[in]    len:       How many bytes to read.
//
// @param[out]   reg_data:  Result.
//
//
// @return       Status of user_i2c_read.
//
//
// @author      Manuel Caballero
// @date        03/September/2019
// @version     03/September/2019   The ORIGIN
// @pre         This is a Bosh pointer function adapted to our system.
// @warning     N/A.
int8_t user_i2c_read ( uint8_t dev_id, uint8_t reg_addr, uint8_t *reg_data, uint16_t len )
{
    int8_t rslt = 0; // Return 0 for Success, non-zero for failure

    // The parameter dev_id can be used as a variable to store the I2C address of the device


    // Data on the bus should be like
    // |------------+---------------------|
    // | I2C action | Data                |
    // |------------+---------------------|
    // | Start      | -                   |
    // | Write      | (reg_addr)          |
    // | Stop       | -                   |
    // | Start      | -                   |
    // | Read       | (reg_data[0])       |
    // | Read       | (....)              |
    // | Read       | (reg_data[len - 1]) |
    // | Stop       | -                   |
    // |------------+---------------------|

    // Read data
    uint32_t aux     =   0;
    aux      =   myBME280._i2c.write ( dev_id, (char*)&reg_addr, 1, true );
    aux      =   myBME280._i2c.read  ( dev_id, (char*)&reg_data[0], len );


    if ( aux == 0 ) {
        rslt     =   0;
    } else {
        rslt     =   0xFF;
    }


    return rslt;
}


// @brief       user_i2c_write ( uint8_t , uint8_t reg_addr, uint8_t *reg_data, uint16_t len )
//
// @details     It adapts I2C writing functionality.
//
// @param[in]    dev_id:    I2C address.
// @param[in]    reg_addr:  Register to be read.
// @param[out]   reg_data:  Data to be written.
// @param[in]    len:       How many bytes to read.
//
// @param[out]   N/A.
//
//
// @return       Status of user_i2c_write.
//
//
// @author      Manuel Caballero
// @date        03/September/2019
// @version     03/September/2019   The ORIGIN
// @pre         This is a Bosh pointer function adapted to our system.
// @warning     N/A.
int8_t user_i2c_write ( uint8_t dev_id, uint8_t reg_addr, uint8_t *reg_data, uint16_t len )
{
    int8_t rslt = 0; // Return 0 for Success, non-zero for failure

    // The parameter dev_id can be used as a variable to store the I2C address of the device


    // Data on the bus should be like
    // |------------+---------------------|
    // | I2C action | Data                |
    // |------------+---------------------|
    // | Start      | -                   |
    // | Write      | (reg_addr)          |
    // | Write      | (reg_data[0])       |
    // | Write      | (....)              |
    // | Write      | (reg_data[len - 1]) |
    // | Stop       | -                   |
    // |------------+---------------------|

    uint32_t     aux     =   0;
    char         cmd[64] =  { 0 };
    uint32_t     i       =   0;

    // Prepare the data to be sent
    cmd[0]   =   reg_addr;
    for ( i = 1; i <= len; i++ ) {
        cmd[i]   =   reg_data[i - 1];
    }

    // Write data
    aux      =   myBME280._i2c.write ( dev_id, &cmd[0], len + 1, false );


    if ( aux == 0 ) {
        rslt     =   0;
    } else {
        rslt     =   0xFF;
    }

    return rslt;
}
@endcode
*/



/*!
 Library for BME280 Combined humidity and pressure sensor.
*/
class BME280
{
public:
    /** Create an BME680 object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param freq    I2C frequency
      */
    BME280 ( PinName sda, PinName scl, uint32_t freq );

    /** Delete BME680 object.
     */
    ~BME280();

    /*!
     *  @brief This API is the entry point.
     *  It reads the chip-id and calibration data from the sensor.
     *
     *  @param[in,out] dev : Structure instance of bme280_dev
     *
     *  @return Result of API execution status
     *  @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t bme280_init(struct bme280_dev *dev);

    /*!
     * @brief This API writes the given data to the register address
     * of the sensor.
     *
     * @param[in] reg_addr : Register address from where the data to be written.
     * @param[in] reg_data : Pointer to data buffer which is to be written
     * in the sensor.
     * @param[in] len : No of bytes of data to write..
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t bme280_set_regs(uint8_t *reg_addr, const uint8_t *reg_data, uint8_t len, const struct bme280_dev *dev);

    /*!
     * @brief This API reads the data from the given register address of the sensor.
     *
     * @param[in] reg_addr : Register address from where the data to be read
     * @param[out] reg_data : Pointer to data buffer to store the read data.
     * @param[in] len : No of bytes of data to be read.
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t bme280_get_regs(uint8_t reg_addr, uint8_t *reg_data, uint16_t len, const struct bme280_dev *dev);

    /*!
     * @brief This API sets the oversampling, filter and standby duration
     * (normal mode) settings in the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[in] desired_settings : Variable used to select the settings which
     * are to be set in the sensor.
     *
     * @note : Below are the macros to be used by the user for selecting the
     * desired settings. User can do OR operation of these macros for configuring
     * multiple settings.
     *
     * Macros         |   Functionality
     * -----------------------|----------------------------------------------
     * BME280_OSR_PRESS_SEL    |   To set pressure oversampling.
     * BME280_OSR_TEMP_SEL     |   To set temperature oversampling.
     * BME280_OSR_HUM_SEL    |   To set humidity oversampling.
     * BME280_FILTER_SEL     |   To set filter setting.
     * BME280_STANDBY_SEL  |   To set standby duration setting.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error.
     */
    int8_t bme280_set_sensor_settings(uint8_t desired_settings, const struct bme280_dev *dev);

    /*!
     * @brief This API gets the oversampling, filter and standby duration
     * (normal mode) settings from the sensor.
     *
     * @param[in,out] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error.
     */
    int8_t bme280_get_sensor_settings(struct bme280_dev *dev);

    /*!
     * @brief This API sets the power mode of the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[in] sensor_mode : Variable which contains the power mode to be set.
     *
     *    sensor_mode           |   Macros
     * ---------------------|-------------------
     *     0                | BME280_SLEEP_MODE
     *     1                | BME280_FORCED_MODE
     *     3                | BME280_NORMAL_MODE
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t bme280_set_sensor_mode(uint8_t sensor_mode, const struct bme280_dev *dev);

    /*!
     * @brief This API gets the power mode of the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[out] sensor_mode : Pointer variable to store the power mode.
     *
     *   sensor_mode            |   Macros
     * ---------------------|-------------------
     *     0                | BME280_SLEEP_MODE
     *     1                | BME280_FORCED_MODE
     *     3                | BME280_NORMAL_MODE
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t bme280_get_sensor_mode(uint8_t *sensor_mode, const struct bme280_dev *dev);

    /*!
     * @brief This API performs the soft reset of the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error.
     */
    int8_t bme280_soft_reset(const struct bme280_dev *dev);

    /*!
     * @brief This API reads the pressure, temperature and humidity data from the
     * sensor, compensates the data and store it in the bme280_data structure
     * instance passed by the user.
     *
     * @param[in] sensor_comp : Variable which selects which data to be read from
     * the sensor.
     *
     * sensor_comp |   Macros
     * ------------|-------------------
     *     1       | BME280_PRESS
     *     2       | BME280_TEMP
     *     4       | BME280_HUM
     *     7       | BME280_ALL
     *
     * @param[out] comp_data : Structure instance of bme280_data.
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t bme280_get_sensor_data(uint8_t sensor_comp, struct bme280_data *comp_data, struct bme280_dev *dev);

    I2C      _i2c;

private:
    /*!
    * @brief This internal API puts the device to sleep mode.
    *
    * @param[in] dev : Structure instance of bme280_dev.
    *
    * @return Result of API execution status.
    * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
    */
    int8_t put_device_to_sleep(const struct bme280_dev *dev);

    /*!
     * @brief This internal API writes the power mode in the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[in] sensor_mode : Variable which contains the power mode to be set.
     *
     * @return Result of API execution status.
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t write_power_mode(uint8_t sensor_mode, const struct bme280_dev *dev);

    /*!
     * @brief This internal API is used to validate the device pointer for
     * null conditions.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t null_ptr_check(const struct bme280_dev *dev);

    /*!
     * @brief This internal API interleaves the register address between the
     * register data buffer for burst write operation.
     *
     * @param[in] reg_addr : Contains the register address array.
     * @param[out] temp_buff : Contains the temporary buffer to store the
     * register data and register address.
     * @param[in] reg_data : Contains the register data to be written in the
     * temporary buffer.
     * @param[in] len : No of bytes of data to be written for burst write.
     */
    void interleave_reg_addr(const uint8_t *reg_addr, uint8_t *temp_buff, const uint8_t *reg_data, uint8_t len);

    /*!
     * @brief This internal API reads the calibration data from the sensor, parse
     * it and store in the device structure.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t get_calib_data(struct bme280_dev *dev);

    /*!
     *  @brief This internal API is used to parse the temperature and
     *  pressure calibration data and store it in the device structure.
     *
     *  @param[out] dev : Structure instance of bme280_dev to store the calib data.
     *  @param[in] reg_data : Contains the calibration data to be parsed.
     */
    void parse_temp_press_calib_data(const uint8_t *reg_data, struct bme280_dev *dev);

    /*!
     *  @brief This internal API is used to parse the humidity calibration data
     *  and store it in device structure.
     *
     *  @param[out] dev : Structure instance of bme280_dev to store the calib data.
     *  @param[in] reg_data : Contains calibration data to be parsed.
     */
    void parse_humidity_calib_data(const uint8_t *reg_data, struct bme280_dev *dev);

#ifdef BME280_FLOAT_ENABLE

    /*!
     * @brief This internal API is used to compensate the raw pressure data and
     * return the compensated pressure data in double data type.
     *
     * @param[in] uncomp_data : Contains the uncompensated pressure data.
     * @param[in] calib_data : Pointer to the calibration data structure.
     *
     * @return Compensated pressure data.
     * @retval Compensated pressure data in double.
     */
    double compensate_pressure(const struct bme280_uncomp_data *uncomp_data,
                               const struct bme280_calib_data *calib_data);

    /*!
     * @brief This internal API is used to compensate the raw humidity data and
     * return the compensated humidity data in double data type.
     *
     * @param[in] uncomp_data : Contains the uncompensated humidity data.
     * @param[in] calib_data : Pointer to the calibration data structure.
     *
     * @return Compensated humidity data.
     * @retval Compensated humidity data in double.
     */
    double compensate_humidity(const struct bme280_uncomp_data *uncomp_data,
                               const struct bme280_calib_data *calib_data);

    /*!
     * @brief This internal API is used to compensate the raw temperature data and
     * return the compensated temperature data in double data type.
     *
     * @param[in] uncomp_data : Contains the uncompensated temperature data.
     * @param[in] calib_data : Pointer to calibration data structure.
     *
     * @return Compensated temperature data.
     * @retval Compensated temperature data in double.
     */
    double compensate_temperature(const struct bme280_uncomp_data *uncomp_data,
                                  struct bme280_calib_data *calib_data);

#else

    /*!
     * @brief This internal API is used to compensate the raw temperature data and
     * return the compensated temperature data in integer data type.
     *
     * @param[in] uncomp_data : Contains the uncompensated temperature data.
     * @param[in] calib_data : Pointer to calibration data structure.
     *
     * @return Compensated temperature data.
     * @retval Compensated temperature data in integer.
     */
    int32_t compensate_temperature(const struct bme280_uncomp_data *uncomp_data,
                                   struct bme280_calib_data *calib_data);

    /*!
     * @brief This internal API is used to compensate the raw pressure data and
     * return the compensated pressure data in integer data type.
     *
     * @param[in] uncomp_data : Contains the uncompensated pressure data.
     * @param[in] calib_data : Pointer to the calibration data structure.
     *
     * @return Compensated pressure data.
     * @retval Compensated pressure data in integer.
     */
    uint32_t compensate_pressure(const struct bme280_uncomp_data *uncomp_data,
                                 const struct bme280_calib_data *calib_data);

    /*!
     * @brief This internal API is used to compensate the raw humidity data and
     * return the compensated humidity data in integer data type.
     *
     * @param[in] uncomp_data : Contains the uncompensated humidity data.
     * @param[in] calib_data : Pointer to the calibration data structure.
     *
     * @return Compensated humidity data.
     * @retval Compensated humidity data in integer.
     */
    uint32_t compensate_humidity(const struct bme280_uncomp_data *uncomp_data,
                                 const struct bme280_calib_data *calib_data);

#endif

    /*!
     * @brief This internal API is used to identify the settings which the user
     * wants to modify in the sensor.
     *
     * @param[in] sub_settings : Contains the settings subset to identify particular
     * group of settings which the user is interested to change.
     * @param[in] desired_settings : Contains the user specified settings.
     *
     * @return Indicates whether user is interested to modify the settings which
     * are related to sub_settings.
     * @retval True -> User wants to modify this group of settings
     * @retval False -> User does not want to modify this group of settings
     */
    uint8_t are_settings_changed(uint8_t sub_settings, uint8_t desired_settings);

    /*!
     * @brief This API sets the humidity oversampling settings of the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t set_osr_humidity_settings(const struct bme280_settings *settings, const struct bme280_dev *dev);

    /*!
     * @brief This internal API sets the oversampling settings for pressure,
     * temperature and humidity in the sensor.
     *
     * @param[in] desired_settings : Variable used to select the settings which
     * are to be set.
     * @param[in] dev : Structure instance of bme280_dev.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t set_osr_settings(uint8_t desired_settings,
                            const struct bme280_settings *settings,
                            const struct bme280_dev *dev);

    /*!
     * @brief This API sets the pressure and/or temperature oversampling settings
     * in the sensor according to the settings selected by the user.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[in] desired_settings: variable to select the pressure and/or
     * temperature oversampling settings.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t set_osr_press_temp_settings(uint8_t desired_settings,
                                       const struct bme280_settings *settings,
                                       const struct bme280_dev *dev);

    /*!
     * @brief This internal API fills the pressure oversampling settings provided by
     * the user in the data buffer so as to write in the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[out] reg_data : Variable which is filled according to the pressure
     * oversampling data provided by the user.
     */
    void fill_osr_press_settings(uint8_t *reg_data, const struct bme280_settings *settings);

    /*!
     * @brief This internal API fills the temperature oversampling settings provided
     * by the user in the data buffer so as to write in the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[out] reg_data : Variable which is filled according to the temperature
     * oversampling data provided by the user.
     */
    void fill_osr_temp_settings(uint8_t *reg_data, const struct bme280_settings *settings);

    /*!
     * @brief This internal API sets the filter and/or standby duration settings
     * in the sensor according to the settings selected by the user.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[in] desired_settings : variable to select the filter and/or
     * standby duration settings.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t set_filter_standby_settings(uint8_t desired_settings,
                                       const struct bme280_settings *settings,
                                       const struct bme280_dev *dev);

    /*!
     * @brief This internal API fills the filter settings provided by the user
     * in the data buffer so as to write in the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[out] reg_data : Variable which is filled according to the filter
     * settings data provided by the user.
     */
    void fill_filter_settings(uint8_t *reg_data, const struct bme280_settings *settings);

    /*!
     * @brief This internal API fills the standby duration settings provided by the
     * user in the data buffer so as to write in the sensor.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[out] reg_data : Variable which is filled according to the standby
     * settings data provided by the user.
     */
    void fill_standby_settings(uint8_t *reg_data, const struct bme280_settings *settings);

    /*!
     * @brief This internal API parse the oversampling(pressure, temperature
     * and humidity), filter and standby duration settings and store in the
     * device structure.
     *
     * @param[out] dev : Structure instance of bme280_dev.
     * @param[in] reg_data : Register data to be parsed.
     */
    void parse_device_settings(const uint8_t *reg_data, struct bme280_settings *settings);

    /*!
     * @brief This internal API reloads the already existing device settings in the
     * sensor after soft reset.
     *
     * @param[in] dev : Structure instance of bme280_dev.
     * @param[in] settings : Pointer variable which contains the settings to
     * be set in the sensor.
     *
     * @return Result of API execution status
     * @retval zero -> Success / +ve value -> Warning / -ve value -> Error
     */
    int8_t reload_device_settings(const struct bme280_settings *settings, const struct bme280_dev *dev);

    /*!
     *  @brief This API is used to parse the pressure, temperature and
     *  humidity data and store it in the bme280_uncomp_data structure instance.
     *
     *  @param[in] reg_data     : Contains register data which needs to be parsed
     *  @param[out] uncomp_data : Contains the uncompensated pressure, temperature
     *  and humidity data.
     */
    void bme280_parse_sensor_data(const uint8_t *reg_data, struct bme280_uncomp_data *uncomp_data);

    /*!
     * @brief This API is used to compensate the pressure and/or
     * temperature and/or humidity data according to the component selected by the
     * user.
     *
     * @param[in] sensor_comp : Used to select pressure and/or temperature and/or
     * humidity.
     * @param[in] uncomp_data : Contains the uncompensated pressure, temperature and
     * humidity data.
     * @param[out] comp_data : Contains the compensated pressure and/or temperature
     * and/or humidity data.
     * @param[in] calib_data : Pointer to the calibration data structure.
     *
     * @return Result of API execution status.
     * @retval zero -> Success / -ve value -> Error
     */
    int8_t bme280_compensate_data(uint8_t sensor_comp,
                                  const struct bme280_uncomp_data *uncomp_data,
                                  struct bme280_data *comp_data,
                                  struct bme280_calib_data *calib_data);


    uint32_t _BME280_Addr;
};

#endif /* BME280_H_ */
/** @}*/
