/**
 * @brief       BM1383AGLV.cpp
 * @details     Ultra-Small, Low-Power, I2C-Compatible, 3.3-kSPS, 12-Bit ADCs. With Internal Reference, Oscillator, and Programmable Comparator.
 *              Function file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        22/July/2019
 * @version     22/July/2019    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */

#include "BM1383AGLV.h"


BM1383AGLV::BM1383AGLV ( PinName sda, PinName scl, uint32_t addr, uint32_t freq )
    : _i2c              ( sda, scl )
    , _BM1383AGLV_Addr  ( addr )
{
    _i2c.frequency( freq );
}


BM1383AGLV::~BM1383AGLV()
{
}


/**
 * @brief       BM1383AGLV_GetDeviceID ( BM1383AGLV_data_t* )
 *
 * @details     It gets the device identification.
 *
 * @param[in]    N/A
 *
 * @param[out]   myDeviceID:      Device ID ( ID1 and ID2 ).
 *
 *
 * @return       Status of BM1383AGLV_GetDeviceID.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         This function uses auto-increment.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetDeviceID ( BM1383AGLV_data_t* myDeviceID )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]     =   BM1383AGLV_ID1;
  aux        =   _i2c.write ( _BM1383AGLV_Addr, &cmd[0], 1U, true );
  aux        =   _i2c.read  ( _BM1383AGLV_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

  /* Parse the data  */
  myDeviceID->id1   =   cmd[0];
  myDeviceID->id2   =   cmd[1];



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_SetPowerDown ( BM1383AGLV_data_t )
 *
 * @details     It sets the power-down mode.
 *
 * @param[in]    myPwrDown:       Power down mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of BM1383AGLV_SetPowerDown.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_SetPowerDown ( BM1383AGLV_data_t myPwrDown )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Write the register   */
  cmd[0]     =   BM1383AGLV_POWER_DOWN;
  cmd[1]     =   myPwrDown.pwr_down;
  aux        =   _i2c.write ( _BM1383AGLV_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_GetPowerDown ( BM1383AGLV_data_t* )
 *
 * @details     It gets the power-down mode.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myPwrDown:       Power down mode.
 *
 *
 * @return       Status of BM1383AGLV_GetPowerDown.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetPowerDown ( BM1383AGLV_data_t* myPwrDown )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd    =   BM1383AGLV_POWER_DOWN;
  aux    =   _i2c.write ( _BM1383AGLV_Addr, &cmd, 1U, true );
  aux    =   _i2c.read  ( _BM1383AGLV_Addr, &cmd, 1U );

  /* Parse the data  */
  myPwrDown->pwr_down    =   (BM1383AGLV_power_down_pwr_down_t)cmd;



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_SetSoftReset ( BM1383AGLV_data_t )
 *
 * @details     It produces a soft-reset.
 *
 * @param[in]    myRSTB:          Reset mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of BM1383AGLV_SetSoftReset.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_SetSoftReset ( BM1383AGLV_data_t myRSTB )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Write the register   */
  cmd[0]     =   BM1383AGLV_RESET;
  cmd[1]     =   myRSTB.rstb;
  aux        =   _i2c.write ( _BM1383AGLV_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_GetSoftResetFlag ( BM1383AGLV_data_t* )
 *
 * @details     It gets the soft-reset flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRSTB:          Soft-reset flag.
 *
 *
 * @return       Status of BM1383AGLV_GetSoftResetFlag.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetSoftResetFlag ( BM1383AGLV_data_t* myRSTB )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd    =   BM1383AGLV_RESET;
  aux    =   _i2c.write ( _BM1383AGLV_Addr, &cmd, 1U, true );
  aux    =   _i2c.read  ( _BM1383AGLV_Addr, &cmd, 1U );

  /* Parse the data  */
  myRSTB->rstb   =   (BM1383AGLV_reset_rstb_t)cmd;



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_SetModeControl ( BM1383AGLV_data_t )
 *
 * @details     It sets the mode control.
 *
 * @param[in]    myModeCont:      AVE_NUM, DREN and MODE.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of BM1383AGLV_SetModeControl.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_SetModeControl ( BM1383AGLV_data_t myModeCont )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register to mask   */
  cmd[0]     =   BM1383AGLV_MODE_CONTROL;
  aux        =   _i2c.write ( _BM1383AGLV_Addr, &cmd[0], 1U, true );
  aux        =   _i2c.read  ( _BM1383AGLV_Addr, &cmd[1], 1U );

  /* Mask the register and update it     */
  cmd[1]    &=  ~( MODE_CONTROL_AVE_NUM_MASK | MODE_CONTROL_DREN_MASK | MODE_CONTROL_MODE_MASK );
  cmd[1]    |=   ( myModeCont.ave_num | myModeCont.dren | myModeCont.mode );
  aux        =   _i2c.write ( _BM1383AGLV_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_GetModeControl ( BM1383AGLV_data_t* )
 *
 * @details     It gets the mode control.
 *
 * @param[in]    N/A
 *
 * @param[out]   myModeCont:      AVE_NUM, DREN and MODE.
 *
 *
 * @return       Status of BM1383AGLV_GetModeControl.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetModeControl ( BM1383AGLV_data_t* myModeCont )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd    =   BM1383AGLV_MODE_CONTROL;
  aux    =   _i2c.write ( _BM1383AGLV_Addr, &cmd, 1U, true );
  aux    =   _i2c.read  ( _BM1383AGLV_Addr, &cmd, 1U );

  /* Parse the data  */
  myModeCont->ave_num    =   (BM1383AGLV_mode_control_ave_num_t)( cmd & MODE_CONTROL_AVE_NUM_MASK );
  myModeCont->dren       =   (BM1383AGLV_mode_control_dren_t)( cmd & MODE_CONTROL_DREN_MASK );
  myModeCont->mode       =   (BM1383AGLV_mode_control_mode_t)( cmd & MODE_CONTROL_MODE_MASK );



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_GetStatus ( BM1383AGLV_data_t* )
 *
 * @details     It gets the status flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRD_DRDY:       Pressure and temperature measurement data ready bit.
 *
 *
 * @return       Status of BM1383AGLV_GetStatus.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetStatus ( BM1383AGLV_data_t* myRD_DRDY )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd    =   BM1383AGLV_STATUS;
  aux    =   _i2c.write ( _BM1383AGLV_Addr, &cmd, 1U, true );
  aux    =   _i2c.read  ( _BM1383AGLV_Addr, &cmd, 1U );

  /* Parse the data  */
  myRD_DRDY->rd_drdy     =   (BM1383AGLV_reset_rd_drdy_t)( cmd );



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_GetRawPressure ( BM1383AGLV_data_t* )
 *
 * @details     It gets the raw pressure.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRawP:          Raw Pressure value.
 *
 *
 * @return       Status of BM1383AGLV_GetRawPressure.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         This function uses auto-increment.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetRawPressure ( BM1383AGLV_data_t* myRawP )
{
  char     cmd[3]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]     =   BM1383AGLV_PRESSURE_MSB;
  aux        =   _i2c.write ( _BM1383AGLV_Addr, &cmd[0], 1U, true );
  aux        =   _i2c.read  ( _BM1383AGLV_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

  /* Parse the data  */
  myRawP->rawPressure    =   cmd[0];
  myRawP->rawPressure  <<=   8U;
  myRawP->rawPressure   |=   cmd[1];
  myRawP->rawPressure  <<=   8U;
  myRawP->rawPressure   |=   cmd[2];

  //myRawP->rawPressure  >>=     2U;



  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_GetRawTemperature ( BM1383AGLV_data_t* )
 *
 * @details     It gets the raw temperature.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRawT:          Raw Temperature value.
 *
 *
 * @return       Status of BM1383AGLV_GetRawTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         This function uses auto-increment.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetRawTemperature ( BM1383AGLV_data_t* myRawT )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]     =   BM1383AGLV_TEMPERATURE_MSB;
  aux        =   _i2c.write ( _BM1383AGLV_Addr, &cmd[0], 1U, true );
  aux        =   _i2c.read  ( _BM1383AGLV_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

  /* Parse the data  */
  myRawT->rawTemperature    =    cmd[0];
  myRawT->rawTemperature  <<=    8U;
  myRawT->rawTemperature   |=    cmd[1];




  if ( aux == I2C_SUCCESS )
  {
    return   BM1383AGLV_SUCCESS;
  }
  else
  {
    return   BM1383AGLV_FAILURE;
  }
}



/**
 * @brief       BM1383AGLV_GetPressure ( BM1383AGLV_data_t* )
 *
 * @details     It gets the pressure.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myP:             Pressure value.
 *
 *
 * @return       Status of BM1383AGLV_GetPressure.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetPressure ( BM1383AGLV_data_t* myP )
{
  BM1383AGLV_status_t aux;

  /* Read the register   */
  aux    =   BM1383AGLV_GetRawPressure ( &(*myP) );

  /* Parse the data  */
  myP->pressure  =   ( ( ( ( myP->rawPressure >> 16U ) & 0xFF ) * 16384.0 ) + ( ( myP->rawPressure >> 8U ) & 0xFF ) * 64.0 ) + ( myP->rawPressure & 0xFF );
  myP->pressure /=  2048.0;



  return   aux;
}



/**
 * @brief       BM1383AGLV_GetTemperature ( BM1383AGLV_data_t* )
 *
 * @details     It gets the temperature.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myT:             Temperature value.
 *
 *
 * @return       Status of BM1383AGLV_GetTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        02/October/2019
 * @version     02/October/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
BM1383AGLV::BM1383AGLV_status_t BM1383AGLV::BM1383AGLV_GetTemperature ( BM1383AGLV_data_t* myT )
{
  BM1383AGLV_status_t aux;

  /* Read the register   */
  aux    =   BM1383AGLV_GetRawTemperature ( &(*myT) );

  /* Parse the data  */
  myT->temperature   =   ( ( ( ( myT->rawTemperature >> 8U ) & 0xFF ) * 256.0 ) + ( myT->rawTemperature & 0xFF ) );
  myT->temperature  /=  32.0;



  return   aux;
}