/**
 * @brief       BH1750.h
 * @details     Digital 16-bit Serial Output Type Ambient Light Sensor IC.
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        10/August/2017
 * @version     10/August/2017    The ORIGIN
 * @pre         NaN
 * @warning     NaN
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef BH1750_H
#define BH1750_H

#include "mbed.h"


/**
    Example:
    
    #include "mbed.h"
    #include "BH1750.h"

    BH1750 myLightSensor ( I2C_SDA, I2C_SCL, BH1750::BH1750_ADDR_L, 400000 );   //[todo]
    Serial pc            ( USBTX, USBRX );                              // tx, rx

    Ticker serial;

    DigitalOut myled(LED1);


    void sendDATA ( void )
    {
        float    myLux         =       0;
    
        myled = 0;
    
        myLightSensor.BH1750_TriggerMeasurement ( BH1750::BH1750_ONE_TIME_H_RESOLUTION_MODE );
        wait ( 1 );
        myLightSensor.BH1750_ReadLux    ( &myLux );
    
        pc.printf( "Lux: %0.1f\r\n", myLux );
    
        myled = 1;

    }


    int main() {
        pc.baud ( 115200 );
        serial.attach( &sendDATA, 3 );                      // the address of the function to be attached ( sendDATA ) and the interval ( 3s )
    
        // Let the callbacks take care of everything
        while(1)  sleep();
    }
*/

/*!
 Library for the BH1750 Digital 16-bit Serial Output Type Ambient Light Sensor IC.
*/
class BH1750
{
public:
    /** Represents the different I2C address possibilities for the BH1750
       */
    enum BH1750_ADDRESS {
        BH1750_ADDR_L   =    ( 0x23 << 1 ),         /**< Addr pin = GND */
        BH1750_ADDR_H   =    ( 0x5C << 1 )          /**< Addr pin = VDD */
    };

    enum BH1750_COMMANDS {
        BH1750_POWER_DOWN                        =      0x00,        /*!<   No active state                                                                                                                                  */
        BH1750_POWER_ON                          =      0x01,        /*!<   Waiting for measurement command                                                                                                                  */
        BH1750_RESET                             =      0x07,        /*!<   Reset Data register value. Reset command is not acceptable in Power Down mode                                                                    */
        BH1750_CONTINUOUSLY_H_RESOLUTION_MODE    =      0x10,        /*!<   Start measurement at 1lx resolution. Measurement Time is typically 120ms                                                                         */
        BH1750_CONTINUOUSLY_H_RESOLUTION_MODE2   =      0x11,        /*!<   Start measurement at 0.5lx resolution. Measurement Time is typically 120ms                                                                       */
        BH1750_CONTINUOUSLY_L_RESOLUTION_MODE    =      0x13,        /*!<   Start measurement at 4lx resolution. Measurement Time is typically 16ms                                                                          */
        BH1750_ONE_TIME_H_RESOLUTION_MODE        =      0x20,        /*!<   Start measurement at 1lx resolution. Measurement Time is typically 120ms. It is automatically set to Power Down mode after measurement           */
        BH1750_ONE_TIME_H_RESOLUTION_MODE2       =      0x21,        /*!<   Start measurement at 0.5lx resolution. Measurement Time is typically 120ms. It is automatically set to Power Down mode after measurement         */
        BH1750_ONE_TIME_L_RESOLUTION_MODE        =      0x23         /*!<   Start measurement at 4lx resolution. Measurement Time is typically 16ms. It is automatically set to Power Down mode after measurement            */
    };

    enum BH1750_SENSITIVITY {
        BH1750_SENSITIVITY_DEFAULT               =      0x45         /*!<   Measurement Time Register by default. This is for registration of measurement time                                                               */
    };


    enum BH1750_CONSTANT {
        BH1750_SUCCESS                           =      0x00,        /*!<   The communication was fine                                                                                                                       */
        BH1750_FAILURE                           =      0x01,        /*!<   Something went wrong                                                                                                                             */
        I2C_SUCCESS                              =      0x00         /*!<   I2C communication was fine                                                                                                                       */
    };


    /** Create an BH1750 object connected to the specified I2C pins.
     *
     * @param sda     I2C data pin
     * @param scl     I2C clock pin
     * @param addr    I2C slave address
     * @param freq    I2C frequency in Hz.
     */
    BH1750 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete BH1750 object.
     */
    ~BH1750();


    /** It put the BH1750 sensor in the lowest power mode.
     */
    uint32_t  BH1750_PowerDown          ( void );
    
    /** It turns on the BH1750 sensor.
     */
    uint32_t  BH1750_PowerOn            ( void );
    
    /** It clears the Data register.
     */
    uint32_t  BH1750_ResetDataRegister  ( void );
    
    /** Create an BH1750 object connected to the specified I2C pins.
     * @param mode    One-shot/Continuous mode in High, High2 or Low resolution.
     */
    uint32_t  BH1750_TriggerMeasurement ( BH1750_COMMANDS mode );
    
    /** It configures a new sensitivity.
     */
    uint32_t  BH1750_NewSensitivity     ( uint8_t newSensitivity );
    
    /** It reads the raw data from the BH1750 sensor.
     * @param myRawData    2-Byte array.
     */
    uint32_t  BH1750_ReadRawData        ( char* myRawData );
    
    /** It reads a converted data from the BH1750 sensor.
     * @param myLux    Light value.
     */
    uint32_t  BH1750_ReadLux            ( float* myLux );


private:
    I2C      i2c;
    uint32_t BH1750_Addr;
    uint32_t BH1750_Mode;
};

#endif
