/**
 * @brief       AHT20.cpp
 * @details     Humidity and Temperature Sensor.
 *              Function file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ). 
 */

#include "AHT20.h"


AHT20::AHT20 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq )
    : _i2c           ( sda, scl )
    , _AHT20_Addr ( addr )
{
    _i2c.frequency( freq );
}


AHT20::~AHT20()
{
}



/**
 * @brief       AHT20_Calibrate    ( void )
 * @details     It calibrates the device.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Status of AHT20_Calibrate
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022        The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
AHT20::AHT20_status_t  AHT20::AHT20_Calibrate ( void )
{
    char     cmd[3]  =   { 0 };
    uint32_t aux;


    /* Update the register  */
    cmd[0]   =   AHT20_INITIALIZATION;
    cmd[1]   =   INITIALIZATION_DATA_1;
    cmd[2]   =   INITIALIZATION_DATA_2;
    aux      =   _i2c.write ( _AHT20_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




    if ( aux == I2C_SUCCESS )
    {
        return   AHT20_SUCCESS;
    }
    else
    {
        return   AHT20_FAILURE;
    }
}



/**
 * @brief       AHT20_SoftReset    ( void )
 * @details     It performs a soft-reset.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Status of AHT20_SoftReset
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022        The ORIGIN
 * @pre         The user must consider that the time required for soft-reset does not exceed 20 ms.
 * @warning     N/A.
 */
AHT20::AHT20_status_t  AHT20::AHT20_SoftReset ( void )
{
    char     cmd    =   0U;
    uint32_t aux;


    /* Update the register  */
    cmd  =   AHT20_SOFTRESET;
    aux  =   _i2c.write ( _AHT20_Addr, (char*)&cmd, 1U, false );




    if ( aux == I2C_SUCCESS )
    {
        return   AHT20_SUCCESS;
    }
    else
    {
        return   AHT20_FAILURE;
    }
}



/**
 * @brief       AHT20_TriggerMeasurement ( void )
 * @details     It triggers a new measurement data (raw data).
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Status of AHT20_TriggerMeasurement
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022        The ORIGIN
 * @pre         The user must consider to wait for 80ms for the measurement to be completed.
 * @warning     N/A.
 */
AHT20::AHT20_status_t  AHT20::AHT20_TriggerMeasurement ( void )
{
    char     cmd[3]  =   {0U};
    uint32_t aux;


    /* Update the register  */
    cmd[0]   =   AHT20_TRIGGER_MEASUREMENT;
    cmd[1]   =   TRIGGER_MEASUREMENT_DATA_1;
    cmd[2]   =   TRIGGER_MEASUREMENT_DATA_2;
    aux      =   _i2c.write ( _AHT20_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



    if ( aux == I2C_SUCCESS )
    {
        return   AHT20_SUCCESS;
    }
    else
    {
        return   AHT20_FAILURE;
    }
}



/**
 * @brief       AHT20_TriggerStatus ( void )
 * @details     It triggers the state byte.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Status of AHT20_TriggerStatus
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022        The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
AHT20::AHT20_status_t  AHT20::AHT20_TriggerStatus ( void )
{
    char     cmd    =   0U;
    uint32_t aux;

    /* Update the register   */
    cmd  =   AHT20_STATUS;
    aux  =   _i2c.write ( _AHT20_Addr, (char*)&cmd, 1U, false );



    if ( aux == I2C_SUCCESS )
    {
        return   AHT20_SUCCESS;
    }
    else
    {
        return   AHT20_FAILURE;
    }
}



/**
 * @brief       AHT20_GetStatus ( uint8_t* )
 * @details     It gets the state byte.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myState:           State byte.
 *
 *
 * @return      Status of AHT20_GetStatus
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022        The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
AHT20::AHT20_status_t  AHT20::AHT20_GetStatus ( uint8_t* myState )
{
    uint32_t aux;

    /* Read the register     */
    aux  =   _i2c.read ( _AHT20_Addr, (char*)&myState, 1U );


    if ( aux == I2C_SUCCESS )
    {
        return   AHT20_SUCCESS;
    }
    else
    {
        return   AHT20_FAILURE;
    }
}



/**
 * @brief       AHT20_GetAllData ( AHT20_user_data_t* )
 * @details     It gets all the raw data.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myAllData:         All the data.
 *
 *
 * @return      Status of AHT20_GetAllData
 *
 * @author      Manuel Caballero
 * @date        08/February/2022        The ORIGIN
 * @pre         AHT20_TriggerMeasurement function must be called before using AHT20_GetAllData function.
 * @pre         The user must consider to wait for 80ms for the measurement to be completed.
 * @warning     N/A.
 */
AHT20::AHT20_status_t  AHT20::AHT20_GetAllData ( AHT20_user_data_t* myAllData )
{
    char     cmd[7] =   {0U};
    uint32_t aux;


    /* Read the register     */
    aux  =   _i2c.read ( _AHT20_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

    /* Parse the data    */
    myAllData->state                         =   cmd[0];

    myAllData->humidity.raw_humidity         =   cmd[1];
    myAllData->humidity.raw_humidity       <<=   8U;
    myAllData->humidity.raw_humidity        |=   cmd[2];
    myAllData->humidity.raw_humidity       <<=   8U;
    myAllData->humidity.raw_humidity        |=   ( cmd[3] & 0b11110000 );
    myAllData->humidity.raw_humidity       >>=   4U;

    myAllData->temperature.raw_temperature   =   ( cmd[3] & 0b00001111 );
    myAllData->temperature.raw_temperature <<=   8U;
    myAllData->temperature.raw_temperature  |=   cmd[4];
    myAllData->temperature.raw_temperature <<=   8U;
    myAllData->temperature.raw_temperature  |=   cmd[5];

    myAllData->crc                           =   cmd[6];



    if ( aux == I2C_SUCCESS )
    {
        return   AHT20_SUCCESS;
    }
    else
    {
        return   AHT20_FAILURE;
    }
}



/**
 * @brief       AHT20_ProcessTemperature ( uint32_t myRawTemperature )
 * @details     It processes the temperature data.
 *
 * @param[in]    myRawTemperature:   Raw temperature data.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Temperature data is processed
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022        The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
float  AHT20::AHT20_ProcessTemperature ( uint32_t myRawTemperature )
{
    /* Calculate the temperature     */
    return  ( ( myRawTemperature * 200.0 )/1048576.0 - 50.0 );
}



/**
 * @brief       AHT20_ProcessHumidity ( uint32_t myRawHumidity )
 * @details     It processes the humidity data.
 *
 * @param[in]    myRawHumidity:   Raw humidity data.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Humidity data is processed
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022        The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
float  AHT20::AHT20_ProcessHumidity ( uint32_t myRawHumidity )
{
    /* Calculate the humidity    */
    return  ( myRawHumidity * 100.0 )/1048576.0;
}