/**
 * @brief       AHT20.h
 * @details     Humidity and Temperature Sensor.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        08/February/2022
 * @version     08/February/2022    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef AHT20_H
#define AHT20_H

#include "mbed.h"


/**
    Example:
@code
#include "mbed.h"
#include "AHT20.h"

AHT20 myAHT20   ( I2C_SDA, I2C_SCL, AHT20::AHT20_ADDRESS, 100000 );
Serial pc       ( USBTX, USBRX );

DigitalOut  myled ( LED1 );
Ticker      newReading;


//@brief Variables.
uint32_t    myState = 0UL;

//@brief   FUNCTION PROTOTYPES
void    changeDATA     ( void );



//@brief FUNCTION FOR APPLICATION MAIN ENTRY.
int main()
{
    AHT20::AHT20_user_data_t    myAHT20_Data;
    AHT20::AHT20_status_t       aux;

    pc.baud ( 115200 );

    myled   =   1;
    wait(3);
    myled   =   0;

    // Reset the device
    aux  =   myAHT20.AHT20_SoftReset ();
    wait_ms(80);

    // Get the state byte
    aux  =   myAHT20.AHT20_TriggerStatus ();
    wait_ms(80);

    aux  =   myAHT20.AHT20_GetStatus ( (uint8_t*)&myAHT20_Data.state );

    // Check if the device is calibrated
    if ( ( myAHT20_Data.state & AHT20::STATUS_CAL_MASK ) == AHT20::STATUS_CAL_UNCALIBRATED ) {
        // Calibrate the sensor
        aux  =   myAHT20.AHT20_Calibrate ();
    }

    newReading.attach( &changeDATA, 1 );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 1s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        myled = 1;

        if ( myState == 1UL ) {
            // Trigger a new measurement data
            aux  =  myAHT20.AHT20_TriggerMeasurement ();
            wait_ms(80);

            // Get the temperature value
            aux  =  myAHT20.AHT20_GetAllData ( (AHT20::AHT20_user_data_t*)&myAHT20_Data );

            // Process the temperature data
            myAHT20_Data.temperature.temperature = myAHT20.AHT20_ProcessTemperature ( myAHT20_Data.temperature.raw_temperature );

            // Process the humidity data
            myAHT20_Data.humidity.humidity = myAHT20.AHT20_ProcessHumidity ( myAHT20_Data.humidity.raw_humidity );

            // Transmit result over the UART
            pc.printf( "Temp: %0.2f C | RH: %0.3f %%\r\n", myAHT20_Data.temperature.temperature, myAHT20_Data.humidity.humidity  );

            myState  =   0UL;                                                   // Reset the variable
        }

        myled = 0;
    }
}


// @brief       changeDATA ( void  )
//
// @details     It changes myState variable
//
// @param[in]    N/A
//
// @param[out]   N/A.
//
//
// @return       N/A.
//
//
// @author      Manuel Caballero
// @date        08/February/2022
// @version     08/February/2022   The ORIGIN
// @pre         N/A
// @warning     N/A.
void changeDATA ( void )
{
    myState = 1UL;
}
@endcode
*/


/*!
 Library for the AHT20 Humidity and Temperature Sensor.
*/
class AHT20
{
public:
    /**
    * @brief   DEFAULT ADDRESS
    */
    typedef enum {
        AHT20_ADDRESS     =   ( 0x38 << 1U )           /*!<   AHT20 I2C Address                    */
    } AHT20_address_t;




    /**
      * @brief   COMMAND REGISTERS
      */
    typedef enum {
        AHT20_TRIGGER_MEASUREMENT   =   0xAC,           /*!<  Trigger measurement command           */
        AHT20_STATUS                =   0x71,           /*!<  Status command                        */
        AHT20_INITIALIZATION        =   0xBE,           /*!<  Initialization                        */
        AHT20_SOFTRESET             =   0xBA            /*!<  Soft reset                            */
    } AHT20_basic_commands_t;



// TRIGGER MESUREMENT
    /**
      * @brief   DATA
      *             NOTE: N/A
      */
    typedef enum {
        TRIGGER_MEASUREMENT_DATA_1  =   0x33,           /*!<  Data 1                                */
        TRIGGER_MEASUREMENT_DATA_2  =   0x00            /*!<  Data 2                                */
    } AHT20_trigger_measurement_t;



// STATUS
    /**
      * @brief   BUS_INDICATION <7>
      *             NOTE: N/A
      */
    typedef enum {
        STATUS_BUS_INDICATION_MASK  =   ( 1U << 7U ),   /*!<  BUS_INDICATION mask                   */
        STATUS_BUS_INDICATION_BUSY  =   ( 1U << 7U ),   /*!<  Busy in measurement                   */
        STATUS_BUS_INDICATION_FREE  =   ( 0U << 7U )    /*!<  Free in dormant state                 */
    } AHT20_status_bus_indication_t;



    /**
      * @brief   CAL <3>
      *             NOTE: N/A
      */
    typedef enum {
        STATUS_CAL_MASK             =   ( 1U << 3U ),   /*!<  CAL mask                              */
        STATUS_CAL_CALIBRATED       =   ( 1U << 3U ),   /*!<  Calibrated                            */
        STATUS_CAL_UNCALIBRATED     =   ( 0U << 3U )    /*!<  Uncalibrated                          */
    } AHT20_status_cal_t;



// INITIALIZATION
    /**
      * @brief   INITIALIZATION
      *             NOTE: N/A
      */
    typedef enum {
        INITIALIZATION_DATA_1       =   0x08,           /*!<  Data 1                                */
        INITIALIZATION_DATA_2       =   0x00            /*!<  Data 2                                */
    } AHT20_initialization_t;





#ifndef AHT20_VECTOR_STRUCT_H
#define AHT20_VECTOR_STRUCT_H
    /* Temperature data */
    typedef struct {
        uint32_t raw_temperature;                       /*!<  Raw temperature data                  */
        float    temperature;                           /*!<  Temperature                           */
    } AHT20_temperature_data_t;


    /* Humidity data    */
    typedef struct {
        uint32_t raw_humidity;                          /*!<  Raw humidity data                     */
        float    humidity;                              /*!<  Humidity                              */
    } AHT20_humidity_data_t;


    /* User's variables */
    typedef struct {
        AHT20_temperature_data_t    temperature;        /*!<  Temperature variables                 */
        AHT20_humidity_data_t       humidity;           /*!<  Humidity variables                    */

        uint8_t                     state;              /*!<  State                                 */

        uint8_t                     crc;                /*!<  CRC                                   */
    } AHT20_user_data_t;
#endif


    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        AHT20_SUCCESS    =       0U,
        AHT20_FAILURE    =       1U,
        I2C_SUCCESS      =       0U                   /*!<   I2C communication was fine        */
    } AHT20_status_t;




    /** Create an AHT20 object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency
      */
    AHT20 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete AHT20 object.
     */
    ~AHT20();

    /** It calibrates the device.
    */
    AHT20_status_t  AHT20_Calibrate             ( void                          );

    /** It performs a soft-reset.
     */
    AHT20_status_t  AHT20_SoftReset             ( void                          );

    /** It triggers a new measurement data (raw data).
     */
    AHT20_status_t  AHT20_TriggerMeasurement    ( void                          );

    /** It triggers the state byte.
     */
    AHT20_status_t  AHT20_TriggerStatus         ( void                          );

    /** It gets the state byte.
     */
    AHT20_status_t  AHT20_GetStatus             ( uint8_t* myState              );

    /** It gets all the raw data.
     */
    AHT20_status_t  AHT20_GetAllData            ( AHT20_user_data_t* myAllData  );

    /** It processes the temperature data.
     */
    float  AHT20_ProcessTemperature             ( uint32_t myRawTemperature     );

    /** It processes the humidity data.
     */
    float  AHT20_ProcessHumidity                ( uint32_t myRawHumidity        );



private:
    I2C      _i2c;
    uint32_t _AHT20_Addr;
};

#endif
