/**
 * @brief       ADXL345.h
 * @details     3-Axis, ±2 g/±4 g/±8 g/±16 g Digital Accelerometer.
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        19/August/2017
 * @version     19/August/2017    The ORIGIN
 * @pre         Although the ADXL345 can be handled by SPI or I2C, this library is only
 *              adapted to be used by I2C.
 * @pre         FIFO functions are NOT implemented.
 * @warning     NaN
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef ADXL345_H
#define ADXL345_H

#include "mbed.h"


/**
    Example:

#include "mbed.h"
#include "BH1750.h"
#include "ADXL345.h"

ADXL345 myAccelerometer ( I2C_SDA, I2C_SCL, ADXL345::ADXL345_ALT_ADDRESS_LOW, 400000 );  
Serial pc               ( USBTX, USBRX );                              // tx, rx


Ticker serial;

DigitalOut myled(LED1);


void sendDATA ( void )
{
    ADXL345::Vector_f    myXYZvalues;

    myled = 0;

    myAccelerometer.ADXL345_ReadScaledData ( &myXYZvalues );
    pc.printf( "X: %0.3f, Y: %0.3f, Z: %0.3f\r\n", myXYZvalues.Xmg_Axis, myXYZvalues.Ymg_Axis, myXYZvalues.Zmg_Axis );

    myled = 1;

}


int main() {
    char myID = 0;

    pc.baud ( 115200 );

    myAccelerometer.ADXL345_Init   ( ADXL345::BW_RATE_LOW_POWER_Disabled, ADXL345::BW_RATE_RATE_100HZ, ADXL345::DATA_FORMAT_INT_INVERT_Disabled, ADXL345::DATA_FORMAT_FULL_RES_Enabled,
                                     ADXL345::DATA_FORMAT_JUSTIFY_Disabled, ADXL345::DATA_FORMAT_RANGE_4_G );


    myAccelerometer.ADXL345_PowerMode ( ADXL345::MEASURE_MODE );

    myAccelerometer.ADXL345_GetID ( &myID );
    pc.printf( "ID: %x\r\n", myID );

    serial.attach( &sendDATA, 0.5 );                      // the address of the function to be attached ( sendDATA ) and the interval ( 0.5s )

    while(1)  sleep();
}
*/

/*!
 Library for the ADXL345 3-Axis, ±2 g/±4 g/±8 g/±16 g Digital Accelerometer.
*/
class ADXL345
{
public:
    /**
    * @brief   DEFAULT ADDRESSES
    */
    typedef enum {
        ADXL345_ALT_ADDRESS_LOW     =   ( 0x53 << 1 ),                          /**< Addr pin = GND */
        ADXL345_ALT_ADDRESS_HIGH    =   ( 0x1D << 1 )                           /**< Addr pin = VDD */
    } ADXL345_address_t;


    /**
    * @brief   REGISTERS MAP
    */
#define ADXL345_DEVID                           0x00                            /*!<   Device ID                                                    */
#define ADXL345_THRESH_TAP                      0x1D                            /*!<   Tap threshold                                                */
#define ADXL345_OFSX                            0x1E                            /*!<   X-axis offset                                                */
#define ADXL345_OFSY                            0x1F                            /*!<   Y-axis offset                                                */
#define ADXL345_OFSZ                            0x20                            /*!<   Z-axis offset                                                */
#define ADXL345_DUR                             0x21                            /*!<   Tap duration                                                 */
#define ADXL345_LATENT                          0x22                            /*!<   Tap latency                                                  */
#define ADXL345_WINDOW                          0x23                            /*!<   Tap window                                                   */
#define ADXL345_THRESH_ACT                      0x24                            /*!<   Activity threshold                                           */
#define ADXL345_THRESH_INACT                    0x25                            /*!<   Inactivity threshold                                         */
#define ADXL345_TIME_INACT                      0x26                            /*!<   Inactivity time                                              */
#define ADXL345_ACT_INACT_CTL                   0x27                            /*!<   Axis enable control for activity and inactivity detection    */
#define ADXL345_THRESH_FF                       0x28                            /*!<   Free-fall threshold                                          */
#define ADXL345_TIME_FF                         0x29                            /*!<   Free-fall time                                               */
#define ADXL345_TAP_AXES                        0x2A                            /*!<   Axis control for single tap/double tap                       */
#define ADXL345_ACT_TAP_STATUS                  0x2B                            /*!<   Source of single tap/double tap                              */
#define ADXL345_BW_RATE                         0x2C                            /*!<   Data rate and power mode control                             */
#define ADXL345_POWER_CTL                       0x2D                            /*!<   Power-saving features control                                */
#define ADXL345_INT_ENABLE                      0x2E                            /*!<   Interrupt enable control                                     */
#define ADXL345_INT_MAP                         0x2F                            /*!<   Interrupt mapping control                                    */
#define ADXL345_INT_SOURCE                      0x30                            /*!<   Source of interrupts                                         */
#define ADXL345_DATA_FORMAT                     0x31                            /*!<   Data format control                                          */
#define ADXL345_DATAX0                          0x32                            /*!<   X-Axis Data 0                                                */
#define ADXL345_DATAX1                          0x33                            /*!<   X-Axis Data 1                                                */
#define ADXL345_DATAY0                          0x34                            /*!<   Y-Axis Data 0                                                */
#define ADXL345_DATAY1                          0x35                            /*!<   Y-Axis Data 1                                                */
#define ADXL345_DATAZ0                          0x36                            /*!<   Z-Axis Data 0                                                */
#define ADXL345_DATAZ1                          0x37                            /*!<   Z-Axis Data 1                                                */
#define ADXL345_FIFO_CTL                        0x38                            /*!<   FIFO control                                                 */
#define ADXL345_FIFO_STATUS                     0x39                            /*!<   FIFO status                                                  */


    /* Commands Registers */
    /**
    * @brief   DEVID
    */
#define DEVID_ID                                0xE5                            /*!<   A fixed device ID code                                       */


    /**
    * @brief   ACT_INACT_CTL
    */
    /* Bit 7 : ACT ac/dc ( D7 ). */
    typedef enum {
        ACTIVITY_DC_Enabled     =   1,                                          /*!<  Activity DC-Coupled Enabled.                                   */
        ACTIVITY_AC_Enabled     =   2                                           /*!<  Activity AC-Coupled Enabled.                                   */
    } ADXL345_act_inact_ctl_activity_dc_coupled_t;


    /* Bit 6 - 4 : ACT_X ( D6 ), ACT_Y ( D5 ) y ACT_Z enable ( D4 ). */
    typedef enum {
        ACT_X_Enabled     =   1,                                                /*!<  X-axis Enabled.                                                */
        ACT_Y_Enabled     =   2,                                                /*!<  Y-axis Enabled.                                                */
        ACT_Z_Enabled     =   3,                                                /*!<  Z-axis Enabled.                                                */
        ACT_YZ_Enabled    =   4,                                                /*!<  YZ-axis Enabled.                                               */
        ACT_XZ_Enabled    =   5,                                                /*!<  XZ-axis Enabled.                                               */
        ACT_XY_Enabled    =   6,                                                /*!<  XY-axis Enabled.                                               */
        ACT_XYZ_Enabled   =   7,                                                /*!<  XYZ-axis Enabled.                                              */
        ACT_XYZ_Disabled  =   8                                                 /*!<  XYZ-axis Disabled.                                             */
    } ADXL345_act_inact_ctl_activity_axis_t;


    /* Bit 3 : INACT ac/dc ( D3 ). */
    typedef enum {
        INACTIVITY_DC_Enabled     =   1,                                        /*!<  Inactivity DC-Coupled Enabled.                                 */
        INACTIVITY_AC_Enabled     =   2                                         /*!<  Inactivity AC-Coupled Enabled.                                 */
    } ADXL345_act_inact_ctl_inactivity_dc_coupled_t;


    /* Bit 2 - 0 : INACT_X ( D2 ), INACT_X ( D1 ) y INACT_X enable ( D0 ). */
    typedef enum {
        INACT_X_Enabled     =   1,                                              /*!<  X-axis Enabled.                                                */
        INACT_Y_Enabled     =   2,                                              /*!<  Y-axis Enabled.                                                */
        INACT_Z_Enabled     =   3,                                              /*!<  Z-axis Enabled.                                                */
        INACT_YZ_Enabled    =   4,                                              /*!<  YZ-axis Enabled.                                               */
        INACT_XZ_Enabled    =   5,                                              /*!<  XZ-axis Enabled.                                               */
        INACT_XY_Enabled    =   6,                                              /*!<  XY-axis Enabled.                                               */
        INACT_XYZ_Enabled   =   7,                                              /*!<  XYZ-axis Enabled.                                              */
        INACT_XYZ_Disabled  =   8                                               /*!<  XYZ-axis Disabled.                                             */
    } ADXL345_act_inact_ctl_inactivity_axis_t;



    /**
    * @brief   TAP_AXES
    */
    /* Bit 3 : SUPPRESS ( D3 ). */
    typedef enum {
        SUPPRESS_Enabled     =   1,                                             /*!<  SUPPRESS Enabled.                                             */
        SUPPRESS_Disabled    =   2                                              /*!<  SUPPRESS Disabled.                                            */
    } ADXL345_tap_axes_suppress_t;

    /* Bit 2 - 0 : TAP_X ( D2 ), TAP_Y ( D1 ) y TAP_Z enable ( D0 ). */
    typedef enum {
        TAP_X_Enabled     =   1,                                                /*!<  X-axis Enabled.                                                */
        TAP_Y_Enabled     =   2,                                                /*!<  Y-axis Enabled.                                                */
        TAP_Z_Enabled     =   3,                                                /*!<  Z-axis Enabled.                                                */
        TAP_YZ_Enabled    =   4,                                                /*!<  YZ-axis Enabled.                                               */
        TAP_XZ_Enabled    =   5,                                                /*!<  XZ-axis Enabled.                                               */
        TAP_XY_Enabled    =   6,                                                /*!<  XY-axis Enabled.                                               */
        TAP_XYZ_Enabled   =   7,                                                /*!<  XYZ-axis Enabled.                                              */
        TAP_XYZ_Disabled  =   8                                                 /*!<  XYZ-axis Disabled.                                             */
    } ADXL345_tap_axes_axis_t;



    /**
    * @brief   BW_RATE
    */
    /* Bit 4 : LOW_POWER ( D4 ). */
    typedef enum {
        BW_RATE_LOW_POWER_Enabled       =       true,                           /*!<  LOW_POWER Enabled.                                              */
        BW_RATE_LOW_POWER_Disabled      =       false                           /*!<  LOW_POWER Disabled.                                             */
    } AXDL345_bw_rate_low_power_t;


    /* Bit 3 - 0 : RATE ( D3 - D0 ). */
    /* Normal operation */
    typedef enum {
        BW_RATE_RATE_3200HZ         =       0x0F,                               /*!<  RATE 3200 Hz.                                                   */
        BW_RATE_RATE_1600HZ         =       0x0E,                               /*!<  RATE 1600 Hz.                                                   */
        BW_RATE_RATE_800HZ          =       0x0D,                               /*!<  RATE 800 Hz.                                                    */
        BW_RATE_RATE_400HZ          =       0x0C,                               /*!<  RATE 400 Hz.                                                    */
        BW_RATE_RATE_200HZ          =       0x0B,                               /*!<  RATE 200 Hz.                                                    */
        BW_RATE_RATE_100HZ          =       0x0A,                               /*!<  RATE 100 Hz. ( DEFAULT )                                        */
        BW_RATE_RATE_50HZ           =       0x09,                               /*!<  RATE 50 Hz.                                                     */
        BW_RATE_RATE_25HZ           =       0x08,                               /*!<  RATE 25 Hz.                                                     */
        BW_RATE_RATE_12_5HZ         =       0x07,                               /*!<  RATE 12.5 Hz.                                                   */
        BW_RATE_RATE_6_25HZ         =       0x06,                               /*!<  RATE 6.25 Hz.                                                   */
        BW_RATE_RATE_3_13HZ         =       0x05,                               /*!<  RATE 3.13 Hz.                                                   */
        BW_RATE_RATE_1_56HZ         =       0x04,                               /*!<  RATE 1.56 Hz.                                                   */
        BW_RATE_RATE_0_78HZ         =       0x03,                               /*!<  RATE 0.78 Hz.                                                   */
        BW_RATE_RATE_0_39HZ         =       0x02,                               /*!<  RATE 0.39 Hz.                                                   */
        BW_RATE_RATE_0_20HZ         =       0x01,                               /*!<  RATE 0.20 Hz.                                                   */
        BW_RATE_RATE_0_10HZ         =       0x00                                /*!<  RATE 0.10 Hz.                                                   */
    } AXDL345_bw_rate_rate_t;



    /**
      * @brief   POWER_CTL
      */
    /* Bit 5 : LINK ( D5 ). */
    typedef enum {
        LINK_Enabled                =       1,                                  /*!<  LINK Enabled.                                                   */
        LINK_Disabled               =       0                                   /*!<  LINK Disabled                                                   */
    } AXDL345_power_ctl_link_t;

    /* Bit 4 : AUTO_SLEEP ( D4 ). */
    typedef enum {
        AUTO_SLEEP_Enabled          =       1,                                  /*!<  AUTO_SLEEP Enabled.                                              */
        AUTO_SLEEP_Disabled         =       0                                   /*!<  AUTO_SLEEP Disabled                                              */
    } AXDL345_power_ctl_auto_sleep_t;

    /* Bit 3 : MEASURE ( D3 ). */
    typedef enum {
        MEASURE_MODE                =       true,                               /*!<  MEASURE Enabled.                                                 */
        STANDBY_MODE                =       false                               /*!<  MEASURE Disabled, Standby mode.                                  */
    } AXDL345_power_ctl_measure_t;

    /* Bit 2 : SLEEP ( D2 ). */
    typedef enum {
        SLEEP_Enabled               =       1,                                  /*!< SLEEP Enabled.                                                    */
        SLEEP_Disabled              =       0                                   /*!< SLEEP Disabled                                                    */
    } AXDL345_power_ctl_sleep_t;

    /* Bit 1 - 0 : WAKEUP ( D1 - D0 ). */
    typedef enum {
        WAKEUP_FREQ_8HZ             =       0,                                  /*!< WAKEUP frequency 8Hz.                                             */
        WAKEUP_FREQ_4HZ             =       1,                                  /*!< WAKEUP frequency 4Hz.                                             */
        WAKEUP_FREQ_2HZ             =       2,                                  /*!< WAKEUP frequency 2Hz.                                             */
        WAKEUP_FREQ_1HZ             =       3                                   /*!< WAKEUP frequency 1Hz.                                             */
    } AXDL345_power_ctl_wakeup_t;


    /**
      * @brief   INT_MAP
      */
    /* Bit 7 - 0 : */
    typedef enum {
        INT_DATA_READY              =       0x80,                               /*!<  Data ready.                                                       */
        INT_SINGLE_TAP              =       0x40,                               /*!<  single tap.                                                       */
        INT_DOUBLE_TAP              =       0x20,                               /*!<  double tap.                                                       */
        INT_ACTIVITY                =       0x10,                               /*!<  activity.                                                         */
        INT_INACTIVITY              =       0x08,                               /*!<  inactivity.                                                       */
        INT_FREE_FALL               =       0x04,                               /*!<  Free fall.                                                        */
        INT_WATERMARK               =       0x02,                               /*!<  Watermark.                                                        */
        INT_OVERRUN                 =       0x01                                /*!<  Overrun.                                                          */
    } AXDL345_int_map_t;

    typedef enum {
        INT1_PIN                    =       0x01,                               /*!<  PIN 1                                                             */
        INT2_PIN                    =       0x02                                /*!<  PIN 2                                                             */
    } AXDL345_int_pin_t;


    /**
      * @brief   DATA_FORMAT
      */
    /* Bit 7 : SELF_TEST ( D7 ). */
    typedef enum {
        DATA_FORMAT_SELF_TEST_Enabled       =       true,                       /*!<  SELF_TEST Enabled.                                                */
        DATA_FORMAT_SELF_TEST_Disabled      =       false                       /*!<  SELF_TEST Disabled.                                               */
    } ADXL345_data_format_self_test_t;


    /* Bit 6 : SPI ( D6 ). */
    typedef enum {
        DATA_FORMAT_SPI_Enabled             =       true,                       /*!<  3-wire SPI Enabled.                                               */
        DATA_FORMAT_SPI_Disabled            =       false                       /*!<  4-wire SPI Enabled.                                               */
    } ADXL345_data_format_spi_t;


    /* Bit 5 : INT_INVERT ( D5 ). */
    typedef enum {
        DATA_FORMAT_INT_INVERT_Enabled             =       true,                /*!<  INT_INVERT Enabled.                                                */
        DATA_FORMAT_INT_INVERT_Disabled            =       false                /*!<  INT_INVERT Disabled.                                               */
    } ADXL345_data_format_int_invert_t;


    /* Bit 3 : FULL_RES ( D3 ). */
    typedef enum {
        DATA_FORMAT_FULL_RES_Enabled             =       true,                  /*!<  FULL_RES Enabled.                                                */
        DATA_FORMAT_FULL_RES_Disabled            =       false                  /*!<  FULL_RES Disabled.                                               */
    } ADXL345_data_format_full_res_t;


    /* Bit 2 : JUSTIFY ( D2 ). */
    typedef enum {
        DATA_FORMAT_JUSTIFY_Enabled             =       true,                   /*!<  JUSTIFY Enabled.                                                */
        DATA_FORMAT_JUSTIFY_Disabled            =       false                   /*!<  JUSTIFY Disabled.                                               */
    } ADXL345_data_format_justify_t;


    /* Bit 1 - 0 : RANGE ( D1 - D0 ). */
    typedef enum {
        DATA_FORMAT_RANGE_2_G       =       0x00,                               /*!<  RANGE ±2g.                                                      */
        DATA_FORMAT_RANGE_4_G       =       0x01,                               /*!<  RANGE ±4g.                                                      */
        DATA_FORMAT_RANGE_8_G       =       0x02,                               /*!<  RANGE ±8g.                                                      */
        DATA_FORMAT_RANGE_16_G      =       0x03                                /*!<  RANGE ±16g.                                                     */
    } ADXL345_data_format_range_t;



    /**
      * @brief   FIFO_CTL
      */
    /* Bit 7 - 6 : FIFO_MODE ( D7 - D6 ). */
#define FIFO_CTL_FIFO_MODE_Pos                    0x07        /*!<  Position of FIFO_MODE field.                                      */
#define FIFO_CTL_FIFO_MODE_Mask                   0xC0        /*!<  Bit mask of FIFO_MODE field.                                      */
#define FIFO_CTL_FIFO_MODE_BYPASS                 0x00        /*!<  FIFO_MODE BYPASS mode.                                            */
#define FIFO_CTL_FIFO_MODE_FIFO                   0x01        /*!<  FIFO_MODE FIFO mode.                                              */
#define FIFO_CTL_FIFO_MODE_STREAM                 0x02        /*!<  FIFO_MODE STREAM mode.                                            */
#define FIFO_CTL_FIFO_MODE_TRIGGER                0x03        /*!<  FIFO_MODE TRIGGER mode.                                           */

    /* Bit 5 : TRIGGER ( D5 ). */
#define TRIGGER_FIFO_MODE_Pos                     0x05        /*!<  Position of TRIGGER field.                                        */
#define TRIGGER_FIFO_MODE_Mask                    0x20        /*!<  Bit mask of TRIGGER field.                                        */
#define TRIGGER_FIFO_MODE_Enabled                 0x01        /*!<  TRIGGER Enabled.                                                  */
#define TRIGGER_FIFO_MODE_Disabled                0x00        /*!<  TRIGGER Disabled.                                                 */


    /**
      * @brief   FIFO_STATUS
      */
    /* Bit 7 : FIFO_TRIG ( D7 ). */
#define FIFO_STATUS_FIFO_TRIG_Pos                  0x07        /*!<  Position of FIFO_TRIG field.                                     */
#define FIFO_STATUS_FIFO_TRIG_Mask                 0x80        /*!<  Bit mask of FIFO_TRIG field.                                     */
#define FIFO_STATUS_FIFO_TRIG_Enabled              0x01        /*!<  FIFO_TRIG Enabled.                                               */
#define FIFO_STATUS_FIFO_TRIG_Disabled             0x00        /*!<  FIFO_TRIG Disabled.                                              */




#ifndef VECTOR_STRUCT_H
#define VECTOR_STRUCT_H
    typedef struct {
        int16_t XAxis;                                                          /*!<  X-axis raw data                                                 */
        int16_t YAxis;                                                          /*!<  Y-axis raw data                                                 */
        int16_t ZAxis;                                                          /*!<  Z-axis raw data                                                 */
    } Vector_t;                                                                 /*!<  Struct to get the raw data                                      */


    typedef struct {
        float Xmg_Axis;                                                         /*!<  X-axis scaled data                                              */
        float Ymg_Axis;                                                         /*!<  X-axis scaled data                                              */
        float Zmg_Axis;                                                         /*!<  X-axis scaled data                                              */
    } Vector_f;                                                                 /*!<  Struct to get the scaled data                                   */
#endif




    /**
    * @brief   INTERNAL CONSTANTS
    */
    typedef enum {
        ADXL345_SUCCESS     =       0,                                          /*!<   The communication was fine     */
        ADXL345_FAILURE     =       1,                                          /*!<   Something went wrong           */
        I2C_SUCCESS         =       0                                           /*!<   I2C communication was fine     */
    } ADXL345_status_t;


    /** Create an ADXL345 object connected to the specified I2C pins.
     *
     * @param sda     I2C data pin
     * @param scl     I2C clock pin
     * @param addr    I2C slave address
     * @param freq    I2C frequency in Hz.
     */
    ADXL345 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete ADXL345 object.
     */
    ~ADXL345();


    /** It configures the device.
     */
    ADXL345_status_t  ADXL345_Init                          ( AXDL345_bw_rate_low_power_t LOWPOWER, AXDL345_bw_rate_rate_t RATE,
            ADXL345_data_format_int_invert_t INT_INVERT, ADXL345_data_format_full_res_t FULLRESOLUTION, ADXL345_data_format_justify_t JUSTIFY,
            ADXL345_data_format_range_t RANGE );

    /** It sets the device in Standby mode or Measure mode.
     */
    ADXL345_status_t  ADXL345_PowerMode                     ( AXDL345_power_ctl_measure_t MEASUREMODE );

    /** It gets the device ID
     */
    ADXL345_status_t  ADXL345_GetID                         ( char* myID );

    /** It gets the range.
     */
    ADXL345_status_t  ADXL345_GetRange                      ( ADXL345_data_format_range_t* myRANGE );

    /** It sets the range.
     */
    ADXL345_status_t  ADXL345_SetRange                      ( ADXL345_data_format_range_t  myRANGE );

    /** It gets the rate.
     */
    ADXL345_status_t  ADXL345_GetRate                       ( AXDL345_bw_rate_rate_t* myRATE );

    /** It sets the rate.
     */
    ADXL345_status_t  ADXL345_SetRate                       ( AXDL345_bw_rate_rate_t  myRATE );

    /** It reads the raw data.
     */
    ADXL345_status_t  ADXL345_ReadRawData                   ( Vector_t* myXYZVector );

    /** It reads the scaled data.
     */
    ADXL345_status_t  ADXL345_ReadScaledData                ( Vector_f* myScaled_XYZVector );

    /** It sets the free fall threshold.
     */
    ADXL345_status_t  ADXL345_SetFreeFallThreshold          ( char myThreshold );

    /** It sets the free fall duration.
     */
    ADXL345_status_t  ADXL345_SetFreeFallDuration           ( char myDuration );

    /** It sets the tap threshold.
     */
    ADXL345_status_t  ADXL345_SetTapThreshold               ( char myThreshold );

    /** It sets the tap duration.
     */
    ADXL345_status_t  ADXL345_SetTapDuration                ( char myDuration );

    /** It sets the douvble tap latency.
     */
    ADXL345_status_t  ADXL345_SetDoubleTapLatency           ( char myLatency );

    /** It sets the double tap window.
     */
    ADXL345_status_t  ADXL345_SetDoubleTapWindow            ( char myWindow );

    /** It sets the tap axis.
     */
    ADXL345_status_t  ADXL345_SetTap_Axis                   ( ADXL345_tap_axes_axis_t myAxisEnabled );

    /** It sets the suppress for double tap.
     */
    ADXL345_status_t  ADXL345_SetDouble_Suppress            ( ADXL345_tap_axes_suppress_t mySuppressEnabled );

    /** It sets the activity threshold.
     */
    ADXL345_status_t  ADXL345_SetActivityThreshold          ( char myActivityThreshold );

    /** It sets the activity AC or DC coupled.
     */
    ADXL345_status_t  ADXL345_SetActivity_AC_DC_Coupled     ( ADXL345_act_inact_ctl_activity_dc_coupled_t myDC_Coupled );

    /** It sets the activity axis.
     */
    ADXL345_status_t  ADXL345_SetActivity_Axis              ( ADXL345_act_inact_ctl_activity_axis_t myAxisEnabled );

    /** It reads the ACT_TAP status register.
     */
    ADXL345_status_t  ADXL345_Read_ACT_TAP_STATUS           ( char* myACT_TAP_STATUS );

    /** It sets the inactivity threshold.
     */
    ADXL345_status_t  ADXL345_SetInactivityThreshold        ( char myInactivityThreshold );

    /** It sets the inactivity time.
     */
    ADXL345_status_t  ADXL345_SetTimeInactivity             ( char myTimeInactivity );

    /** It sets the inactivity AC or DC coupled.
     */
    ADXL345_status_t  ADXL345_SetInactivity_AC_DC_Coupled   ( ADXL345_act_inact_ctl_inactivity_dc_coupled_t myDC_Coupled );

    /** It sets the inactivity axis.
     */
    ADXL345_status_t  ADXL345_SetInactivity_Axis            ( ADXL345_act_inact_ctl_inactivity_axis_t myAxisEnabled );

    /** It sets the interrupt map.
     */
    ADXL345_status_t  ADXL345_SetInterruptsMap              ( AXDL345_int_map_t myInterruptsMap, AXDL345_int_pin_t myInterruptPin );

    /** It enables the interrupts.
     */
    ADXL345_status_t  ADXL345_EnableInterrupts              ( AXDL345_int_map_t myInterruptsEnabled );

    /** It reads the interrupt sources.
     */
    ADXL345_status_t  ADXL345_ReadInterruptSource           ( char* myIntSource );


private:
    I2C      i2c;
    uint32_t ADXL345_Addr;
};

#endif