
#include "mbed.h"
#include <stdio.h>
#include <errno.h>

// Block devices
#if COMPONENT_SPIF
#include "SPIFBlockDevice.h"
#endif

#if COMPONENT_DATAFLASH
#include "DataFlashBlockDevice.h"
#endif 

#if COMPONENT_SD
#include "SDBlockDevice.h"
#endif 

#include "HeapBlockDevice.h"

// File systems
#include "LittleFileSystem.h"
#include "FATFileSystem.h"

#include "stm32l475e_iot01_tsensor.h"

// Physical block device, can be any device that supports the BlockDevice API
/*SPIFBlockDevice bd(
        MBED_CONF_SPIF_DRIVER_SPI_MOSI,
        MBED_CONF_SPIF_DRIVER_SPI_MISO,
        MBED_CONF_SPIF_DRIVER_SPI_CLK,
        MBED_CONF_SPIF_DRIVER_SPI_CS);*/

#define BLOCK_SIZE 512
HeapBlockDevice bd(4096, BLOCK_SIZE);

// File system declaration
LittleFileSystem fs("fs");

// Set up the button to trigger an erase
InterruptIn irq(BUTTON1);
void erase() {
    printf("Initializing the block device... ");
    fflush(stdout);
    int err = bd.init();
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }

    printf("Erasing the block device... ");
    fflush(stdout);
    err = bd.erase(0, bd.size());
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }

    printf("Deinitializing the block device... ");
    fflush(stdout);
    err = bd.deinit();
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
}


// Entry point for the example

Ticker toggle_write_to_file;
EventQueue queue(16 * EVENTS_EVENT_SIZE);
Thread t;
static FILE *f;
InterruptIn button(USER_BUTTON);

void writeTemToFile(){
    float temp = BSP_TSENSOR_ReadTemp();
    printf("\nTEMPERATURE = %.2f degC\n", temp);
    
    fprintf(f, "%f\n", temp);
    fflush(stdout); 
    fflush(f); 
}

void readTempFile(){
    fflush(stdout);
    fflush(f); 
    fseek(f, 0, SEEK_SET);
    printf("Temperature time line: \n");
    while (!feof(f)) {
        float temp;
        fscanf(f, "%f", &temp);
        printf("%f degC ", temp);
    }
    printf("\n");
    fflush(stdout);
}

void readSensorInterupt() {
    queue.call(writeTemToFile);
}

void printSensorInterupt() {
    queue.call(readTempFile);    
}

int main() {
    // Setup the erase event on button press, use the event queue
    // to avoid running in interrupt context
    irq.fall(mbed_event_queue()->event(erase));

    // Try to mount the filesystem
    printf("Mounting the filesystem... ");
    fflush(stdout);
    int err = fs.mount(&bd);
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        // Reformat if we can't mount the filesystem
        // this should only happen on the first boot
        printf("No filesystem found, formatting... ");
        fflush(stdout);
        err = fs.reformat(&bd);
        printf("%s\n", (err ? "Fail :(" : "OK"));
        if (err) {
            error("error: %s (%d)\n", strerror(-err), err);
        }
    }

    // Open the numbers file
    printf("Opening \"/fs/numbers.txt\"... ");
    fflush(stdout);
    f = fopen("/fs/numbers.txt", "r +");
    printf("%s\n", (!f ? "Fail :(" : "OK"));
    if (!f) {
        // Create the numbers file if it doesn't exist
        printf("No file found, creating a new file... ");
        fflush(stdout);
        f = fopen("/fs/numbers.txt", "w+");
        printf("%s\n", (!f ? "Fail :(" : "OK"));
        if (!f) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
        
        printf("Seeking file... ");
        fflush(stdout);
        err = fseek(f, 0, SEEK_SET);
        printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
        if (err < 0) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    }
    
    t.start(callback(&queue, &EventQueue::dispatch_forever));
    printf("Start sampling... \n");
    BSP_TSENSOR_Init();
    toggle_write_to_file.attach(&readSensorInterupt, 5);
    button.rise(&printSensorInterupt);
}