/**
 * Library for DS2408 on top of OneWire library
*/

#ifndef XDOT_DS2408_H
#define XDOT_DS2408_H

#include "../config.h"
#include "OneWire.h"

const uint8_t DS2408_NRETRIES = 5;

/**
 *  @class DS2408
 *  @brief This class abstracts communicating with the DS2408 port expander.
 *  The OneWire library is used for physical layer communication and this class
 *  simply makes it easy to communicate with this one chip.
 */
class DS2408
{
public:
    enum DS2408RegAddr {
        pioLogicStateReg         = 0x88,
        pioOutputLatchStateReg   = 0x89,
        pioActivityStateReg      = 0x8A,
        condSearchChSelMaskReg   = 0x8B,
        condSearchChPolMaskReg   = 0x8C,
        ctrlStatusReg            = 0x8D
    };

    /**
    * @brief DS2408 constructor
    *
    * @details Just initialized internal variables and disables test mode
    *
    * On Entry:
    *     @param[in] owMaster - reference to 1-wire master
    *     @param[in] romAddr - 64-bit address of ROM
    *
    * On Exit:
    *
    * @return
    */
    DS2408(OneWire *owMaster, uint8_t romAddr[8]);

    /**
    * @brief registerRead()
    *
    * @details reads state of pio
    *
    *
    * On Exit:
    *     @param[in]  addr - per datasheet valid registers are from 0x88 to 0x8D.
    *     Note value will be automatically promoted to 16-bit value.
    *     @param[out] val - register value
    *
    * @return Result of operation zero for success
    */
    CmdResult registerRead(uint8_t addr, uint8_t &val);

    CmdResult registerReadReliable(uint8_t addr, uint8_t &val);

    /**
    * @brief pioLogicRead()
    *
    * @details Uses regisrterRead to get logic values
    *
    *
    * On Exit:
    *     @param[out] val - lsb represents the state of the pio
    *
    * @return Result of operation zero for success
    */
    CmdResult pioLogicRead(uint8_t &val);

    /**
    * @brief pioLogicReliableRead()
    *
    * @details Uses regisrterRead to get logic values.  Then reads again to check for bit errors.
    *
    *
    * On Exit:
    *     @param[out] val - lsb represents the state of the pio
    *
    * @return Result of operation zero for success
    */
    CmdResult pioLogicReliableRead(uint8_t &val);
    // TODO implement other register based functions

    /**
    * @brief pioLogicReliableWrite()
    *
    * @details writes to pio.  Note 0 means active pull down and 1 high-Z to
    * allow PIO to float high.  This will automatically retry for DS2408_NRETRIES
    * times.
    *
    * On Entry:
    *    @param[in] val - Value for IO.
    *
    * @return CmdResult - result of operation
    */
    CmdResult pioLogicReliableWrite(uint8_t val);

    /**
    * @brief pioLogicWrite()
    *
    * @details writes to pio.  Note 0 means active pull down and 1 high-Z to
    * allow PIO to float high.
    *
    * On Entry:
    *    @param[in] val - Value for IO.
    *
    * @return CmdResult - result of operation
    */
    CmdResult pioLogicWrite(uint8_t val);

private:
    OneWire *mMaster;
    uint8_t mRomAddr[8];

    /**
    * @brief disableTestMode()
    *
    * @details Per datasheet if slew power on has a specific slew rate the
    * chip may enter a test mode.  See page 38 of application information.
    *
    * @return CmdResult - result of operation
    */
    CmdResult disableTestMode();

};

#endif
