#include "mbed.h"
#include <string>
#include "../../config.h"
#include "BaseboardIO.h"
//#include "MTSLog.h"
#include "dot_util.h"
#include "MyLog.h"

#ifdef __TEST_BBIO__
mDot* dot = NULL; // Used by dot-utils
Serial pc(USBTX, USBRX); // Externally defined

const int VERSION = 7;

char* bool2Str(bool in) {
    if (in) {
        return "Asserted\    "; // Extra space for alignment
    }
    else {
        return "Not Asserted";
    }
}

volatile uint8_t ccInIntCnt;
volatile uint8_t tamperIntCnt;
volatile uint8_t pairBtnIntCnt;
volatile uint8_t pvdIntCnt;

void ccInIntCallback () {
    ccInIntCnt++;
}
void tamperIntCallback () {
    tamperIntCnt++;
}
void pairBtnIntCallback () {
    pairBtnIntCnt++;
}

class MenuManager
{
private:
    uint8_t mCurrSel; // Current selection
    BaseboardIO *mBbio;
    bool validInput(uint8_t in) {
        if (in > 0 && in <= 8)
            return true;
        else {
            return false;
        }
    }
public:
    MenuManager() {
        mCurrSel = 0;
        mBbio = NULL;
    }
    void regBBIO (BaseboardIO *bbio) {
        mBbio = bbio;
    }
    uint8_t getCurrentSel() {
        return mCurrSel;
    }
    void applyInput(uint8_t in) {
        if (validInput(in)) {
//            mCurrSel = in;
            if (in == 1) {
                mBbio->ledOn();
            }
            else if (in == 2) {
                mBbio->ledOff();
            }
            else if (in == 3) {
                mBbio->relayNormal();
            }
            else if (in == 4) {
                mBbio->relayAlert();
            }
            else if (in == 5) {
                mBbio->serialRx(true);
            }
            else if (in == 6) {
                mBbio->serialRx(false);
            }
            else if (in == 7) {
                mBbio->serialTx(true);
            }
            else if (in == 8) {
                mBbio->serialTx(false);
            }
        }
    }
    void printMenu() {
        mBbio->sampleUserSwitches();
        pc.printf("\r\n\r\n");
        pc.printf("===============================================\r\n");
        pc.printf("= Baseboard I/O Tester v%02d                   =\r\n", VERSION);
        pc.printf("===============================================\r\n");
        pc.printf("===============================================\r\n");
        pc.printf("= Selection Options                           =\r\n");
        pc.printf("===============================================\r\n");
        pc.printf("= 0: Just refresh                             =\r\n");
        pc.printf("= 1: Turn on LED                              =\r\n");
        pc.printf("= 2: Turn off LED                             =\r\n");
        pc.printf("= 3: Toggle Relay Normal                      =\r\n");
        pc.printf("= 4: Toggle Relay Alert                       =\r\n");
        pc.printf("= 5: Turn on 232 RX                           =\r\n");
        pc.printf("= 6: Turn off 232 RX                          =\r\n");
        pc.printf("= 7: Turn on 232 TX (Note RX on as well)      =\r\n");
        pc.printf("= 8: Turn off 232 TX                          =\r\n");
        pc.printf("===============================================\r\n");
        pc.printf("= Status and Counters                         =\r\n");
        pc.printf("===============================================\r\n");
        pc.printf("= Pair btn. State: %s  IntCnt: %02d   =\r\n",
                bool2Str(mBbio->isPairBtn()), pairBtnIntCnt);
        pc.printf("= Tamper. State: N/A             IntCnt: %04d =\r\n",
                tamperIntCnt);
        pc.printf("= CCIN. State: %s      IntCnt: %02d   =\r\n",
                bool2Str(mBbio->isCCInAlert()), ccInIntCnt);
        pc.printf("= PVD (BROKEN). State: N/A       IntCnt: %02d   =\r\n",
                pvdIntCnt);
        pc.printf("= CC Normally Closed. State: %s     =\r\n", bool2Str(mBbio->isCCNC()));
        pc.printf("= Is TX. State: %s                  =\r\n", bool2Str(mBbio->isTx()));
        pc.printf("= Is LoraWAN. State: %s             =\r\n", bool2Str(mBbio->isLoRaWANMode()));
        pc.printf("= Is Serial En. State: %s           =\r\n", bool2Str(mBbio->isSerialEnabled()));
        pc.printf("= Rotary Switch 1.  Value: %02d                 =\r\n", mBbio->rotarySwitch1());
        pc.printf("= Rotary Switch 2.  Value: %02d                 =\r\n", mBbio->rotarySwitch2());
        pc.printf("===============================================\r\n");
    }
};

char WAIT_CHARS [] = {'-', '\\', '|', '/'};
/**
 * Checks that in idle state all the IOs are pulled up.
 */
int main ()
{
    MenuManager menuMgr;
    CmdResult result;
    ccInIntCnt = 0;
    tamperIntCnt = 0;
    pairBtnIntCnt = 0;
    pvdIntCnt = 0;

    pc.baud(115200);
    mts::MTSLog::setLogLevel(mts::MTSLog::TRACE_LEVEL);
    MyLog::setLogLevel(MyLog::DEBUG_LEVEL);

    wait(1.0);

    pc.printf("===============================================\r\n");
    pc.printf("= Baseboard Constructor Starting              =\r\n");
    BaseboardIO bbio;
    pc.printf("= Baseboard Constructor Finished              =\r\n");

    pc.printf("= Baseboard Init Starting                     =\r\n");
    result = bbio.init(true); // Overwrite current NVM
//    result = bbio.init(false);
    if (result == cmdSuccess) {
        pc.printf("= Baseboard Init Finished Successfully    =\r\n");
    }
    else {
        pc.printf("= Baseboard Init Finished with Error      =\r\n");
    }

    Callback<void()> ccInIntObj (&ccInIntCallback);
    Callback<void()> tamperIntObj (&tamperIntCallback);
    Callback<void()> pairBtnIntObj (&pairBtnIntCallback);
    bbio.regCCInInt(ccInIntObj);
    bbio.regTamperInt(tamperIntObj);
    bbio.regPairBtnInt(pairBtnIntObj);

    menuMgr.regBBIO(&bbio);
    menuMgr.printMenu();

    uint8_t waitCharIdx = 0;
    while (true) {

        if (pc.readable()) {
            char menuInput = pc.getc();
            menuInput -= '0'; // Convert to raw interger value
            menuMgr.applyInput(menuInput);
            menuMgr.printMenu();
        }
        else {
            pc.printf("\b%c", WAIT_CHARS[waitCharIdx]);
            if (waitCharIdx >= sizeof(WAIT_CHARS)) {
                waitCharIdx = 0;
            }
            else {
                waitCharIdx++;
            }
        }
        wait(0.1);
    }
    return 0;
}

#endif
