#include "mbed.h"
#include <string>
#include "../../config.h"

#ifdef __TEST_LRR__
DigitalOut gpio0(GPIO0);
DigitalOut gpio1(GPIO1);
DigitalOut gpio2(GPIO2);
DigitalOut gpio3(GPIO3);
DigitalOut wake_DOUT(WAKE);
DigitalOut i2cOut1(I2C1_SCL);
DigitalOut i2cOut2(I2C1_SDA);

DigitalOut uartCts(UART1_CTS);
DigitalOut uartRts(UART1_RTS);
DigitalOut uartTx(UART1_TX);
DigitalOut uartRx(UART1_RX);

DigitalOut mosi(SPI_MOSI);
DigitalOut sck(SPI_SCK);

DigitalOut nss(SPI_NSS, 1); // if low then miso on the flash becomes output
DigitalInOut miso(SPI_MISO, PIN_INPUT, PullNone, 0);

Serial pc(USBTX, USBRX); // Externally defined

const static std::string PIN_NAMES [] = {
       "Pin  2: UART_TX ", // Idx  0
       "Pin  3: UART_RX ", // Idx  1
       "Pin  4: MISO    ", // Idx  2
       "Pin  6: SCL     ", // Idx  3
       "Pin  7: SDA     ", // Idx  4
       "Pin 11: MOSI    ", // Idx  5
       "Pin 12: UART_CTS", // Idx  6
       "Pin 13: WAKE    ", // Idx  7
       "Pin 15: GPIO2   ", // Idx  8
       "Pin 16: UART_RTS", // Idx  9
       "Pin 17: NSS     ", // Idx 10
       "Pin 18: SCK     ", // Idx 11
       "Pin 19: GPIO1   ", // Idx 12
       "Pin 20: GPIO3   "  // Idx 13
};

const uint8_t config0PinIdx[] = {0,1,  3,4,5,6,7,8,9,10,11,12,13};
const uint8_t config1PinIdx[] = {0,1,2,3,4,5,6,7,8,9,   11,12,13};

class MenuManager
{
private:
    uint8_t mCurrSel; // Current selection
    bool validInput(uint8_t in) {
        return in <= 1; // Either 0, 1
    }
public:
    MenuManager() {
        mCurrSel = 0;
    }
    uint8_t getCurrentSel() {
        return mCurrSel;
    }
    void applyInput(uint8_t in) {
        if (validInput(in)) {
            mCurrSel = in;
        }
    }
    void printMenu() {
        pc.printf("===============================================\r\n");
        pc.printf("= LRR I/O Tester                              =\r\n");
        pc.printf("===============================================\r\n");
        pc.printf("= Option 0: MISO Disabled, NSS Enabled        =\r\n");
        pc.printf("= Option 1: MISO Enabled,  NSS Disabled       = \r\n");
        pc.printf("= Current Selection is %d                      =\r\n", mCurrSel);
        pc.printf("===============================================\r\n");
        pc.printf("= Details:                                    =\r\n");
        if (mCurrSel == 0) {
            pc.printf("= The following pins are toggling:            =\r\n");
            pc.printf("= Pin #: Name                                 =\r\n");
            for (unsigned int i=0; i < sizeof(config0PinIdx); i++) {
                pc.printf("= %s                            =\r\n", PIN_NAMES[i].c_str());
            }
        }
        else if (mCurrSel == 1) {
            pc.printf("= The following pins are toggling:            =\r\n");
            pc.printf("= Pin #: Name                                 =\r\n");
            for (unsigned int i=0; i < sizeof(config1PinIdx); i++) {
                pc.printf("= %s                            =\r\n", PIN_NAMES[i].c_str());
            }
        }
        pc.printf("===============================================\r\n");
    }
};

/**
 * Checks that in idle state all the IOs are pulled up.
 */
int main ()
{
    pc.baud(115200);

    MenuManager menuMgr;
    menuMgr.printMenu();

    while (true) {
        gpio0 = !gpio0;
        gpio1 = !gpio1;
        gpio2 = !gpio2;
        gpio3 = !gpio3;
        wake_DOUT = !wake_DOUT;
        i2cOut1 = !i2cOut1;
        i2cOut2 = !i2cOut2;

        uartCts = !uartCts;
        uartRts = !uartRts;
        uartTx = !uartTx;
        uartRx = !uartRx;

        mosi = !mosi;
        sck = !sck;

        if (menuMgr.getCurrentSel() == 0) {
            miso.input();
            nss = !nss;
        }
        else if (menuMgr.getCurrentSel() == 1) {
            nss = 1; // Disable flash
            miso.output();
            miso = !miso;
        }

        if (pc.readable()) {
            char menuInput = pc.getc();
            menuInput -= '0'; // Convert to raw interger value
            menuMgr.applyInput(menuInput);
            menuMgr.printMenu();
        }
        else {
            pc.printf("*");
        }
        wait(1.0);
    }
    return 0;
}
#endif
