/*
 * testAPI.cpp
 *
 *  Created on: Feb 3, 2017
 *      Author: mbriggs
 */
#include "mbed.h"
#include "greentea-client/test_env.h"
#include "unity.h"
#include "utest.h"
#include "rtos.h"

// API under test
#include "DS2408.h"

#define __TEST__ 1

#define USE_RELIABLE 1

using namespace utest::v1;

DigitalOut switchedIO(I2C_SCL, 0); // Always enable Switched IO for test
OneWire OWMaster(I2C_SDA);
uint8_t ds2408Addr[8];

/**
 * A test that the OneWire Device is on the bus.  If not all other tests will fail.
 */
void test_search() {
    uint8_t result;
    for (int i=0; i<10; i++) {
        OWMaster.reset();
        OWMaster.reset_search(); // TODO Check if this is helpful
        wait(1.0);
        result = OWMaster.search(ds2408Addr);
        if (result == 1) {
            break;
        }
    }
    TEST_ASSERT(result == 1);
}

/**
 * Checks that in idle state all the IOs are pulled up.
 */
void test_idle_state() {
    DS2408 ds2408(&OWMaster, ds2408Addr);
    CmdResult result;

    #ifdef USE_RELIABLE
    result = ds2408.pioLogicReliableWrite(0xFF); // Leave all IOs high Z e.g. should float high
    #else
    result = ds2408.pioLogicWrite(0xFF); // Leave all IOs high Z e.g. should float high
    #endif
    TEST_ASSERT_MESSAGE(result == cmdSuccess, "Error returned during write operation");

    uint8_t val = 0;
    #ifdef USE_RELIABLE
    result = ds2408.pioLogicReliableRead(val);
    #else
    result = ds2408.pioLogicRead(val);
    #endif
    TEST_ASSERT_MESSAGE(result == cmdSuccess, "Error returned during read operation");
    TEST_ASSERT_MESSAGE(val == 0xFF, "Bits not correct");
}

/**
 * This test walks a zero bit for ds2408.  Note this requires a ds2408 with all
 * IOs pulled up since each will pulled low in sequence.
 */
void test_walk_the_zero() {
    DS2408 ds2408(&OWMaster, ds2408Addr);

    #ifdef USE_RELIABLE
    result = ds2408.pioLogicReliableWrite(0xFF); // Leave all IOs high Z e.g. should float high
    #else
    result = ds2408.pioLogicWrite(0xFF); // Leave all IOs high Z e.g. should float high
    #endif
    TEST_ASSERT_MESSAGE(result == cmdSuccess, "Error returned during write operation");

    uint8_t readVal = 0;
    #ifdef USE_RELIABLE
    result = ds2408.pioLogicReliableRead(readVal);
    #else
    result = ds2408.pioLogicRead(readVal);
    #endif
    TEST_ASSERT_MESSAGE(result == cmdSuccess, "Error returned during read operation");
    TEST_ASSERT_MESSAGE(readVal == 0xFF, "Bits not correct");

    uint8_t writeVal;
    for (int i=0; i<8; i++) {
        writeVal = ~(1 << i);
        #ifdef USE_RELIABLE
        result = ds2408.pioLogicReliableWrite(writeVal); // Leave all IOs high Z e.g. should float high
        #else
        result = ds2408.pioLogicWrite(writeVal); // Leave all IOs high Z e.g. should float high
        #endif
        TEST_ASSERT_MESSAGE(result == cmdSuccess, "Error returned during write operation");

        #ifdef USE_RELIABLE
        result = ds2408.pioLogicReliableRead(readVal);
        #else
        result = ds2408.pioLogicRead(readVal);
        #endif
        TEST_ASSERT_MESSAGE(result == cmdSuccess, "Error returned during read operation");
        TEST_ASSERT_MESSAGE(readVal == writeVal, "Bits not correct");
    }
}


utest::v1::status_t test_setup(const size_t number_of_cases) {
    // Setup Greentea using a reasonable timeout in seconds
    GREENTEA_SETUP(40, "default_auto");
    return verbose_test_setup_handler(number_of_cases);
}

// Test cases
Case cases[] = {
    Case("Testing search", test_search),
//    Case("Testing init", test_init),
    Case("Testing idle state", test_idle_state),
    Case("Test walking the zero", test_walk_the_zero)
};

Specification specification(test_setup, cases);

// Entry point into the tests
int main() {
    return !Harness::run(specification);
}
