/* 
   Name: "classLaserSensingTrajectory.h"
   
   Description: This is a collection of LaserPoint(s), with methods to analyse the trajectory. 
                Each object of BaseObject class contain one instance of this class; parameters to analyse the saccade (ex: threshold, etc), can be set in a per-object basis. 
                Derived classes of BaseObject may have special methods to further analyse the "segmented" saccade trajectory (to compute recentering vectors, etc). 
                Finally, this object is heavily accessed by the GLOBAL INSTANCE of laserSensingDisplay (lsd), which is the "displaying engine" running in an interrupt. 
*/

#ifndef LSDTRAJECTORY_H
#define LSDTRAJECTORY_H

#include <vector>
#include "matrixClass.h"

// Thresholding mode: 
enum thresholdingMode {FIXED, AUTO};
//(1) fixed threshold (in case of FIXED threshold mode)
#define FIXED_THRESHOLD 35
//(2) Autothreshold (in case of AUTO threshold mode):
// CONTRAST RATIO to compute autoThreshold:
// MIN_CONTRAST_RATIO is the minimum contrast between max and min intensity necessary to "accept" a black and white zone:
#define MIN_CONTRAST_RATIO 1.8//1.7 // 3 seems good when lookup table does not work 
// THRESHOLD_FACTOR is where the threshold is actually placed between the min and max detected (with good contrast):
#define THRESHOLD_FACTOR 0.5 //0.75 // 2/3 or 1/2 are good values 
#define MIN_ACCEPTABLE_INTENSITY 16 // if maxI< this then we consider all the saccade on something black

// ================= The minimal structure for a "projected/sensed" laser point =================
typedef struct LaserPoint {
    //public: 
   LaserPoint(float x, float y) //: v2={x,y} //filling structs "by list"... we could have custom constructors for them though (but what would be the advantage?)
   {v2.x=x; v2.y=y;};
   LaserPoint(V2 _v2) : v2(_v2)
   {};
   
   // Setting method only for COORDINATES (otherwise we can use the default copy constructor '=', but in case of re-projecting, we don't need to change the 
   // values of the sensed data:
   void setCoord(const V2& _v2) {v2=_v2;};
   
   // Data:
   V2 v2; // position of the point in image plane (after rendering - it's an UNSIGNED SHORT, because it is in "laser projector pixels", i.e, DA units)
   
   // Per point displaying attributes: can include color, or polarization, or modulation, but also TIME for posing on the point, etc 
   // (for the time being, the time one poses on a point can be controlled by just REPEATING the point... not nice, but well, it's a start).
   // unsigned char myColor;  // override now by the per-object color property now. 
   
    // Sensed intensity for this point (in the future, color?)
    unsigned char intensity; // detected intensity (in fact, this will be the REFLECTIVITY RATIO if using LUT table, and it's between 0 and 1, but we will multiply by 255 to avoid using a float. 
    signed char lightZone; // the thresholded light zone (allow for negative values?)
} LaserPoint;

class LaserSensingTrajectory  {

public:

    LaserSensingTrajectory();
    ~LaserSensingTrajectory();

    int size() {return(lsdTrajectory.size());} 

    // METHODS:
    bool processSensedData(); // Basic analysis of the saccade (classification into two zones). It return the value of lightTouched for commodity.

    void setDelayMirrors(int); // in general, the delay will depend on the number of points being DISPLAYED (in other terms, on the size of lsdTrajectory).
    void addDelayMirrors(int add_delay);

// PARAMETER ADJUSTEMENT FOR SENSING METHODS (note: perhaps better in a separated class?)
    // NOTE: parameters for sensing the data may depend on the number of points, etc. Therefore, I provide setting methods and instance variables 
    // for each instance of the class
 
    void setThresholdMode(unsigned char value) {modeThreshold=(value>0? AUTO : FIXED);};
    void setFixedThreshold(unsigned char value) {fixedThreshold=value;};
    void setMinContrastRatio(float value) {min_contrast_ratio=value;};
    void setThresholdFactor(float value) {threshold_factor=value;};
    void setMinAcceptableIntensity(unsigned char value) {min_acceptable_intensity=value;};
    void multMinContrastRatio(float multfactor) {min_contrast_ratio*=multfactor;};
    void multThresholdFactor(float multfactor) {threshold_factor*=multfactor;};

 //  PARAMETERS FOR SACCADE ANALYSIS:
    // Adjustement of mirror delay:
    unsigned char delayMirrorSamples; // it could be in the laser renderer, but by putting it here we can have more per-object fine tunning
    // parameters for thresholding and thresholding mode:
    thresholdingMode modeThreshold;
    float min_contrast_ratio, threshold_factor, min_acceptable_intensity;
    unsigned char autoThreshold, fixedThreshold; // 0 to 255
    
    // TRAJECTORY DATA:
    vector <LaserPoint> lsdTrajectory;
    //unsigned char displayColor; // per BaseObject color (now in the baseObject class)
    
    // Statistics and tests:
    //float lightRatio;
    unsigned char maxI, minI;     // Max and Min intensity RATIOS (normalized between 0 and 255)
    bool lightTouched;     // true if something went over the autoThreshold for the whole loop
};

#endif



