#ifndef SCENE_H
#define SCENE_H

#include <vector> 
#include "matrixClass.h"
#include "classLaserSensingTrajectory.h" // BaseObjet use instances of the classes LaserPoint and LaserSensingTrajectory (cannot forward declare only)
//#include "LaserRenderer.h"

// Forward declared classes (attention: a pointer to an "incomplete class" - eg, only forward declared - is not allowed. It is just like unsing an actual object...)
class LaserRenderer;
class laserSensingDisplay;

// A base class for an "object":
// Note: all the 3d coordinates of the object are in PROJECTOR coordinates in fact... we don't store a POSE for each object, but instead the
// final RT*vi where vi are the points of the objects in local coordinates. 
// Also, we store IN THE OBJECT the list of projected points (that is, the 3d points projected by using K). The object has methods to study this 
// list (because projected points are also "sensed") and get information about light or dark zones.
// Now, to be able to test things (where the object was "touched", etc), we may need to mantain a correspondence between the projected points 
// and the original 3d point. There are two options: put them into a single structure "laserpoint", or use two separate arrays. 
class BaseObject {
 public: 
 
    friend class Scene;
    friend class LaserRenderer;
    friend class laserSensingDisplay;
 
    BaseObject(unsigned int _id=0):myID(_id) {}
    ~BaseObject();
    
    // Setters:
    void setID(int _id) {myID=_id;}
    void setColor( unsigned char _color) {myColor=_color;} // color is coded in three bits in fact (for RGB laser), or more or less...
    // Per-object adjustement of mirror parameters (could be automated if we know the number of points, etc. For the time being, this is manual):
    
    // Getters: 
    int size() {return(vertexArray.size());} // ATTN: in principle, the vertexArray size and the myDisplaySensingBuffer number of points should be equal!! we could check this here?
    int color() {return(myColor);}
    int ID() {return (myID);}
    Box3d getEnclosingBox();
    
     // Building methods:
     // NOTE: I will try to have an encapsulated class, so I prefer not to refer to global variables (like the renderer lsr)
    void addVertex(V3& _v3);// add a vertex in LOCAL COORDINATES
    void addVertex(V3& _v3, Mat44& _RT); // this will be usefull to add a vertex USING THE CURRENT MODELVIEW (in global lsr object) when 
    // creating objects using the "open-gl" like "begin/end" wrappers. NOTE: it is NOT projected and added to the myDisplaySensingBuffer yet.
    void clear(); // this deletes all the vertices and their projections from *this BaseObject instance
    
    // Transformation of vertices:
    void transform(Mat44& _RT); // this transform all the vertices of the object by _RT
    
    // void render(LaserRenderer* ptr_lsr); // this will project ALL the 3d points in the object vertexArray and store into display sensing buffer (myDisplaySensingBuffer), using 
    // the current projection matrix and the current projecting mode in the global lsr object. NOTE: THE PREVIOUS DISPLAY BUFFER for this object is CLEARED. This ensures that when 
    // "rendered", we have the same number of projected and 3d points 
    
     // Sensing methods (note: only way to get data - touched, not touched, etc - through getters that will re-execute the processing on new data... 
     // (we could have a boolean to check if new data is available, but since the displaying is done very fast, it is likely that we may need to re-compute
     // the processing anyway...)
    bool sense(void); 
    unsigned char maxIntensity(void);
    unsigned char minIntensity(void);
    
    // Virtual methods? for more complex testing - like overall direction, or even motion in case of "sprites"
    // ...
    // NOTE: as always, I run into this annoying problem: if I have a vector of pointers to the base class, AND virtual methods, then I can store all the
    // childrens pointers in the vector (using downcast "dynamic type casting" which won't slice the object). But then, it means the base class declaration 
    // needs to know somehting about the children methods! (at least their types and names). THIS IS REALLY ANNOYING, because I would like to add new kind
    // of children objects without needing to modify the base class declaration! How to do that? The thing is, sometimes I may need to call to a child method; 
    // if the pointer points to an object that does not have this method implemented, NOTHING should happen (pure virtual function). Otherwise, it should work
    // as specified by this child. This very much reminds me to weakly typed Objective-C polymorphic arrays... (NSArrays). How to do this in C++?
  
  unsigned char myColor; // per object color (public because it will be used intensively by the displaying engine)
  
   LaserSensingTrajectory displaySensingBuffer;
   
 private:
 
 // PRIVATE INSTANCE VARIABLES:
   int myID;
    
    // Array of 3d coordinates and corresponding 2D projections with sense data:
    // These are private, so we can ensure that their size is the same... 
    vector<V3> vertexArray;
    //LaserSensingTrajectory displaySensingBuffer; // the object displaySensingBuffer contains the trajectory of v2 projected points, as well
    // as some methods to perform the analysis of the saccade. NOTE: I could have made BaseObject a CHILD of the clas LaserSensingTrajectory. But I prefer to 
    // have a clear separation between sensing parameters "tweaking" (like mirror delays, setting methods and variables) and the more pure graphical BaseObject  

    Box3d enclosingBox; // this will be filled when rendering the object or when explicitly computing the enclosing box
    
    // FUTURE WORK: 
    // - OBJECTS should have their own transformation matrix, and methods (inherited from a base class?) to modify it or recreate it. This way we will 
    // avoid transforming the object 3d points independently, which will create problems because approximations. Eventually, these objects will have 
    // behaviours that will affect this transformation matrix (as a function of time). When rendering an object, the object original 3d points will be first modified by 
    // the object's own RT, THEN by the global modelview. This is exactly like in an openGL program...
};

// ==============================================================================================================================================


// The scene, which is just a collection of objets, plus some methods to add objects, etc:
class Scene {
public:

    friend class LaserRenderer;
    friend class laserSensingDisplay;
    
    Scene(): numTouchedObjects(0) {clear();}
    ~Scene();

    // TO THINK ABOUT: METHODS TO TRANSFORM THE COORDINATES OF THE POINTS IN THE SCENE (affecting 3d coordinates) and to RERENDER, 
    // or even TO TRANSFORM AND RENDER (but without modifying the original 3d points). 

 // Create/destroy new objects:
    // TWO WAYS to create objects: 
    // (1) Use a strategy somehow similar to OpenGL, by using a "begin/end" structure 
    // (2) Making a specific object class inheriting from Object struct (or class), and then ADD to the Scene:
     
    void addObject(BaseObject* ptr_newObject);
    void addCurrentObject(); // this adds the "current" object (pointed by ptr_currentObject)
     
    void transform(Mat44& _RT); // this transform all the objects of the scene by _RT
    
    // Rendering whole scene:
    // void render(LaserRenderer* ptr_lsr);
    
    void clear(); // "delete" all the objects in the scene (attn! if we store the objects in the scene as POINTERS, we must 'delete' the pointer created with 'new', 
    // so the memory for the object is liberated by calling its destructor, and THEN clear the vector of pointers)
    void deleteObject(int _id); // selective deletion (if we know the object index ; perhaps using a STL map with names?)
    
     int totalObjects(); // number of objects in the scene 
     int totalPoints(); // total number of points in the scene
     
     int sense();
    
//   Methods to "rearrange" the objects in the scene for rendering: 
    // TO DO ........................   

    // Pointer to the current object being built (in case we use begin/end methods for creating and adding new objects to the scene).
    // In the future, we may have methods to set this pointer to some other object for adding vertices, changing color, etc. Perhaps not necessary...
    BaseObject* ptr_currentObject; // made it public because accessed by the WRAPPER FUNCTION "vertex" (can be made a friend...)

//private:
    
    // The array of objects in this scene: 
    vector <BaseObject*> objectArray; // better to use a vector of pointer: the advantage would be, we could have a polymorphic superclass "Objects" with virtual methods...
    
     //bool touchedScene;
     int numTouchedObjects; // only modified and accessed when calling sense() method
};

#endif



