#include "mbed.h"
#include "Flipper.h"

// The following parameters can be changed by serial commands:
// NOTE: program 9 on the LCC-230 gives a freq of 150Hz. If frame rate for the camera is 30fps, then the nb of cycles per frame is 150/30=5
// for 28.09, then the of cycles per frame is 150/28.09=5.339
float CAMERA_FPS=58.75;//28.0915;//28.09;
unsigned long CAMERA_TRIGGER_PERIOD=1000000.0/CAMERA_FPS; // it is recommendable that this division gives an integer!
float EXPOSURE_TIME_FACTOR=1.0/3;
unsigned long EXPOSURE_TIME=EXPOSURE_TIME_FACTOR*CAMERA_TRIGGER_PERIOD;
unsigned long WAITING_TIME=CAMERA_TRIGGER_PERIOD-EXPOSURE_TIME;

// NOTE: since the delays (waiting time and exposure time) are in microseconds, we NEED to have an integer number of cycles per CAMERA frame...
// (no pb if there is not an integer INSIDE the exposure time, but better to have it). Hence the "approximate" value:
float approximate_NbCylclesExposureTime=6;
unsigned long NB_CYCLES_PER_CAMERA_FRAME=1.0*approximate_NbCylclesExposureTime/EXPOSURE_TIME_FACTOR;

unsigned long LED_PERIOD=1000000.0/(NB_CYCLES_PER_CAMERA_FRAME*CAMERA_FPS); //in us
// NOTE: there is no warangy this division will be integer! PLUS it must be a multiple of 2!! mmm...

unsigned int NB_FRAMES_TOGGLE_QP=1; // this is the nb of camera frames before toggling the modulation (note: there is no synchronization!)

enum SignalMode {PHASE=0, QUADRATURE, TOGGLE_PHASE_QUADRATURE};
SignalMode currentSignalMode=TOGGLE_PHASE_QUADRATURE; //NOTE: all the leds mode affected by this at the same time

//Trigger cameraTrigger(cameraTriggerPin,1.0*CAMERA_FPS,(unsigned int)EXPOSURE_TIME);
Trigger cameraTrigger(cameraTriggerPin,(unsigned int)WAITING_TIME-27,(unsigned int)EXPOSURE_TIME);
Flipper ledSource(ledPin, LED_PERIOD);
Shutter lcdShutter(shutterPin, LED_PERIOD);

Serial pc(USBTX, USBRX); // tx, rx
void processSerial();

int main()
{
    pc.baud(9600);


    cameraTrigger.setQPToggleFrames(NB_FRAMES_TOGGLE_QP);
    cameraTrigger.start();
    
    ledSource.start();
    
    // add some delay?
   // wait_us(100);
    lcdShutter.mixSignal(true);
    lcdShutter.start();

    // spin in a main loop. flipper will interrupt it to call flip
    while(1) {
        processSerial();
    }
}

// String to store ALPHANUMERIC DATA (i.e., integers, floating point numbers, unsigned ints, etc represented as DEC) sent wirelessly:
char incomingValue[5];
char incomingByte;
char indexString=0;

void processSerial()
{

    while(pc.readable()>0) {
        char incomingByte =pc.getc();

        if (incomingByte >='0' && incomingByte <='9') {
            incomingValue[indexString]=incomingByte;
            indexString++;
        } else { // check for terminator (i.e. "CONTROL CODES"):
            if (incomingByte=='P') {
                currentSignalMode=PHASE;
                cameraTrigger.toggleQP(false);
                ledSource.flip();
            } else if (incomingByte=='Q') {
                currentSignalMode=QUADRATURE;
                cameraTrigger.toggleQP(false);
                ledSource.flip();
            } else if (incomingByte=='T') {
                currentSignalMode=TOGGLE_PHASE_QUADRATURE; // periodic toogling (every a TOGGLE_QP_PERIOD)
                cameraTrigger.toggleQP(true);
            } else if (incomingByte=='s') { // simulated shutter OFF
                lcdShutter.mixSignal(false);
            } else if (incomingByte=='S') { // simulated shutter ON
                lcdShutter.mixSignal(true);
            
            // CAMERA PARAMETERS:
            } else if (incomingByte == 'F') { // camera frame rate
                incomingValue[indexString]=0; // termination for the string
                indexString=0;
                cameraTrigger.setFrameRate(atoi(incomingValue));   
            } else if (incomingByte == 'E') { // control Exposure time:
                incomingValue[indexString]=0; // termination for the string
                indexString=0;
                cameraTrigger.setExposure(atoi(incomingValue));
            } else if (incomingByte == 'L') { // get the LED PERIOD (to adjust the real LCD shutter)
               // incomingValue[indexString]=0; // termination for the string
               // indexString=0;
                pc.printf("\n\nCAMERA FPS: %f fps\n", CAMERA_FPS);
                pc.printf("CAMERA TRIGGER PERIOD: %i us\n", CAMERA_TRIGGER_PERIOD);
                pc.printf("EXPOSURE TIME: %i us\n", EXPOSURE_TIME);
                pc.printf("\nLED PERIOD: %i us\n", LED_PERIOD);
                pc.printf("LED FREQ: %f Hz\n", 1.0/LED_PERIOD*1000000);
                pc.printf("TOGGLE QP every: %i frames\n", NB_FRAMES_TOGGLE_QP);
            }
            
        }
    }
}