/*
 * Copyright (c) 2015 ARM Limited. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "EthernetInterface.h"
#include "mbed-client/m2minterfacefactory.h"
#include "mbed-client/m2mdevice.h"
#include "mbed-client/m2minterfaceobserver.h"
#include "mbed-client/m2minterface.h"
#include "mbed-client/m2mobjectinstance.h"
#include "mbed-client/m2mresource.h"
#include "security.h"
#include "ns_trace.h"

#include "mbed.h"
#include "simple_client.h"


Serial output(USBTX, USBRX);
AnalogIn currentSensor(A0);
Ticker getCurrent;


#if defined(TARGET_K64F)
#define OBS_BUTTON SW2
#define UNREG_BUTTON SW3
#endif


EthernetInterface eth;
// Instantiate the class which implements
// LWM2M Client API
MbedClient mbed_client;

// Set up Hardware interrupt button.
InterruptIn obs_button(OBS_BUTTON);
InterruptIn unreg_button(UNREG_BUTTON);

// Network interaction must be performed outside of interrupt context
Semaphore updates(0);
volatile bool registered = false;
osThreadId mainThread;

void unregister() {
    registered = false;
    updates.release();
}

void update() {
	updates.release();
}

volatile float max_value = 0;
volatile float amplitude_current = 0;
volatile float effective_value = 0;

void calcCurrent()
{
    int count = 0;
    float x = 0;
    while(count++ < 10) {
        x = currentSensor.read();
        if(x == 0 && _relay == 0) {
            max_value = 0;
        } else if (x > max_value)  {
            max_value = x;
        }
    }
    amplitude_current = max_value*3.3/800*2000000;
    effective_value=amplitude_current/1.414;
}

// Entry point to the program
int main() {
   
    // Keep track of the main thread
    mainThread = osThreadGetId();

    // Sets the console baud-rate
    output.baud(115200);
    output.printf("Its Aliiiive!!!\r\n");

    // This sets up the network interface configuration which will be used
    // by LWM2M Client API to communicate with mbed Device server.
    eth.init(); //Use DHCP
    eth.connect();

    //lwipv4_socket_init();
    output.printf("IP address %s\r\n", eth.getIPAddress());

    // On press of SW3 button on K64F board, example application
    // will call unregister API towards mbed Device Server
    //unreg_button.fall(&mbed_client,&MbedClient::test_unregister);
    unreg_button.fall(&unregister);

    // On press of SW2 button on K64F board, example application
    // will send observation towards mbed Device Server
    obs_button.fall(&update);
    
    getCurrent.attach(&calcCurrent, 0.5f);

    // Create LWM2M Client API interface to manage register and unregister
    mbed_client.create_interface();

    // Create LWM2M server object specifying mbed device server
    // information.
    M2MSecurity* register_object = mbed_client.create_register_object();

    // Create LWM2M device object specifying device resources
    // as per OMA LWM2M specification.
    M2MDevice* device_object = mbed_client.create_device_object();

    // Create Generic object specifying custom resources
    M2MObject* generic_object = mbed_client.create_generic_object();

    // Add all the objects that you would like to register
    // into the list and pass the list for register API.
    M2MObjectList object_list;
    //object_list.push_back(device_object);
    object_list.push_back(generic_object);

    mbed_client.set_register_object(register_object);

    // Register with mbed Device Connector
    mbed_client.test_register(register_object, object_list);
    registered = true;
       uint32_t count = 0;

    while (true) {
        int update = updates.wait(1000);

        if (!registered) {
            break;
        } else if (update) {
            mbed_client.update_resource();
        } else if(++count%25 == 0) {
            mbed_client.test_update_register();
            output.printf("\r\nRegistration Updated\r\n");
        }
        output.printf(".");
        mbed_client.updateCurrent(effective_value);
        output.printf("max = %f, amp=%f eff = %fmA, \r\n",max_value,amplitude_current,effective_value);
    }

    mbed_client.test_unregister();
}


