/**
    Program to implement synchronous data flow:
    This program reads a file sdgconf.txt and constructs a synchronous data flow graph (SDFG) and a topology matrix.
    It then reads input samples from input.txt and saves these values in an input buffer to enable faster processing.
    After processing all K samples X times, 
**/

/**
    Header Files and Global Variables
**/
#include "mbed.h"
#include "Rationals.h"

#include <vector>
#include <string>
#include <queue>

// mbed variables
DigitalOut led1(LED1);

DigitalOut pin21(p21);
DigitalOut pin22(p22);
DigitalOut pin23(p23);

Serial pc(USBTX, USBRX);    // for host PC interaction

LocalFileSystem local("local");

Timer t;                    // to read timer values

signed long inBuffer[4096] __attribute__ ((section("AHBSRAM0")));
signed long outBuffer[4096] __attribute__ ((section("AHBSRAM1")));

// program variables
bool NO_SCHEDULE = false;

vector <int> edge_array (32);       //stores values read from each line in input file in integer form

vector< vector<Rationals> > matrix; // topology matrix of Rational Numbers
vector<Rationals> nullspace;        // nullspace of Rational Numbers

//integer matrices
vector< vector<int> > tau;          // topology matrix
vector<int> q;                      // nullspace

vector<int> buffer;                 // b[0] -- initial buffer state
vector<int> buffer_next;            // b[n] -- to store buffer state after a node is processed

vector<char> node_types;            // to store nodes for input processing
vector<int> schedule;               // stores final feasible schedule

vector< queue<signed int> > fifos;  // FIFO queues -- one for each edge/arc

#include "sdfg.h"                   // header file for nullspace and schedule computation

//constants that can be read -- need vectors to account for multiple M nodes and multiple D,U and C nodes
vector<signed int> CD;
vector<int> NsamplesD;
vector<int> NsamplesU;
vector<int> Kvalue;
int iterM=0,iterD=0,iterU=0,iterC=0;// iterators for each of these

int X, K;                           // stores X = # of repetitions and K = # of samples
#include "inputops.h"               // header file for input processing operations

/**
    Functions
**/

//parse file - populate sdfg
int parse_conf(FILE *fpointer)
{
    int nodes, edges;    
    char temp[100];
    
    fgets(temp, 100, fpointer);
    sscanf(temp, "%d %d", &nodes, &edges);    
    
    // boundary checks
    if(nodes == 0 || edges > 32768) {
        pc.printf("INCORRECT FORMAT\r\n");
        exit(1);
    }
     
    // having fetched # of nodes and edges -- set size for all vectors
    
    matrix.resize(edges, vector <Rationals>(nodes, Rationals(0,1)));   
    nullspace.resize(nodes, Rationals(-1,1));
    
    tau.resize(edges, vector<int> (nodes));
    q.resize(nodes);
    buffer.resize(edges);
    buffer_next.resize(edges);
    node_types.resize(nodes);
    
    fifos.resize(edges);            //ONE Q per edge.
   
    pc.printf("%d %d\r\n", nodes, edges);
    
    char c;
    int ci = 0;                     //stores index of node types -- like I, A, F...
    
    int j;
    
    int check = 0;
    
    char line[100];
    while ( fgets ( line, sizeof line, fpointer ) != NULL ) /* read a line -each line is a node, use ci*/
      {          
          vector<string> v = explode(" ", line);
          edge_array.resize(v.size()-1);
          
          sscanf(v[0].c_str(),"%c", &c);
          node_types[ci]=c;
          
          for(int i=1; i< v.size(); i++) {
            sscanf(v[i].c_str(),"%d", &edge_array[i-1]);            
            //pc.printf("%d ", edge_array[i-1]);            
           }
           
           //we now have the node type and the edge list.
           // time to populate matrix 
           switch(node_types[ci])
           {
            case 'I': // I oe1 [oe2 ...]
                if(check > 0)
                   return(1);       //error state -- only ONE input node possible
                check++;
                if(edge_array.size()<1)
                    return(1);
                for (j=0; j < v.size()-1; j++) //for each outgoing edge
                    matrix[edge_array[j]] [ci] = Rationals(1,1);
                //pc.printf("%d\r\n",c);                       
               break;
            case 'O': // O ie
                //pc.printf("%d\r\n",c);
                // check that there is only one output node
                if(check>1)
                    return(1);        //error state -- only ONE output node possible                
                check++;
                if(edge_array.size() > 1) //check that it has only one input edge.
                    return(1);
                matrix[edge_array[0]] [ci] = Rationals(-1,1);             
                break;
            case 'A': //A ie1 ie2 oe1 [oe2 ..]
                // input check
                //pc.printf("%d\r\n",c);           
                if(edge_array.size() < 3) //check that it has atleast input edges and one output edge.
                    return(1);
                matrix[edge_array[0]] [ci] = Rationals(-1,1);
                matrix[edge_array[1]] [ci] = Rationals(-1,1);
                for (j=2; j < v.size()-1; j++)      // for each outgoing edge, add matrix[j][ci] = 1
                    matrix[edge_array[j]][ ci] = Rationals(1,1);
                break;
            case 'S': // S ie1 ie2 oe1 [oe2...]
                //pc.printf("%d\r\n",c);
                if(edge_array.size() < 3) //check that it has atleast input edges and one output edge.
                    return(1);
                matrix[edge_array[0]] [ci] = Rationals(-1,1);
                matrix[edge_array[1]] [ci] = Rationals(-1,1);
                for (j=2; j < v.size()-1; j++)      // for each outgoing edge, add matrix[j][ci] = 1
                    matrix[edge_array[j]][ ci] = Rationals(1,1);
                break;
            case 'M': // M c d ie oe1 [oe2..]
                //store 2 constants c and d                
                //pc.printf("%d\r\n",c);
                if(edge_array.size() < 4) //check that it has C,D and one input edge and one output edge.
                    return(1);
                CD.push_back(edge_array[0]); //C
                CD.push_back(edge_array[1]); //D
                matrix[edge_array[2]] [ci] = Rationals(-1,1);
                for (j=3; j < v.size()-1; j++) //for each outgoing edge, add matrix[j][ci] = 1
                    matrix[edge_array[j]][ ci] =Rationals(1,1);
                break;
            case 'D': // D n ie oe1 [oe2...]
                //pc.printf("%d\r\n",c);
                if(edge_array.size() < 3) //check that it has n and one input edges and one output edge.
                    return(1);
                // takes n samples                
                NsamplesD.push_back( edge_array[0]);
                matrix[edge_array[1]] [ci] = Rationals(-edge_array[0],1);
                for (j=2; j < v.size()-1; j++) //for each outgoing edge, add matrix[j][ci] = 1
                    matrix[edge_array[j]][ ci] =Rationals(1,1);
                break;               
            case 'U':            
                //pc.printf("%d\r\n",c);
                if(edge_array.size() < 3) //check that it has n and one input edges and one output edge.
                    return(1);
                NsamplesU.push_back(edge_array[0]);
                matrix[edge_array[1]] [ci] = Rationals(-1,1);
                for (j=2; j < v.size()-1; j++) //for each outgoing edge, add matrix[j][ci] = 1
                    matrix[edge_array[j]][ ci] =Rationals(edge_array[0],1);
                break;
            case 'C':   // what to do here?
                if(edge_array.size() < 2) //check that it has C and one output edge.
                    return(1);
                //store K
                Kvalue.push_back(edge_array[0]);
                for (j=1; j < v.size()-1; j++) //for each outgoing edge, add matrix[j][ci] = 1
                    matrix[edge_array[j]][ ci] =Rationals(1,1);
                //pc.printf("%d\r\n",c);
                break;
            case 'F': //F ie oe1 oe2 [oe3 ...]
                //check that there are atleast 2 output edges
                if(edge_array.size() < 3) //check that it has atleast one input edge and two output edges.
                    return(1);
                //pc.printf("%d\r\n",c);
                matrix[edge_array[0]] [ci] = Rationals(-1,1);
                for (j=1; j < v.size()-1; j++) //for each outgoing edge, add matrix[j][ci] = 1
                    matrix[edge_array[j]][ ci] =Rationals(1,1);
                break;
            case 'E': //currently, take E edge delay [initial sample size or latest, oldest... to account for later]
                buffer[edge_array[0]] = edge_array[1];
                if(edge_array[1] > 0) {
                    for(int sample=0;sample<edge_array[1];sample++)
                        fifos[edge_array[0]].push(edge_array[2+sample]); //fill in with initial sample size
                }
                break;
            default:
                pc.printf("INCORRECT FORMAT\r\n");
                exit(1);
           }
           ci++;           
      }

      pc.printf("Topology Matrix: \r\n");              
      printn(edges, nodes);
      for(int i=0; i<matrix.size();i++) {
        for(int j=0; j<matrix[0].size();j++) {
            tau[i][j]=matrix[i][j].getn();      //since they're all integers anyway
           // pc.printf("%d ",tau[i][j]);
           }
        //pc.printf("\r\n");
       }
      
      // debug info for initial buffer
      /*for(int i=0; i<buffer.size();i++)
            pc.printf("%d ",buffer[i]);          
      pc.printf("\r\nEnd of Buffer\r\n");*/
      
      // we have topology matrix in Rational form. Now, find nullspace, and populate Q in integer form.
      
        //simulate no schedule
        //matrix[15][11]=Rationals(-2,1);     
        
      find_nullspace();   //if it returns from here, it means we have found a nullspace, with a schedule.      
    fclose ( fpointer );    
    return 0;
}


int read_input(FILE *fpointer)
{
    // parse input file
    // read inputs and store in the inBuffer    
    char temp[100];
    //read repetitions
    fgets(temp, 50, fpointer);
    sscanf(temp, "%u",&X);
    if(X<1 || X>65535)
        return 1;
    //read input samples per repetition
    fgets(temp, 50, fpointer);
    sscanf(temp, "%d",&K);
    if(K<1 || K>4096)
        return 1;
    
    //to read K sample lines and store in inBuffer
    int i=0;
    while ( fgets ( temp, sizeof temp, fpointer ) != NULL ) 
      {
        sscanf(temp,"%ld",&inBuffer[i]);
        //pc.printf("I read %ld", inBuffer[i]);
        i++;
      }    
    fclose(fpointer);
    return 0;
}

int main() {
    //start timer
    t.start();
    
    //set host connection
    pc.baud(9600);
    
    //setting pins 21,22, and 23 to 0
    pin21 = 0;
    pin22 = 0;
    pin23 = 0;
    
    //read from local file system
    FILE *lfp = fopen("/local/sdfgconf.txt","r");
    
    //missing file
    if(lfp == NULL) {
        fprintf(stderr, "MISSING CONFIGURATION FILE \r\n");    
        exit(1);
    }
    
    pc.printf("\r\nCreating SDFG and Topology Matrix\r\n");
    int e = parse_conf(lfp);
    // if e = 1, unknown error - other errors taken care of within function
    // if e = 0, succesful parsing
    if (e == 1) {
        fprintf(stderr, "sdgconf:INCORRECT FORMAT \r\n");    
        exit(1);
    }
    //successful return
    //done parsing SDFG
    
    led1 = 1;
    pin21 = 1; //ready    
    pc.printf("READY @%dms\r\n", t.read_ms());  
    //parse input.txt    
    FILE *ifp = fopen("/local/input.txt","r");        
    //missing file
    if(ifp == NULL) {
        error("ERROR: NO INPUT");
    }
    pin22 = 1;
    e = read_input(ifp);
    if(e) {
        error("ERROR: BAD INPUT FORMAT");
    }
    //successful reading        
    pin23 = 1;    
    process_input();    
    pin23 = 0;    
    pc.printf("COMPLETE\r\n");
    return 0;
}