#include "mbed.h"
#include "MODSERIAL.h"

//uCAM command defines, page 8 of uCAM manual
#define uCAM_INITIAL            0xAA01
#define uCAM_GET_PICTURE        0xAA04
#define uCAM_SNAPSHOT           0xAA05
#define uCAM_SET_PACKAGE_SIZE   0xAA06
#define uCAM_SET_BAUD_RATE      0xAA07
#define uCAM_RESET              0xAA08
#define uCAM_DATA               0xAA0A
#define uCAM_SYNC               0xAA0D
#define uCAM_ACK                0xAA0E
#define uCAM_NAK                00xAA0F
#define uCAM_LIGHT              0xAA13

char buf[40];
char picture[4800];

MODSERIAL _cam(p9, p10, 4800);
Serial pc(USBTX,USBRX);
LocalFileSystem local("local");

DigitalOut myled(LED1);

void uCAM_FlushBuffer(void){
    while(_cam.readable()){     //flush the buffer
        char c = _cam.getc();   
    }
}

int uCAM_read(char *str, int numchars, float timeout)
{
    Timer t_Frame;
    int i = 0;
    int timeoutState = 0;
    t_Frame.start();

    while(timeoutState != 3  && (i < numchars)) {
        switch(timeoutState) {
            case 0:
                while(_cam.readable()) { //if characters are in buffer, read them
                    str[i++] = _cam.getc();
                    if(i == numchars) {
                        timeoutState = 3;
                        break;
                    }
                }
                //if no characters in buffer, initiate timeout
                timeoutState = 1;
                break;
            case 1:
                timeout = t_Frame.read() + timeout;    //current time plus timeout time
                timeoutState = 2;
//              pc.printf("Timeout initiated %f\r\n", timeout);
                break;
            case 2:
                if(_cam.readable()) { //check buffer while timeout is running
                    str[i++] = _cam.getc();
                    if(i == numchars) {
                        timeoutState = 3;
                    } else {
                        timeoutState = 0;
                    }
                    break;
                }
                if(t_Frame.read() >= timeout) //if timeout has elapsed, exit the while loop with state 3
                    timeoutState = 3;
                break;
            default:
                timeoutState = 0;
                break;
        }//switch timeoutState
    }//while timeoutState != 2
    return i;   //return number of bytes read
}

void uCAM_Command_Send(int command,char p1,char p2,char p3,char p4)
{
    _cam.putc((char)(command >> 8) & 0xff);
    _cam.putc((char)(command & 0xff));
    _cam.putc(p1);
    _cam.putc(p2);
    _cam.putc(p3);
    _cam.putc(p4);
}

int uCAM_GetACK(int command)
{
    char ser_read[7];
    int i;

    for(i=0; i<7; i++)       //clear the string
        ser_read[i] = 0;

    wait(.1);

    //read serial buffer and wait for ACK (0xAA0E0DXX0000)
    uCAM_read(ser_read, 6, .5);
    if((ser_read[0] == 0xAA) &&
            (ser_read[1] == 0x0E)) {
        if((command & 0xff) == ser_read[2])
            return 1;
    } else
        return 0;
}

int uCAM_Connect(void)
{
    char ser_read[20];
    int i, retries;

    for(i=0; i<20; i++)          //clear the string
        ser_read[i] = 0;
    retries=0;

    while(retries < 60) {
        uCAM_FlushBuffer();

        //Transmit SYNC command
        uCAM_Command_Send(uCAM_SYNC,0x00,0x00,0x00,0x00);

        wait(.2);

        //read serial buffer and wait for ACK (0xAA0E0DXX0000)
        uCAM_read(ser_read, 6, .1);

        if((ser_read[0] == 0xAA) &&
                (ser_read[1] == 0x0E) &&
                (ser_read[2] == 0x0D) &&  //skip ser_read[3]
                (ser_read[4] == 0x00)) {
            //after receiving ACK, wait for SYNC (0xAA0D00000000)
            uCAM_read(ser_read, 6, .100);

            if((ser_read[0] == 0xAA) &&
                    (ser_read[1] == 0x0D) &&
                    (ser_read[2] == 0x00) &&
                    (ser_read[3] == 0x00) &&
                    (ser_read[4] == 0x00) &&
                    (ser_read[5] == 0x00)) {
                //Transmit ACK command
                uCAM_Command_Send(uCAM_ACK,0x0D,0x00,0x00,0x00);

                printf("\r\n uCAM 120 Initialized\r\nDelaying 2 seconds for AGC and AEC \r\n circuits to stabilise before image capture.");
                wait(.5);
                printf(".\r\nFinished\r\n");

                return 1;           //Camera connection successful
            }
        }
        retries++;
    }
    if(retries == 60)
        return 0;

    return -1;
}

int uCAM_send_INITIAL_320x240_8RAW(void)
{
   char ser_read[7];
   int i;

   //uCAM_Command_Send(uCAM_INITIAL,0x00,0x03,0x05,0x00); // p2=0x03-8 bit RAW
                                                        // p3=0x05-320 x 240 bit resolution
                                                        // p4=0x00-raw
    uCAM_Command_Send(uCAM_INITIAL,0x00,0x04,0x01,0x00);
   for(i=0;i<7;i++)         //clear the string
        ser_read[i] = 0;

   //read serial C buffer and wait for ACK (0xAA0E0DXX0000)
   uCAM_read(ser_read, 6, .02); //.01
   
   if((ser_read[0] == 0xAA) &&
      (ser_read[1] == 0x0E) &&
      (ser_read[2] == 0x01)){
        printf("init OK\r\n");
      return 1;
   }
   else
      return 0;
}

int uCAM_send_SNAPSHOT(void)
{
    uCAM_Command_Send(uCAM_SNAPSHOT,0x01,0x00,0x00,0x00); // p1=0x01-Uncompressed Image
                                                            // p2 and p3 = 0, current frame
    wait(.01);

    if(uCAM_GetACK(uCAM_SNAPSHOT)){
        printf("Snapshot OK\r\n");
        return 1;}
    else
        return 0;
}

int uCAM_send_GET_PICTURE_320x240_8GRAY_RAW(FILE *fp)
{
    char ser_read[7];
    unsigned long pic_bytes;
    unsigned int i = 0, j, k;
    unsigned int serbytes_read;
    unsigned int pixel_col;

    pic_bytes = 0;
    serbytes_read = 0;

    while(_cam.readable()){     //flush the buffer
        char c = _cam.getc();   
    }

    uCAM_Command_Send(uCAM_GET_PICTURE,0x01,0x00,0x00,0x00); // p1=0x01-Snapshot picture
    wait(.1);

   if(uCAM_GetACK(uCAM_GET_PICTURE))     //returned get pic
   {
   //read serial C buffer and wait for ACK (0xAA0E0DXX0000)
      char c=uCAM_read(ser_read, 6, .1);

     if((ser_read[0] == 0xAA) &&   //first 2 bytes indicate it was a DATA packet
        (ser_read[1] == 0x0A))
     {
        printf("get pic sent\r\n");
        pic_bytes = (unsigned long)ser_read[3];
        pic_bytes += (unsigned long)ser_read[4] << 8;
        pic_bytes += (unsigned long)ser_read[5] << 16;
        
        printf("pic size = %d\r\n", pic_bytes);

        wait(0.5);
        while(_cam.readable())
        //for (int i = 0; i < pic_bytes; i++)
        {
          char c = _cam.getc();
          picture[i++] = c;
          fputc(c, fp);     //write pixel high byte to file  
        }
        uCAM_Command_Send(uCAM_ACK, 0x0A, 0x00, 0x01, 0x00);
     }
  }
  else
    return -1;
}

void uCAM_set_baud(void)
{
   uCAM_Command_Send(uCAM_SET_BAUD_RATE,0x03,0x00,0x00,0x00);   // set baud to
   wait(.05);
   
   if(uCAM_GetACK(uCAM_SET_BAUD_RATE)){
        printf("Baud rate sucessfully changed");
    }
    else{
        printf("Baud rate NOT sucessfully changed");
    }
   
   _cam.baud(921600);
}

void uCAM_TakePic_RAW_8GRAY_320x240(FILE *fp)
{
    uCAM_send_INITIAL_320x240_8RAW();
    wait(.1);
    uCAM_send_SNAPSHOT();
    wait(.1);
    uCAM_send_GET_PICTURE_320x240_8GRAY_RAW(fp);
}

int uCAM_get_jpeg(FILE *fp){
    
    char ser_read[7];
    unsigned long pic_bytes;
    
    uCAM_FlushBuffer();
    //uCAM_Command_Send(uCAM_INITIAL,0x00,0x07,0x07,0x07); // JPEG,640 x 480,JPEG
    //uCAM_Command_Send(uCAM_INITIAL,0x00,0x07,0x07,0x03); // JPEG,160 x 120,JPEG   
    uCAM_Command_Send(uCAM_INITIAL,0x00,0x07,0x09,0x05); // JPEG,320 x 240,JPEG 
    wait(.05);
        
    if(uCAM_GetACK(uCAM_INITIAL)){     //INITIAL ACK
        printf("ACK received on uCAM_INITIAL\r\n");
    }
    else{
       printf("Did not receive ACK on uCAM_INITIAL\r\n");
    }
                
    uCAM_FlushBuffer();         
    uCAM_Command_Send(uCAM_SET_PACKAGE_SIZE,0x08,0x00,0x02,0x00); //512 bytes

    wait(.05);
            
    if(uCAM_GetACK(uCAM_SET_PACKAGE_SIZE)){     //SET PACKAGE SIZE ACK
        printf("ACK received on uCAM_SET_PACKAGE_SIZE\r\n");
    }
    else{
        printf("Did not receive ACK on SET PACKAGE SIZE\r\n");
    }        
            
    uCAM_FlushBuffer();     
    uCAM_Command_Send(uCAM_SNAPSHOT,0x00,0x00,0x00,0x00); //JPEG

    if(uCAM_GetACK(uCAM_SNAPSHOT)){     //uCAM_SNAPSHOT ACK
        printf("ACK received on uCAM_SNAPSHOT\r\n");
    }
    else{
        printf("Did not receive ACK on uCAM_SNAPSHOT\r\n");
    }

    pic_bytes = 0;
    
    uCAM_FlushBuffer();
    uCAM_Command_Send(uCAM_GET_PICTURE,0x01,0x00,0x00,0x00); // p1=0x01-Snapshot picture
    wait(.1);

    if(uCAM_GetACK(uCAM_GET_PICTURE))     //returned get pic
    {
        //read serial buffer and wait for ACK (0xAA0E0DXX0000)
        uCAM_read(ser_read, 6, 1);

        if((ser_read[0] == 0xAA) &&   //first 2 bytes indicate it was a DATA packet
            (ser_read[1] == 0x0A) &&
            (ser_read[2] == 0x01))      //DATA from snapshot ACK
        {
            pic_bytes = (unsigned long)ser_read[3];
            pic_bytes += (unsigned long)ser_read[4] << 8;
            pic_bytes += (unsigned long)ser_read[5] << 16;

            int packID=0;
            char packID_l, packID_h;
                    
            pc.printf("pic size = %d\r\n", pic_bytes);
            int numPackages = pic_bytes / (512 - 6);
                     
            while(numPackages>=0){
                packID_h = (packID >> 8) & 0xff;
                packID_l = packID & 0xff;
                        
                uCAM_FlushBuffer();
                uCAM_Command_Send(uCAM_ACK, 0x00, 0x00, packID_l, packID_h);    
                        
                char ID_ret[2];
                uCAM_read(ID_ret, 2, .2);
                
                char dataSize[2];
                uCAM_read(dataSize, 2, .2);
                
                char imageData[506];    //512 - 6
                int imageDataSize = ((dataSize[1] & 0xff) << 8) + (dataSize[0] & 0xff);                        
                
                //uCAM_read(imageData, imageDataSize, .2);
                for(int i=0;i<imageDataSize;i++)
                    imageData[i] = _cam.getc();
                
                char verifyCode[2];
                uCAM_read(verifyCode, 2, .2);

                int vCodeCheck=0;
                vCodeCheck += ID_ret[0];
                vCodeCheck += ID_ret[1];
                vCodeCheck += dataSize[0];
                vCodeCheck += dataSize[1];
                
                for(int i=0;i<imageDataSize;i++)
                    vCodeCheck += imageData[i];

                if((verifyCode[0] & 0xff) == (vCodeCheck & 0xff)){                            
                    for(int j=0;j<imageDataSize;j++){   //write image data to file pointer
                        fputc(imageData[j], fp);
                    }                                                                   
                    packID++;
                    numPackages--;
                }
                else{
                    printf("\r\nBad Data in transfer. ReTransfer\r\n");
                    return -2;
                }
            }
           uCAM_Command_Send(uCAM_ACK, 0x00, 0x00, 0xf0, 0xf0);
           return 0; //return 0 on success
        }
    }//data packet
    return -1;
}  //function

int main()
{
    int picture_index = 0;
    char c;
    
    pc.baud(115200);
    _cam.baud(115200);
    wait(1.0);
    
    pc.printf("Send 'c' to start capturing JPEG pictures...\r\n");
    pc.printf("Send 'r' to start capturing RAW pictures...\r\n");
    myled = 0;
    
    while(1) {
        if (pc.readable()) {
            c = pc.getc();
            
            if (c == 'c') {
                pc.printf("Start capturing JPEG picture\r\n");

                sprintf(buf,"/local/%d_out.jpg",picture_index);
                FILE *fp = fopen(buf, "w");
                wait(0.1);
                
                //Take picture and save it to flash memory
                if (uCAM_Connect() != 0) {
                    uCAM_get_jpeg(fp);
                    //uCAM_TakePic_RAW_16COLOR_80x60(fp);
                }
                fclose(fp);
                //get last taken picture from flash and send it to pc
                fp = fopen(buf,"r");
                int c = fgetc(fp);
                pc.putc(c);
                while (c!= EOF) {
                    c = fgetc(fp);
                    pc.putc((char) c);
                }
                fclose(fp);
                wait(0.1);
                picture_index += 1;
            }
            
            if (c == 'r') {
                pc.printf("Start capturing RAW picture\r\n");

                sprintf(buf,"/local/%d_out.raw",picture_index);
                FILE *fp = fopen(buf, "w");
                wait(0.1);
                
                //Take picture and save it to flash memory
                if (uCAM_Connect() != 0) {
                    uCAM_TakePic_RAW_8GRAY_320x240(fp);
                }
                fclose(fp);
                //get last taken picture from flash and send it to pc
                //fp = fopen(buf,"r");
//                int c = fgetc(fp);
//                pc.putc(c);
//                while (c!= EOF) {
//                    c = fgetc(fp);
//                    pc.putc((char) c);
//                }
//                fclose(fp);
                for (int i = 0; i < 4800; i++)
                {
                    pc.putc(picture[i]);
                }
                wait(0.1);
                picture_index += 1;
            }
        }
        myled = !myled;
        wait(0.2);
    }
}
