/*******************************************************************************
 * Copyright (c) 2014 IBM Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *    http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Ian Craggs - initial API and implementation and/or initial documentation
 *******************************************************************************/
 
 /**
  This is a sample program to illustrate the use of the MQTT Client library
  on the mbed platform.  The Client class requires two classes which mediate
  access to system interfaces for networking and timing.  As long as these two
  classes provide the required public programming interfaces, it does not matter
  what facilities they use underneath. In this program, they use the mbed
  system libraries.
 
 */

#include "mbed.h"
#include "EthernetInterfaceIPStack.h"
#include "C027.h"
#include "MDM.h"

//----------------------------------------------------------------------
// You may need to configure these parameters

/** Set your secret SIM pin here "1234"
*/
#define SIMPIN      NULL

/** The APN of your network operator, sometimes it is "internet" 
    check your contract with the network operator
*/
#define APN         "gprs.swisscom.ch"

/** Set the user name for your APN, or NULL if not needed
*/
#define USERNAME    NULL

/** Set the password for your APN, or NULL if not needed
*/
#define PASSWORD    NULL 

C027 c027;

#include "C12832.h"
C12832 lcd(D11, D13, D12, D7, D10);

#include "FP.cpp"
#include "MQTTClient.h"

int arrivedcount = 0;

void messageArrived(MQTT::Message* message)
{
    lcd.cls();
    lcd.locate(0,3);
    printf("Message arrived: qos %d, retained %d, dup %d, packetid %d\n", message->qos, message->retained, message->dup, message->id);
    printf("Payload %.*s\n", message->payloadlen, (char*)message->payload);
    ++arrivedcount;
    lcd.puts((char*)message->payload);
}

int main(int argc, char* argv[])
{   
    // turn on the supplies of the Modem and the GPS
    c027.mdmPower(true);
    wait(2);
    // Create the modem object
    MDMSerial mdm;
    
    // initialize the modem 
    printf("Modem Initialize\r\n");
    MDMParser::DevStatus devStatus;
    bool mdmOk = mdm.init(SIMPIN, &devStatus);
    if (mdmOk)
    {
        // wait until we are connected
        printf("Network Check\r\n");
        MDMParser::NetStatus netStatus;
        while (!mdm.checkNetStatus(&netStatus))
            wait_ms(1000);
    
        printf("Network Join\r\n");
        // join the internet connection 
        MDMParser::IP ip = mdm.join(APN,USERNAME,PASSWORD);
        if (ip != NOIP)
        {
            printf("  IP Address: " IPSTR "\r\n", IPNUM(ip));
            
            IPStack ipstack = IPStack();
            float version = 0.43;
            char* topic = "mbed-sample";
            
            lcd.printf("Version is %f\n", version);
            printf("Version is %f\n", version);
                      
            MQTT::Client<IPStack, Countdown> client = MQTT::Client<IPStack, Countdown>(ipstack);
            
            char* hostname = "m2m.eclipse.org";
            int port = 1883;
            lcd.printf("Connecting to %s:%d\n", hostname, port);
            int rc = ipstack.connect(hostname, port);
            if (rc != 0)
                lcd.printf("rc from TCP connect is %d\n", rc);
         
            MQTTPacket_connectData data = MQTTPacket_connectData_initializer;       
            data.MQTTVersion = 3;
            data.clientID.cstring = "mbed-icraggs";
            rc = client.connect(&data);
            if (rc != 0)
                lcd.printf("rc from MQTT connect is %d\n", rc);
            
            rc = client.subscribe(topic, MQTT::QOS1, messageArrived);   
            if (rc != 0) {
                printf("rc from MQTT subscribe is %d\n", rc);
            }
        
            MQTT::Message message;
        
            // QoS 0
            char buf[100];
            sprintf(buf, "Hello World!  QoS 0 message from app version %f\n", version);
            message.qos = MQTT::QOS0;
            message.retained = false;
            message.dup = false;
            message.payload = (void*)buf;
            message.payloadlen = strlen(buf)+1;
            rc = client.publish(topic, &message);
            while (arrivedcount == 0)
                client.yield(100);
                
            // QoS 1
            sprintf(buf, "Hello World!  QoS 1 message from app version %f\n", version);
            message.qos = MQTT::QOS1;
            message.payloadlen = strlen(buf)+1;
            rc = client.publish(topic, &message);
            while (arrivedcount == 1)
                client.yield(100);
                
            // QoS 2
            sprintf(buf, "Hello World!  QoS 2 message from app version %f\n", version);
            message.qos = MQTT::QOS2;
            message.payloadlen = strlen(buf)+1;
            rc = client.publish(topic, &message);
            while (arrivedcount == 2)
                client.yield(100);
            
            rc = client.unsubscribe(topic);
            if (rc != 0) {
                printf("rc from unsubscribe was %d\n", rc);
            }
            
            rc = client.disconnect();
            if (rc != 0) {
                printf("rc from disconnect was %d\n", rc);
            }

            mdm.disconnect();
        }
    }
    mdm.powerOff();
    c027.mdmPower(false);
    
    lcd.cls();
    lcd.locate(0,3);
    lcd.printf("Finish: %d msgs\n", arrivedcount);
    printf("Finishing with %d messages received\n", arrivedcount);
    
    while(true);
}
