#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "ExperimentServer.h"
#include "QEI.h"
#include "MotorShield.h" 
#include "HardwareSetup.h"

#define NUM_INPUTS 6
#define NUM_OUTPUTS 5

//Measured values
float velocity = 0;
float current = 0;
float theta = 0;

//Set values
float current_d = 0;
float kp = 0;
float ki = 0;
float K = 0;
float b = 0;
float kb = .2;
float D = 0;

//Controller values
float volt = 0;
float duty = 0;

Serial pc(USBTX, USBRX);    // USB Serial Terminal
ExperimentServer server;    // Object that lets us communicate with MATLAB
Timer t;                    // Timer to measure elapsed time of experiment
Ticker ControlLoop;       //Ticker object created

QEI encoderA(PE_9,PE_11, NC, 1200, QEI::X4_ENCODING);  // MOTOR A ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderB(PA_5, PB_3, NC, 1200, QEI::X4_ENCODING);  // MOTOR B ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderC(PC_6, PC_7, NC, 1200, QEI::X4_ENCODING);  // MOTOR C ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderD(PD_12, PD_13, NC, 1200, QEI::X4_ENCODING);// MOTOR D ENCODER (no index, 1200 counts/rev, Quadrature encoding)

MotorShield motorShield(12000); //initialize the motor shield with a period of 12000 clock ticks or ~20kHZ

void current_control()   //Current control function
{
    float input_params[NUM_INPUTS];
    
    float error = 0;
    float sum_error = 0;
    float limit = 100;
    theta = encoderA.getPulses()*(6.2831/1200.0);
    velocity = encoderA.getVelocity()*(6.2831/1200.0);
    current = -(motorShield.readCurrentA()*(30.0/65536.0)-15.0); //read current for motor A in amps. Note: this is a slightly different current sensor so its a different conversion than last lab.            
    error = current_d - current;
    float R = 6.0/2.777;
    float kb = .2;
    

    if (sum_error > limit) {
        sum_error = limit;
    }
    else if (sum_error < -limit) {
        sum_error = limit;
    }
    else {
      sum_error = sum_error + error;
    }  

    volt = kp*error + ki*sum_error + current_d*R + kb*velocity; //CHANGE THIS TO SET CURRENT CONTROLLER VALUES
    
    duty  = volt/12.0;
    if (duty > 1){
        duty = 1;
    }
    if (duty <-1){
        duty = -1;   
    }
    if (duty >= 0){
        motorShield.motorAWrite(duty, 0); //run motor A at "v1" duty cycle and in the forward direction 
    }
    else if (duty < 0){
        motorShield.motorAWrite(abs(duty), 1);
    }
}


int main (void)
{
    // Link the terminal with our server and start it up
    server.attachTerminal(pc);
    server.init();
    
    // Continually get input from MATLAB and run experiments
    float input_params[NUM_INPUTS];
    pc.printf("%f",input_params[0]);
    
    while(1) {
        if (server.getParams(input_params,NUM_INPUTS)) {
            kp = input_params[0];
            ki = input_params[1];
            K = input_params[2];
            b = input_params[3];
            D = input_params[4];
            current_d = input_params[5];

            ControlLoop.attach(&current_control,0.0001); //Run current controller at 10khz
            
            // Setup experiment
            t.reset();
            t.start();
            
            encoderA.reset();
            encoderB.reset();
            encoderC.reset();
            encoderD.reset();

            motorShield.motorAWrite(0, 0); //turn motor A off
            
            //use the motor shield as follows:
            //motorShield.motorAWrite(DUTY CYCLE, DIRECTION), DIRECTION = 0 is forward, DIRECTION =1 is backwards.
             
            // Run experiment
            while( t.read() < 5 ) {
                // Perform control loop logic
                current_d = (-K*theta + b*velocity - D*velocity)/kb; // Set commanded current from impedance controller here.
                //current_d = sin(3*t.read());
                
                // Send data to MATLAB
                float output_data[NUM_OUTPUTS];
                output_data[0] = t.read();
                output_data[1] = theta;
                output_data[2] = velocity;
                output_data[3] = current;
                output_data[4] = volt;

                server.sendData(output_data,NUM_OUTPUTS);      
                        
                wait(.001); //run outer control loop at 1kHz
            }
            // Cleanup after experiment
            ControlLoop.detach();
            server.setExperimentComplete();
            motorShield.motorAWrite(0, 0); //turn motor A off
        } // end if
    } // end while
    
} // end main

