/*CUER Battery Management System - Battery Management Unit (BMU) Code

@author
Zi Koon Pong (zkp20)

@brief
This file contains the core code which runs on the BMU LPC1768 MBED. The primary function of the BMU is to obtain the relevant parameters of the battery pack which we are interested in monitoring and]
placing it on the CAN Bus for the other nodes on the network to utilise. In the event that one of our parameters goes out of a predefined safe threshold (typically specified in the cell manufacturers
datasheet) we begin to limit the performance of the car so that it gets back to a region needed for safe operation (to be discussed further with telemetry).

There are 3 primary communication protocols that the BMU utilises: CAN, SPI and I2C.

-CAN is used to dump battery status information to the rest of the car
-I2C is used to commmunicate with the state of charge (SoC) board
-SPI is used to communicate with the CMUs which are individually addressed

The mbed itself utilises conventional SPI and I2C which both get converted into differential signals to give isoSPI and dI2C to give
greater resilience to EM noise reducing the chance of transmission errors.

dI2C conversion is done by the PCA9615 IC
isoSPI conversion is done by the LTC6820 IC

Most of the low-level code which involves accessing individual registers in the LTC6804 and LTC2943 have been ported over from existing Linduino code provided by Linear Tech (disclaimer below).
If any of the source code is unclear, feel free to contact me through email at zkp20@cam.ac.uk and I will be more than happy to help out.


*** FOR ZI'S REFERENCE ***
Initialise BMU

Check status of SoC and all CMU boards (comms. etc.)

Perform first voltage measurement and check that it coincides with the number of cells we are expecting

Give state of BMS. If ok, go to run. Default into safe state and set error flag.

LOOP
Start off displaying state
Run SoC updating algorithm.
Run current measurement.
Perform voltage measurement.
Run isDischargeNeeded function.
Run isSafetoDischarge function.


Dump following data on CAN network:
- Cell voltage
- Cell temperatures
- State of charge
- Balancing status
- Error flag

Will add helper function for communicating with both CMU (SPI) and SoC (I2C)
*/

/*! LINEAR TECH DISCLAIMER
LTC6804-2 Multicell Battery Monitor
@verbatim
The LTC6804 is a 3rd generation multicell battery stack
monitor that measures up to 12 series connected battery
cells with a total measurement error of less than 1.2mV. The
cell measurement range of 0V to 5V makes the LTC6804
suitable for most battery chemistries. All 12 cell voltages
can be captured in 290uS, and lower data acquisition rates
can be selected for high noise reduction.

Using the LTC6804-2, multiple devices are connected in
parallel to the host processor, with each device
individually addressed.
@endverbatim
REVISION HISTORY
$Revision: 1000 $
$Date: 2013-12-13

Copyright (c) 2013, Linear Technology Corp.(LTC)
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of Linear Technology Corp.

The Linear Technology Linduino is not affiliated with the official Arduino team.
However, the Linduino is only possible because of the Arduino team's commitment
to the open-source community.  Please, visit http://www.arduino.cc and
http://store.arduino.cc , and consider a purchase that will help fund their
ongoing work.

Copyright 2013 Linear Technology Corp. (LTC)
***********************************************************/

#include "BMU.h"
#include "SPI_I2C_Parser.h"
#include "mbed.h"
#include "CAN_Data.h"
#include "CAN_IDs.h"
#include <stdlib.h>

using namespace std;


//Define battery pack cooling fan PWM pins
PwmOut fan1(p21);
PwmOut fan2(p22);

//Define SoC ALCC digital in pin
DigitalIn alcc(p20);
Timer t;

int main ()
{
    //spi.format(8,3); //All data transfer on LTC6804 occur in byte groups. LTC6820 set up such that POL=1 and PHA=3, this corresponds to mode 3 in mbed library. spi.frequency(spiBitrate);
    //spi.frequency(spiBitrate);
    uint8_t write [1][6];
    uint8_t read [1][8];
    uint16_t cellcodes [1][12];
    
    write[0][0]=0x00;
    write[0][1]=0x00;
    write[0][2]=0x00;
    write[0][3]=0x00;
    write[0][4]=0x00;
    write[0][5]=0x00;

    LTC6804_init(MD_FAST, DCP_DISABLED, CELL_CH_ALL, AUX_CH_VREF2);
    //LTC6804_init(MD_NORMAL,DCP_DISABLED,CELL_CH_ALL,AUX_CH_ALL);
    bool swap = 1;
    wake_LTC6804();
    LTC6804_wrcfg(1,write);
    while(1) {
        wake_LTC6804();//ensures CMU's are in ready state and wakes it up from low power mode
        //LTC6804_wrcfg(1,write);
        wait_us(330);
        LTC6804_rdcfg(1,read);
         wait_us(330);
       wake_LTC6804();
        LTC6804_acquireVoltageTx();
         wait_us(930);
        LTC6804_acquireAllVoltageRegRx(0, 1, cellcodes);
        //t.stop();
        //cout << t.read() << endl;
        wait(3);
        for (int i=0; i<8; i++) {
            printf("Reading %d config \r\n", (uint8_t)read[0][i]); 
        }
        for (int i=0; i<12; i++) {
            printf("Reading %d mV \r\n", (uint16_t)cellcodes[0][i]); 
        }
        printf("\r\n"); 
        
        //Test Balancing Code
        
        uint8_t states[12];
        for(int i = 0; i < 12; i++)
        {
            states[i] = swap ^ (i % 2);   
        }
        LTC6804_balance(1,0,states);
        swap = !swap;
    }
}


/*uint16_t vCells[numberOfCMUs][12];


LTC6804_init(MD_FILTERED, DCP_ENABLED, CELL_CH_ALL, AUX_CH_ALL);//set_adc
LTC6804_setConfigReg(

LTC6804_acquireVoltageTx();//adcv
LTC6804_acquireAllVoltageRegRx(0, numberOfCMUs, vCells); //rdcv, 0 means read all cell voltages
*/

