////////////////////////
//
// Simple program for the RETRO to compare the performance of the DisplayN18 and LCD_ST7735 libraries
//
///////////////////////

#include <stdarg.h>
#include "mbed.h"
#include "DisplayN18.h"
#include "Color565.h"
#include "font_IBM.h"
#include "LCD_ST7735.h"

void bottomPrintf_N18(DisplayN18 &disp, const char *szFormat, ...)
{
    char szBuffer[256];
    va_list args;

    va_start(args, szFormat);
    vsprintf(szBuffer, szFormat, args);
    va_end(args);
    disp.drawString(DisplayN18::WIDTH / 2 - (DisplayN18::CHAR_WIDTH + DisplayN18::CHAR_SPACING) * strlen(szBuffer) / 2, DisplayN18::HEIGHT-DisplayN18::CHAR_HEIGHT, szBuffer, DisplayN18::GREEN, DisplayN18::BLACK);
}

int testDisplayN18()
{   // test N18 library functions
    DisplayN18 disp;
    Timer tPerformance;    // timer used for measuring performance
    char szTest[]="123TestText456";
    int iPerfMs=0;
    int iStartUs=0;

    tPerformance.start();      // start the timer

    // text
    iStartUs=tPerformance.read_us();
    disp.clear();
    bottomPrintf_N18(disp, "drawString...");
    for(int i=1; i<100; i++)
    {
        disp.drawString(DisplayN18::WIDTH / 2 - (DisplayN18::CHAR_WIDTH + DisplayN18::CHAR_SPACING) * strlen(szTest) / 2, i, szTest, DisplayN18::GREEN, DisplayN18::BLACK);
    }
    bottomPrintf_N18(disp, "drawString:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // rectangles
    iStartUs=tPerformance.read_us();
    disp.clear();
    bottomPrintf_N18(disp, "drawRect...");
    for(int i=1; i<100; i++)
    {
        disp.drawRect(0, 0, i, i, DisplayN18::GREEN);
    }
    bottomPrintf_N18(disp, "drawRect:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // filled rectangles
    iStartUs=tPerformance.read_us();
    disp.clear();
    bottomPrintf_N18(disp, "fillRect...");
    for(int i=1; i<100; i+=2)
    {
        disp.fillRect(0, 0, i, i, DisplayN18::GREEN);
    }
    bottomPrintf_N18(disp, "fillRect:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // circles
    iStartUs=tPerformance.read_us();
    disp.clear();
    bottomPrintf_N18(disp, "drawCircle...");
    for(int i=1; i<100; i++)
    {
        disp.drawCircle(i, i, i/2, DisplayN18::GREEN);
    }
    bottomPrintf_N18(disp, "drawCircle:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // filled circles
    iStartUs=tPerformance.read_us();
    disp.clear();
    bottomPrintf_N18(disp, "fillCircle...");
    for(int i=1; i<100; i+=2)
    {
        disp.fillCircle(i, i, i/2, DisplayN18::GREEN);
    }
    bottomPrintf_N18(disp, "fillCircle:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // lines
    iStartUs=tPerformance.read_us();
    disp.clear();
    bottomPrintf_N18(disp, "drawLine...");
    for(int i=1; i<100; i++)
    {
        disp.drawLine(0, 0, i, 100, DisplayN18::GREEN);
    }
    bottomPrintf_N18(disp, "drawLine:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    disp.clear();
    bottomPrintf_N18(disp, "Total:%u ms", iPerfMs);
    wait(2);

    return(iPerfMs);
}

void bottomPrintf_ST7735(LCD_ST7735 &disp, const char *szFormat, ...)
{
    char szBuffer[256];
    va_list args;

    va_start(args, szFormat);
    vsprintf(szBuffer, szFormat, args);
    va_end(args);
    disp.drawString(font_ibm, disp.getWidth() / 2 - (8 + 0) * strlen(szBuffer) / 2, disp.getHeight()-8, szBuffer);
}

int testDisplayST7735()
{   // test ST7735 library functions
    LCD_ST7735 disp(
        P0_19,
        P0_20,
        P0_7,
        P0_21,
        P0_22,
        P1_15,
        P0_2,
        LCD_ST7735::RGB);
    Timer tPerformance;    // timer used for measuring performance
    char szTest[]="123TestText456";
    int iPerfMs=0;
    int iStartUs=0;

    tPerformance.start();      // start the timer
    disp.setOrientation(LCD_ST7735::Rotate270, false);
    disp.setForegroundColor(Color565::Blue);
    disp.setBackgroundColor(Color565::Black);
    disp.clearScreen();
    
    // text
    iStartUs=tPerformance.read_us();
    disp.clearScreen();
    bottomPrintf_ST7735(disp, "drawString...");
    for(int i=1; i<100; i++)
    {
        //disp.drawString(font_ibm, disp.getWidth / 2 - (8 + 0) * strlen(buf) / 2, i, buf);
        disp.drawString(font_ibm, 160 / 2 - (8 + 0) * strlen(szTest) / 2, i, szTest);
    }
    bottomPrintf_ST7735(disp, "drawString:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // rectangles
    iStartUs=tPerformance.read_us();
    disp.clearScreen();
    bottomPrintf_ST7735(disp, "drawRect...");
    for(int i=1; i<100; i++)
    {
        disp.drawRect(0, 0, i, i, Color565::Blue);
    }
    bottomPrintf_ST7735(disp, "drawRect:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // filled rectangles
    iStartUs=tPerformance.read_us();
    disp.clearScreen();
    bottomPrintf_ST7735(disp, "fillRect...");
    for(int i=1; i<100; i+=2)
    {
        disp.fillRect(0, 0, i, i, Color565::Yellow, Color565::Blue);
    }
    bottomPrintf_ST7735(disp, "fillRect:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // circles
    iStartUs=tPerformance.read_us();
    disp.clearScreen();
    bottomPrintf_ST7735(disp, "drawCircle...");
    for(int i=1; i<100; i++)
    {
        disp.drawCircle(i, i, i/2, Color565::Blue);
    }
    bottomPrintf_ST7735(disp, "drawCircle:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // filled circles
    iStartUs=tPerformance.read_us();
    disp.clearScreen();
    bottomPrintf_ST7735(disp, "fillCircle...");
    for(int i=1; i<100; i+=2)
    {
        disp.fillCircle(i, i, i/2, Color565::Yellow, Color565::Blue);
    }
    bottomPrintf_ST7735(disp, "fillCircle:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    // lines
    iStartUs=tPerformance.read_us();
    disp.clearScreen();
    bottomPrintf_ST7735(disp, "drawLine...");
    for(int i=1; i<100; i++)
    {
        disp.drawLine(0, 0, i, 100, Color565::Blue);
    }
    bottomPrintf_ST7735(disp, "drawLine:%u ", tPerformance.read_us()-iStartUs);
    iPerfMs+=tPerformance.read_ms();
    wait(0.5);

    disp.clearScreen();
    bottomPrintf_ST7735(disp, "Total:%u ms", iPerfMs);
    wait(2);

    return(iPerfMs);
}


main()
{
    int iDisplayN18=testDisplayN18();
    int iDisplayST7735=testDisplayST7735();
    char buf[256];
    
    DisplayN18 disp;
    disp.clear();
    sprintf(buf,"Totals --->");
    disp.drawString(0, 0, buf, DisplayN18::WHITE, DisplayN18::BLACK);
    sprintf(buf,"N18:    %u ms", iDisplayN18);
    disp.drawString(0, DisplayN18::CHAR_HEIGHT, buf, DisplayN18::GREEN, DisplayN18::BLACK);
    sprintf(buf,"ST7735: %u ms", iDisplayST7735);
    disp.drawString(0, 2*DisplayN18::CHAR_HEIGHT, buf, DisplayN18::BLUE, DisplayN18::BLACK);

    
    DigitalOut led1(P0_9, false);
    DigitalOut led2(P0_8, true);

    while (true) {
        led1 = !led1;
        led2 = !led2;
        wait(.5);
    }

}
