/*
   This VL6180X Expansion board test application performs a range measurement
   and als measurement in polling mode on the onboard embedded top sensor. 
   The result of both the measures are printed on the serial over.  
   get_distance() and get_lux() are synchronous! They block the caller until the
   result will be ready.
*/


/* Includes ------------------------------------------------------------------*/

#include "mbed.h"
#include "XNucleo6180XA1.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <sstream>

/* Definitions ---------------------------------------------------------------*/

/*user parameter -------------------------------------------------------------*/
#define th 150
#define mode 1
#define interval 30

/*board parameter ------------------------------------------------------------*/
#define VL6180X_I2C_SDA   D14 
#define VL6180X_I2C_SCL   D15 
//#define EXPANDER_I2C_ADDRESS    (0x42*2)
//#define RANAGE_MAX (765)
#define RANAGE_MAX (500)
#define DISTMAX (400)
#define OFFSET (-5)
#define DISP_TH (DISTMAX+OFFSET)
#define LUX_TH (1000)
/* Variables -----------------------------------------------------------------*/

static XNucleo6180XA1 *board = NULL;

int dist_scal(int dist);
void disp_dist(int dist, XNucleo6180XA1 *board);
int lux_scal(int lux);
void disp_lux(int dist, XNucleo6180XA1 *board);
void lg_dist(int dist, XNucleo6180XA1 *board);
void lg_lux(int dist, XNucleo6180XA1 *board);

InterruptIn user_button(USER_BUTTON);
DigitalOut green_led(LED2);
int measure_flg=0;
//PortIn p
/* Functions -----------------------------------------------------------------*/

/*=================================== Main ==================================
  Prints on the serial over USB the measured distance and lux.
  The measures are run in single shot polling mode.
=============================================================================*/
void button_pressed()
{
  green_led = 1;
}

void button_released()
{
  green_led = 0;
  measure_flg=1;
}

int main()
{ 
	int status;
	uint32_t lux=0, dist=765;
	uint32_t sdist, slux;
	int mflg=0;

	DevI2C *device_i2c = new DevI2C(VL6180X_I2C_SDA, VL6180X_I2C_SCL);

	/* Creates the 6180XA1 expansion board singleton obj. */
	board = XNucleo6180XA1::instance(device_i2c, A3, A2, D13, D2);
    user_button.rise(&button_pressed);
    user_button.fall(&button_released);

	/* Initializes the 6180XA1 expansion board with default values. */
	status = board->init_board();
	if (status) {
		printf("Failed to init board!\n\r");
		return 0;
	}
#if mode == 1
	while (true) {
		mflg=board->the_switch->rd_switch();
		if(mflg==0){
			board->sensor_top->get_distance(&dist);
			sdist=dist_scal(dist);
    	    disp_dist(sdist, board);
		}else{
			board->sensor_top->get_lux(&lux);
			slux=lux_scal(lux);
    	    disp_lux(slux, board);
    	}
		//printf ("Distance: %d, Lux: %d\n\r", dist, lux);
		//printf ("Distance: %d\n\r", dist);
		//wait_ms(60);
	}
#endif
#if mode == 2
	while (true) {
		mflg=board->the_switch->rd_switch();
		if(mflg==0){
			if(measure_flg){
				board->sensor_top->get_distance(&dist);
				sdist=dist_scal(dist);
				disp_dist(sdist, board);
				measure_flg=0;
			}else{
				disp_dist(sdist, board);
			}
		}else {
			if(measure_flg){
				board->sensor_top->get_lux(&lux);
				slux=lux_scal(lux);
				disp_lux(slux, board);
				measure_flg=0;
			}else{
				disp_lux(lux_scal(lux), board);
			}
		}
	}
#endif
#if mode == 3
	while (true) {
		mflg=board->the_switch->rd_switch();
		if(mflg==0){
			board->sensor_top->get_distance(&dist);
			sdist=dist_scal(dist);
			if(sdist<th){
				lg_dist(sdist, board);
			}else{
				disp_dist(DISP_TH, board);
			}
		}else{
			board->sensor_top->get_lux(&lux);
			slux=lux_scal(lux);
			if(slux<th){
				lg_lux(slux, board);
			}else{
				disp_lux(LUX_TH, board);
			}			
		}
	}
#endif

}

int dist_scal(int dist){
    return (dist*DISTMAX)/RANAGE_MAX + OFFSET;
}

void disp_dist(int dist, XNucleo6180XA1 *board){
	char val[4] = {'R', ' ', ' ', ' '};
	int i;

	if(dist<DISP_TH && dist>0){
    	if(dist/100) sprintf(&val[1], "%d", dist);
	    else if(dist/10) sprintf(&val[2], "%d", dist);
	    else sprintf(&val[3], "%d", dist);
	}else if (dist < 0) val[3]='0';

    if(mode != 1) board->display->display_string(val, 4);
    else {
    	for(i=0;i<5;i++){
    		board->display->display_string(val, 4);
    	}
    }
}

int lux_scal(int lux){
    return lux;
}

void disp_lux(int lux, XNucleo6180XA1 *board){
	char val[4] = {'A', ' ', ' ', ' '};
	int i;
	
    if(lux<LUX_TH){
    	if(lux/100) sprintf(&val[1], "%d", lux);
    	else if(lux/10) sprintf(&val[2], "%d", lux);
    	else sprintf(&val[3], "%d", lux);
    }

    if(mode != 1) board->display->display_string(val, 4);
    else {
    	for(i=0;i<40;i++){
    		board->display->display_string(val, 4);
    	}
    }
}

void lg_dist(int dist, XNucleo6180XA1 *board){
	int i,j;
	
	for(j=0;j<5;j++){
		for(i=0;i<interval;i++){
			disp_dist(dist, board);
		}
		for(i=0;i<interval;i++){
			wait_ms(10);
		}
	}

}

void lg_lux(int lux, XNucleo6180XA1 *board){
	int i,j;
	
	for(j=0;j<5;j++){
		for(i=0;i<interval;i++){
			disp_lux(lux, board);
		}
		for(i=0;i<interval;i++){
			wait_ms(10);
		}
	}

}
