#include "MjSi47xx.h"

MjSi47xx::MjSi47xx(I2C* i2c, uint8_t address, DigitalOut* rst_n)
{
    this->i2c = i2c;
    this->address = address;
    this->rst_n = rst_n;

    reset();
}

void MjSi47xx::reset()
{
    *rst_n = 0;
}

bool MjSi47xx::powerUp(FunctionType func)
{
    switch (func) {
        case FUNCTION_FM:
        case FUNCTION_AM:
            break;
        default:
            return false;
    }

    if (!*rst_n) {
        wait_us(250);
        *rst_n = 1;
        wait_us(1);
    }

    uint8_t arg[2];
    switch (func) {
        case FUNCTION_FM:
            arg[0] = 0x10;  // FM Receive
            arg[1] = 0x05;  // Analog Out
            break;
        case FUNCTION_AM:
            arg[0] = 0x11;  // AM/SW/LW Receive
            arg[1] = 0x05;  // Analog Out
            break;
    }

    writeCmd(0x01, arg, sizeof (arg));      // POWER_UP
    uint8_t status;
    readRes(&status);

    return true;
}

bool MjSi47xx::setProperty(PropertyType property, uint16_t value)
{
    const uint8_t arg[] = { 0x00, property / 0x100, property % 0x100, value / 0x100, value % 0x100, };

    writeCmd(0x12, arg, sizeof (arg));  // SET_PROPERTY
    uint8_t status;
    readRes(&status);

    return true;
}

bool MjSi47xx::getIntStatus(bool* stcInt)
{
    if (stcInt == NULL) return false;

    writeCmd(0x14); // GET_INT_STATUS
    uint8_t status;
    readRes(&status);

    if (stcInt != NULL) *stcInt = status & 0x01 ? true : false;

    return true;
}

bool MjSi47xx::fmTuneFrequency(float frequency)
{
    if (frequency < 64.0f || 108.0f < frequency) return false;

    int freqInt = (int)(frequency * 100.0f);
    const uint8_t arg[] = { 0x00, freqInt / 0x100, freqInt % 0x100, 0x00, };

    writeCmd(0x20, arg, sizeof (arg));  // FM_TUNE_FREQ
    uint8_t status;
    readRes(&status);

    return true;
}

bool MjSi47xx::fmTuneStatus(float* frequency, uint8_t* rssi, uint8_t* snr, uint8_t* multipath)
{
    const uint8_t arg[] = { 0x01, };

    writeCmd(0x22, arg, sizeof (arg));  // FM_TUNE_STATUS
    uint8_t status;
    uint8_t resp[7];
    readRes(&status, resp, sizeof (resp));

    if (frequency != NULL) *frequency = (float)((int)resp[1] * 0x100 + (int)resp[2]) / 100.0f;
    if (rssi != NULL) *rssi = resp[3];
    if (snr != NULL) *snr = resp[4];
    if (multipath != NULL) *multipath = resp[5];

    return true;
}

bool MjSi47xx::amTuneFrequency(float frequency)
{
    if (frequency < 149.0f || 23000.0f < frequency) return false;

    int freqInt = (int)frequency;
    const uint8_t arg[] = { 0x00, freqInt / 0x100, freqInt % 0x100, 0x00, 0x00, };

    writeCmd(0x40, arg, sizeof (arg));  // AM_TUNE_FREQ
    uint8_t status;
    readRes(&status);

    return true;
}

bool MjSi47xx::amTuneStatus(float* frequency, uint8_t* rssi, uint8_t* snr)
{
    const uint8_t arg[] = { 0x01, };
    writeCmd(0x42, arg, sizeof (arg));  // AM_TUNE_STATUS
    uint8_t status;
    uint8_t resp[7];
    readRes(&status, resp, sizeof (resp));

    if (frequency != NULL) *frequency = (float)((int)resp[1] * 0x100 + (int)resp[2]);
    if (rssi != NULL) *rssi = resp[3];
    if (snr != NULL) *snr = resp[4];

    return true;
}

void MjSi47xx::writeCmd(uint8_t cmd, const uint8_t* argv, size_t argc)
{
    i2c->start();
    i2c->write(address);
    i2c->write(cmd);
    while (argc--) {
        i2c->write(*argv++);
    }
    i2c->stop();
}

void MjSi47xx::readRes(uint8_t* status, uint8_t *resv, size_t resc)
{
    // Wait for CTS. (command has been accepted.)
    do {
        i2c->start();
        i2c->write(address | 1);
        *status = i2c->read(0);
        i2c->stop();
    } while (!(*status & 0x80));

    // ERR?
    if (*status & 0x40) {
        if (resv != NULL && resc >= 1) {
            memset(resv, 0, resc);
        }
        return;
    }

    // Receive response.
    i2c->start();
    i2c->write(address | 1);
    *status = i2c->read(resc >= 1 ? 1 : 0);
    while (resc--) {
        *resv++ = i2c->read(resc >= 1 ? 1 : 0);
    }
    i2c->stop();
}


