/* mbed PAW_Sensor Library
 *
 * paw.h
 *
 * Copyright (c) 2014 Hiroaki Matsuda
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef PAW_H
#define PAW_H

#include "mbed.h"

#define STATE_1 0
#define STATE_2 1
#define STATE_3 2
#define STATE_4 3


/** PAW_Sensor Library
 *
 * Example:
 * @code
 * 
#include "mbed.h"
#include "paw.h"

Serial pc( USBTX, USBRX ); 
const unsigned long baudrate = 115200;

// paw( LED1, LED2, PHOTO1, PHOTO2 )
PAW paw( p8, p9, p16, p17 );
paw_value g_value;

PAW paw_2( p10, p11, p18, p19 );
paw_value g_value_2;

Ticker run;


void run_paw_sensor()
{
    if( paw.process_paw() == STATE_1 )
    {
        paw.print( &pc );   
    }
}

int main() 
{
    // Initializing Serial Communication
    pc.baud( baudrate );
    pc.format( 8, Serial::None, 1 );
    
    run.attach_us(&run_paw_sensor, 500);
    
    while(1);
}

 * @endcode
 */

struct paw_value
{
    public: 
        short ch_1;
        short ch_2;
        short ch_3;
        short ch_4;
        short initial_photo_1;
        short initial_photo_2;
        
        paw_value()
            {
                ch_1 = 0;
                ch_2 = 0;
                ch_3 = 0;
                ch_4 = 0;
                initial_photo_1 = 0;
                initial_photo_2 = 0;
            }
};

class PAW {
    
    public:
    
        /** Create a PAW Sensor instance
         *
         * @param led_1 PAW Sensor's Pin 5
         * @param led_2 PAW Sensor's Pin 4
         * @param photo_1 PAW Sensor's Pin 3
         * @param photo_2 PAW Sensor's Pin 2
         */ 
        PAW( PinName led_1, PinName led_2, PinName photo_1, PinName photo_2 );
        
        /** Get values of PAW Sensor
         *
         * @return Return values of photosensor's voltage.
         */ 
        paw_value get_value();
        
        /** Processing of Paw sensor. This fucntion must be performed periodically.
         *
         * @return Return current processing-state.
         */ 
        unsigned char process_paw();
        
        /** Get current processing-state.
         *
         * @return Return current processing-state.
         */ 
        bool get_state();
        
        /** Send values of PAW Sensor through serial-communication.
         *
         * @param Reference to serial object.
         */ 
        void print( Serial* );
        
    protected:
        DigitalOut _led_1;
        DigitalOut _led_2;
        AnalogIn   _photo_1;
        AnalogIn   _photo_2;
        
        unsigned char _state;
        
        paw_value _value;
};

#endif /*** PAW_H ***/