#ifndef RPC_H
#define RPC_H

#include "mbed.h"

namespace physcom {
    
#define RPC_MAX_STRING 256

/** 
 * The RPC class used to parse and call RPCFunctions
 * This class should not be instantiated. It is the base class of RPCFunction.
 * The only relevant use of this class is to call the static function RPC::call.
 *
 * A serial connection must be opened on the USB port of the mbed: 
 * @code
 * Serial pc(USBTX, USBRX);
 * @endcode
 *
 * In order not to block the execution of the program while awaiting input on the
 * serial connection, a callback function can be defined. This callback should
 * execute a call to the static function RPC::call(char* input, char* output).
 * An example callback function:
 * @code
 * void RPCSerial() {
 *    pc.gets(input, RPC_MAX_STRING);
 *    RPC::call(input, output); 
 *    pc.printf("%s\n", output);
 * }
 * @endcode
 *
 * This function can be attached as a callback and be executed each time new
 * input is received on the serial connection:
 * @code
 * pc.attach(&RPCSerial, Serial::RxIrq);
 * @endcode
 *
 * RPCFunctions can be defined by passing a pointer to a function to the 
 * constructor of the RPCFunction call. The RPCFunction must have two parameters
 * of type char*, representing the input arguments and output result of the 
 * function. In case multiple values are used as arguments or results, they 
 * should be separated by spaces. The folowing example implements a function
 * which doubles the values of two integer arguments and returns the results
 * to the RPC caller:
 * @code
 * void double_numbers(char * input, char * output);
 * RPCFunction MyDoubleRPC(&double_numbers, "MyDoubleRPC");
 * 
 * void double_numbers(char * input, char * output){
 *    const char *DELIMITER = " ";
 *    char* first_string = strtok (input, DELIMITER);
 *    char* second_string = strtok (NULL, DELIMITER);
 *    int first_int = atoi(first_string);
 *    int second_int = atoi(second_string);
 *    sprintf(output, "%d %d", first_int*2, second_int*2);
 * }
 * @endcode
 *
 * An Example mbed program can be seeen below:
 * @code
 * #include "mbed.h"
 * #include "physcom.h"
 * 
 * using namespace physcom;
 * 
 * Serial pc(USBTX, USBRX);
 * 
 * // Define RPC functions
 * void read_value(char * input, char * output);
 * RPCFunction ReadValue(&read_value, "ReadValue");
 * void set_value(char * input, char * output);
 * RPCFunction SetValue(&set_value, "SetValue");
 * 
 * // mbed-local variables
 * DigitalOut myled(LED1);
 * int local_value = 50; 
 * 
 * // receive commands, and send back the responses
 * char input[RPC_MAX_STRING], output[RPC_MAX_STRING];
 * 
 * // Callback exectued whenever there is new input on the serial connection
 * void RPCSerial() {
 *     pc.gets(input, RPC_MAX_STRING);
 *     RPC::call(input, output); 
 *     pc.printf("%s\n", output);
 * }
 * 
 * int main() {
 *     
 *     // Attaching the callback to the Serial interface 
 *     // RPCSerial() will be executed whenever a serial interrupt is generated
 *     pc.attach(&RPCSerial, Serial::RxIrq);
 *     
 *     while(1) {
 *         myled = 1;
 *         wait(0.2);
 *         myled = 0;
 *         wait(0.2);
 *     }
 * }
 * 
 * void read_value(char * input, char * output){
 *     sprintf(output, "%d", local_value);
 * }
 *  
 * void set_value(char * input, char * output){
 *     local_value = atoi(input);
 *     output[0] = '\0'; // No result, so setting an empty string as output
 * }
 * @endcode
 *
 *
 *
 * An example MATLAB program that calls the RPCFunctions defined on the mbed in 
 * the previous example:
 * @code
 * import mbed.*
 *
 * % Open a serial connection to mbed on port COM11
 * mymbed = SerialRPC('COM11', 9600)
 * mymbed.reset();
 * pause(0.5);
 * 
 * ME = []; % exception vector for try-catch
 * try 
 *     % Attach to an Existing RPCFunction on mbed
 *     ReadValue = RPCFunction(mymbed, 'ReadValue');
 *     SetValue = RPCFunction(mymbed, 'SetValue');
 *    
 *     % Run its methods
 *     value = ReadValue.run(' ')
 *     SetValue.run('45')
 *     value = ReadValue.run(' ')
 * catch ME
 *     disp('Error:');
 *     disp(ME.message);
 * end
 *
 * % cleanup
 * mymbed.delete;
 * clear;
 * @endcode
 */
class RPC {

public:

    RPC(const char *name = NULL);

    virtual ~RPC();

    /**
    * Static method. Should be called everytime input is read on the serial
    * connection. RPC::call(input, output) will parse the input and call the
    * apropriate RPCFunction to compute the output and populate the result 
    * string.
    * @param Input string received from the serial connection
    * @param Output string rcomputed by executing the RPC function. Should be 
    * sent back to the caller over the serial connection. 
    */ 
    static bool call(char *input, char *output);

    /* Function lookup
     *  Lookup and return the object that has the given name.
     *
     * Variables
     *  name - the name to lookup.
     */
    static RPC *lookup(const char *name);
    
    virtual void run(char *arguments, char *result);

protected:
    static RPC *_head;
    RPC *_next;
    char *_name;

};

} // namespace physcom

#endif
