#ifndef PHYSCOM_OV_H
#define PHYSCOM_OV_H

#pragma once 
#include "mbed.h"
#include "ov7670reg.h"

#define OV7670_WRITE (0x42)
#define OV7670_READ  (0x43)
#define OV7670_WRITEWAIT (20)
#define OV7670_NOACK (0)
#define OV7670_REGMAX (201)
#define OV7670_I2CFREQ (100000) 

namespace physcom {

/**
 * Class used to controll the OV7670 camera. 
 * Using the constructor:
 * @code
 * OV7670 camera (
 *     p28,p27,            // SDA,SCL(I2C / SCCB)
 *     p23,NC,p25,         // VSYNC,HREF,WEN(FIFO)  
 *     Port0,0x07878000,   // PortIn data        p18(P0.26),p17(P0.25),p16(P0.24),p15(P0.23),p11(P0.18),p12(P0.17),p14(P0.16),p13(P0.15)
 *     p26,p29,p30         // RRST,OE,RCLK
 * ); 
 * @endcode
 * Before snapping a piture, the camera needs to be initialized. During the 
 * initialization, the format and resolution are set. Possible formats:
 * YUV422, RGB565. Possible resolutions: VGA (640 x 480), QVGA (320 x 240), 
 * QQVGA (160 x 120). See Init(char c, int n).
 */
class OV7670
{
    public:
            
        OV7670(
            PinName sda,    // Camera I2C port
            PinName scl,    // Camera I2C port
            PinName vs,     // VSYNC
            PinName hr,     // HREF
            PinName we,     // WEN
                      
            PortName port,  // 8bit bus port
            int mask,       // 0b0000_0M65_4000_0321_L000_0000_0000_0000 = 0x07878000
            
            PinName rt,     // /RRST
            PinName o,      // /OE
            PinName rc      // RCLK      
            );
        
        ~OV7670();
        
        void CaptureNext(void);              // capture request
        bool CaptureDone(void);              // capture done? (with clear)
        void WriteReg(int addr,int data);    // write to camera
        int ReadReg(int addr);               // read from camera
        void Reset(void);                    // reset reg camera
        int Init(char c, int n);             // Old init reg
        int Init(char *format, int n);       // init reg    
        void VsyncHandler(void);             // New vsync handler
        void HrefHandler(void);              // href handler
        int ReadOnebyte(void);               // Data Read
        void ReadStart(void);                // Data Start
        void ReadStop(void);                 // Data Stop
       
        
    private:
        I2C _i2c;
        InterruptIn vsync,href;
        
        DigitalOut wen;
        PortIn data;
        DigitalOut rrst,oe,rclk;
        volatile int LineCounter;
        volatile int LastLines;
        volatile bool CaptureReq;
        volatile bool Busy;
        volatile bool Done; 
        char *format_temp;
};

} // Namespace physcom

#endif