#ifndef M3PI_H
#define M3PI_H

#include "mbed.h"
#include "platform.h"

#ifdef MBED_RPC
#include "rpc.h"
#endif

#define SEND_RAW_SENSOR_VALUES 0x86
#define SEND_CALIBRATED_SENSOR_VALUES 0x87
#define SEND_BATTERY_MILLIVOLTS 0xB1
#define DO_PRINT 0xB8
#define M1_FORWARD 0xC1
#define M1_BACKWARD 0xC2
#define M2_FORWARD 0xC5
#define M2_BACKWARD 0xC6
#define AUTO_CALIBRATE 0xBA

namespace physcom {

/** m3pi control class
 *
 */
class M3pi :  public Stream {
    
private:
    float m1_speed;
    float m2_speed;

    // Public functions
public:

    /** Create the m3pi object connected to the default pins
     *
     * @param nrst GPIO pin used for reset. Default is p23
     * @param tx Serial transmit pin. Default is p9
     * @param rx Serial receive pin. Default is p10
     */
    M3pi();

    /** Create the m3pi object connected to specific pins
     *
     */
    M3pi(PinName nrst, PinName tx, PinName rx);

    /** Force a hardware reset of the 3pi
     */
    void reset (void);
    
    
    /** This function will smoothly stop the motors of the robot by gradually 
     * slowing down.
     * Contributed by Menno Vermeulen and Daniel Trujillo
     */
    void stop();
    
    /** Activate motor
     * @param motor. Values: 0: Left; 1: Right
     * @param speed Float value representing the speed of the motor. Negative 
     * values will turn the wheels in reverse. The possible values of this 
     * parameter range between -1 and 1. The values are normalized, so speed=1 
     * means the motor will activate at full (100%) power, while speed=0.2 would
     * activate the motor with 20% of its maximum power.
     * A good starting value can be speed=0.1. In most cases the speed value you 
     * will use should be between -0.5 and 0.5. 
     */
    void activate_motor (int motor, float speed);

    /** Reads the values of the active light sensors
     * @param sensors Array of 5 int elements that will be populated with
     * the sensor values
     */
    void raw_sensors (int sensors[5]);

    /** Reads the values of the active light sensors. The result values are 
     * normalized in the range 0-1000 based on the calibration. The function
     * sensor_auto_calibrate() should be called once at the start of the program
     * before reading the calibrated values. 
     *  
     * @param sensors Array of 5 int elements that will be populated with
     * the sensor values
     */
    void calibrated_sensors (int sensors[5]);

    /** Read the battery voltage on the 3pi
     * @returns battery voltage as a float
     */
    float battery(void);
    
    
    /** Calibrate the sensors. This turns the robot left then right, 
     * looking for a line. Make sure that when this function is called, the
     * robot is positioned on the black line on white background. 
     * The values of the calibration are storred internaly and used 
     * when computing the normalized result values of calibrated_sensors().
     */
    char sensor_auto_calibrate(void);

private :

    DigitalOut _nrst;
    Serial _ser;
    
    virtual int _putc(int c);
    virtual int _getc();

};


} // Namespace physcom

#endif
