// BaseUvc.cpp

#include "USBHostConf.h"
#include "USBHost.h"

#include "BaseUvc.h"

//#define ISO_DEBUG 1
#ifdef ISO_DEBUG
#define ISO_DBG(x, ...) std::printf("[%s:%d]"x"\r\n", __PRETTY_FUNCTION__, __LINE__, ##__VA_ARGS__);
#else
#define ISO_DBG(...)  while(0);
#endif

#define TEST_ASSERT(A) while(!(A)){fprintf(stderr,"\n\n%s@%d %s ASSERT!\n\n",__PRETTY_FUNCTION__,__LINE__,#A);exit(1);};

void BaseUvc::poll(int millisec)
{
    HCITD* itd = m_isoEp->isochronousReceive(millisec);
    if (itd) {
        uint8_t cc = itd->ConditionCode();
        report_cc_count[cc]++;
        if (cc == 0) {
            uint16_t frame = itd->StartingFrame();
            int fc = itd->FrameCount();
            uint8_t* buf = const_cast<uint8_t*>(itd->buf); 
            int mps = m_isoEp->m_PacketSize;
            for(int i = 0; i < fc; i++) {
                uint16_t psw = itd->OffsetPSW[i];
                cc = psw>>12;
                if (cc == 0 || cc == 9) {
                    int len = psw & 0x7ff;
                    onResult(frame, buf, len);
               }
               report_ps_cc_count[cc]++;
               buf += mps;
               frame++;
            }
        }
        delete itd;
    }
}

USB_TYPE BaseUvc::Control(int req, int cs, int index, uint8_t* buf, int size)
{
    if (req == SET_CUR) {    
        return host->controlWrite(dev,
                    USB_HOST_TO_DEVICE | USB_REQUEST_TYPE_CLASS | USB_RECIPIENT_INTERFACE, 
                    req, cs<<8, index, buf, size);
    }
    return host->controlRead(dev,
                USB_DEVICE_TO_HOST | USB_REQUEST_TYPE_CLASS | USB_RECIPIENT_INTERFACE, 
                req, cs<<8, index, buf, size);
}

USB_TYPE BaseUvc::setInterfaceAlternate(uint8_t intf, uint8_t alt)
{
    return host->controlWrite(dev, USB_HOST_TO_DEVICE | USB_RECIPIENT_INTERFACE,
                                   SET_INTERFACE, alt, intf, NULL, 0);
}

void BaseUvc::onResult(uint16_t frame, uint8_t* buf, int len)
{
  if(m_pCbItem && m_pCbMeth)
    (m_pCbItem->*m_pCbMeth)(frame, buf, len);
  else if(m_pCb)
    m_pCb(frame, buf, len);
}

void BaseUvc::setOnResult( void (*pMethod)(uint16_t, uint8_t*, int) )
{
    m_pCb = pMethod;
    m_pCbItem = NULL;
    m_pCbMeth = NULL;
}
    
void BaseUvc::clearOnResult()
{
    m_pCb = NULL;
    m_pCbItem = NULL;
    m_pCbMeth = NULL;
}


HCITD::HCITD(BaseEp* obj, uint16_t FrameNumber, int FrameCount, uint16_t PacketSize) {
    Control = 0xe0000000           | // CC ConditionCode NOT ACCESSED
             ((FrameCount-1) << 24)| // FC FrameCount
                  TD_DELAY_INT(0)  | // DI DelayInterrupt
                 FrameNumber;        // SF StartingFrame
    BufferPage0 = const_cast<uint8_t*>(buf);
    BufferEnd = const_cast<uint8_t*>(buf) + PacketSize * FrameCount - 1;
    Next = NULL; 
    ep = obj;
    uint32_t addr = reinterpret_cast<uint32_t>(buf);
    for(int i = 0; i < FrameCount; i++) {
        uint16_t offset = addr & 0x0fff;
        if ((addr&0xfffff000) == (reinterpret_cast<uint32_t>(BufferEnd)&0xfffff000)) {
            offset |= 0x1000;
        }
        OffsetPSW[i] = 0xe000|offset;
        addr += PacketSize;
    }
}

IsochronousEp::IsochronousEp(int addr, uint8_t ep, uint16_t size):BaseEp(addr, ep, size)
{
    ISO_DBG("%p FA:%d EP:%02X MPS:%d\n", this, addr, ep, size);
    TEST_ASSERT(m_pED);
    
    m_pED->Control |= (1 << 15); // F Format ITD

    TEST_ASSERT(size >= 128 && size <= 1023);
    m_PacketSize = size;
    m_FrameCount = 4; // 1-8
    TEST_ASSERT(m_FrameCount >= 1 && m_FrameCount <= 8);
    m_itd_queue_count = 0;
    reset();
    HCITD* itd = new_HCITD(this);
    m_pED->TailTd = reinterpret_cast<HCTD*>(itd);
    m_pED->HeadTd = reinterpret_cast<HCTD*>(itd); 
    TEST_ASSERT(itd);
    if (itd == NULL) {
        return;
    }
    _HCCA* hcca = reinterpret_cast<_HCCA*>(LPC_USB->HcHCCA);
    TEST_ASSERT(hcca);
    if (hcca == NULL) {
        return;
    }
    hcca->enqueue(m_pED);
}

void IsochronousEp::reset(int delay_ms)
{
    m_FrameNumber = LPC_USB->HcFmNumber + delay_ms;
}

HCITD* IsochronousEp::new_HCITD(BaseEp* obj)
{
    HCITD* itd = new(m_PacketSize*m_FrameCount)HCITD(obj, m_FrameNumber, m_FrameCount, m_PacketSize);
    if (itd == NULL) {
        return NULL;
    } 
    m_FrameNumber += m_FrameCount;
    return itd;
}

HCITD* IsochronousEp::isochronousReceive(int millisec)
{
    TEST_ASSERT(m_itd_queue_count >= 0);
    while(m_itd_queue_count < 3 && m_itd_queue_count < HCTD_QUEUE_SIZE) {
        HCITD* itd = reinterpret_cast<HCITD*>(m_pED->TailTd);
        TEST_ASSERT(itd);
        if (itd == NULL) {
            return NULL;
        }
        HCITD* blank_itd = new_HCITD(this);
        TEST_ASSERT(blank_itd);
        if (blank_itd == NULL) {
            return NULL;
        }
        itd->Next = blank_itd;
        m_pED->TailTd = reinterpret_cast<HCTD*>(blank_itd);
        m_itd_queue_count++;
        //DBG_IED(m_pED);
        enable(); // Enable Periodic
    }
    
    HCITD* itd = get_queue_HCITD(millisec);
    if (itd) {
        m_itd_queue_count--;
    }
    return itd;
}

HCITD* IsochronousEp::get_queue_HCITD(int millisec)
{
    for(int i = 0; i < 16; i++) {
        osEvent evt = m_queue.get(millisec);
        if (evt.status == osEventMessage) {
            HCITD* itd = reinterpret_cast<HCITD*>(evt.value.p);
            TEST_ASSERT(itd);
            return itd;
        } else if (evt.status == osOK) {
            continue;
        } else if (evt.status == osEventTimeout) {
            return NULL;
        } else {
            ISO_DBG("evt.status: %02x\n", evt.status);
            TEST_ASSERT(evt.status == osEventMessage);
            return NULL;
        }
    }
    return NULL;
}

void IsochronousEp::enable()
{
    LPC_USB->HcControl |= OR_CONTROL_PLE;
}

void IsochronousEp::disconnect()
{
    m_pED->setSkip();
    ISO_DBG("m_itd_queue_count: %d", m_itd_queue_count);
    Timer t;
    t.reset();
    t.start();
    while(m_itd_queue_count > 0 && t.read_ms() <= (8*3)) {
        HCITD* itd = get_queue_HCITD(0);
        if (itd) {
            ISO_DBG("ITD: %p", itd);
            delete itd;
            m_itd_queue_count--;
            t.reset();
        }
    }
    ISO_DBG("m_itd_queue_count: %d, t_ms: %d", m_itd_queue_count, t.read_ms());
    HCITD* head = reinterpret_cast<HCITD*>(reinterpret_cast<uint32_t>(m_pED->HeadTd)&~3); // delete Halted and Toggle Carry bit
    TEST_ASSERT(head);
    HCITD* tail = reinterpret_cast<HCITD*>(m_pED->TailTd);
    TEST_ASSERT(tail);
    while(head != tail) {
        HCITD* next = head->Next;
        TEST_ASSERT(next);
        ISO_DBG("ED ITD:%p next:%p", head, next);        
        delete head;
        TEST_ASSERT(m_itd_queue_count > 0);
        m_itd_queue_count--;
        head = next;
    }
    TEST_ASSERT(m_itd_queue_count == 0);
    delete head;
    
    _HCCA* hcca = reinterpret_cast<_HCCA*>(LPC_USB->HcHCCA);
    TEST_ASSERT(hcca);
    hcca->dequeue(m_pED);
    delete m_pED;
}

BaseEp::BaseEp(int addr, uint8_t ep, uint16_t size, int lowSpeed):m_td_queue_count(0)
{
    ISO_DBG("%p FA=%d EN=%02x MPS=%d S=%d\n", this, addr, ep, size, lowSpeed);
    TEST_ASSERT(size >= 8 && size <= 1023);    
    TEST_ASSERT(lowSpeed == 0 || lowSpeed == 1);
    m_pED  = new _HCED(addr, ep, size, lowSpeed);
    TEST_ASSERT(m_pED);
}
