#ifndef AK7451_H
#define AK7451_H

#include "mbed.h"

/**
 * This is a device driver of AK7451 with SPI interface.
 *
 * @note AK7451 is a high speed angle sensor IC manufactured by AKM.
 * 
 * Example:
 * @code
 * #include "mbed.h"
 * #include "ak7451.h"
 * 
 * 
 * int main() {
 *     // Creates an instance of SPI
 * }
 * @endcode
 */
class AK7451
{
public:

    /**
     * Available opration modes in AK7451.
     */
    typedef enum {
        AK7451_NORMAL_MODE  = 0x0000,   /**< Normal mode operation. */
        AK7451_USER_MODE    = 0x050F,   /**< User mode operation. */
    } OperationMode;

    /**
     * Status of function. 
     */
    typedef enum {
        SUCCESS,                    /**< The function processed successfully. */
        ERROR,                      /**< General Error */
        ERROR_IN_USER_MODE,         /**< Error in user mode. */
        ERROR_IN_NORMAL_MODE,       /**< Error in normal mode. */
        ERROR_PARITY,               /**< Parity bit error. */
        ERROR_ABNORMAL_STRENGTH,    /**< Abnormal strength error. */
    } Status;

    /**
     * Constructor.
     *
     */
    AK7451();

    /**
     * Destructor.
     *
     */
     ~AK7451();

    /**
     * begin
     *
     * @param *spi pointer to SPI instance
     * @param *cs pointer to DigitalOut instance for CS
     */
    void begin(SPI *spi, DigitalOut *cs);
    
    /**
     * Check the connection. 
     *
     * @note Connection check is performed by reading a register which has a fixed value and verify it.
     *
     * @return Returns SUCCESS when succeeded, otherwise returns another code.
     */
    Status checkConnection();

    /** 
     * Writes data to EEPROM on the device. 
     * @param address EEPROM address
     * @param data data to be written
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status writeEEPROM(char address, const char *data);

    /** 
     *  Reads data from EEPROM on the device. 
     * @param address EEPROM address
     * @param data data to read
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status readEEPROM(char address, char *data);

    /** 
     * Writes data to register on the device. 
     * @param address register address
     * @param data data to be written
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status writeRegister(char address, const char *data);

    /** 
     *  Reads data from register on the device. 
     * @param address register address
     * @param data data to read
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status readRegister(char address, char *data);

    /**
     * Sets device operation mode.
     *
     * @param mode device opration mode
     *
     * @return Returns SUCCESS when succeeded, otherwise returns another code.
     */
    Status setOperationMode(OperationMode mode);

    /**
     * Gets device operation mode.
     *
     * @return Returns OperationMode.
     */
    OperationMode getOperationMode();

    /**
     * Reads angle data from the device.
     *
     * @param angle pointer to read angle data buffer
     *
     * @return Returns SUCCESS when succeeded, otherwise returns another code.
     */
    Status readAngle(char *angle);
    
    /**
     * Measures and reads angle, magnetic flux density and abnormal state code while in the user mode.
     *
     * @param angle pointer to angle data buffer
     * @param density magnetic flux density
     * @param abnormal_state abnormal state
     *
     * @return Returns SUCCESS when succeeded, otherwise returns another code.
     */
    Status readAngleMeasureCommand(char *angle, char *density, char *abnormal_state);
    
    /**
     * Measures current angle and sets the value to EEPROM as zero angle position.
     *
     * @return Returns SUCCESS when succeeded, otherwise returns another code.
     */
    Status setAngleZero();
    
    /**
     * Sets the value to EEPROM as zero angle position.
     *
     * @param angle zero angle position
     *
     * @return Returns SUCCESS when succeeded, otherwise returns another code.
     */
    Status setAngleZero(const char *angle);

private:        
    /**
     * Holds a pointer to an SPI object. 
     */
    SPI *_spi;

    /**
     * Holds a DigitalOut oblject for CS; 
     */
    DigitalOut *_cs;
    
    /**
     * Holds current mode 
     */
    OperationMode operationMode;

    /** 
     *  Reads data from device. 
     * @param operation_code OPCODE 
     * @param address memory/register addredd
     * @param *data pointer to the read buffer. length=2 fixed.
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status read(char operation_code, char address, char *data);

    /** 
     * Writes data to the device. 
     * @param operation_code OPCODE 
     * @param address memory/register addredd
     * @param *data pointer to the read buffer. length=2 fixed.
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status write(char operation_code, char address, const char *data);

    /** 
     * Checks parity bits sub function
     * @param data data 12bit read data
     * @param parity parity bit status
     * @param error error bit status
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status parityCheckSub(const char data, const char parity, const char error);

    /** 
     * Checks parity bits
     * @param data 2 byte read data to chek parity
     * @return Returns SUCCESS when succeeded, otherwise returns another.
     */
    Status parityCheck(const char *data);
};

#endif