/**
  ******************************************************************************
  * @file    main.cpp 
  * @author  KTE makabe
  * @version V1.00
  * @date    2016/05/26
  * @brief   UC MIRAI program body
  ******************************************************************************
  *
  ******************************************************************************  
  */ 

/* Includes ------------------------------------------------------------------*/
#include "mbed.h"

#define VALID_PWM // PWM有効

/* Private typedef -----------------------------------------------------------*/
/*状態*/
typedef enum {
    Idle = 0,
    MoveL,
    MoveR,
    Origin,
    Error,
} State_Enum;

/* Private define ------------------------------------------------------------*/
#define STOP()     {FWD=0; RVS=0;}
#define FORWARD()  {FWD=MVR; RVS=0;}
#define REVERSE()  {FWD=0; RVS=MVR;}
#define BRAKE()    {FWD=1; RVS=1;}

#define IS_TIMEOUT()   (g.cnt[0] <= 0)
#define IS_OVER()      (OL == 1) /*過負荷*/
#define IS_ERROR()     (IS_TIMEOUT() || IS_OVER())

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/*ピン定義*/
DigitalOut db0(dp9);
DigitalOut db1(dp10);
DigitalOut db2(dp11);

AnalogIn MVR(dp13); // モータ速度Input
DigitalIn OL(dp14); // 過負荷検知(H)
PwmOut FWD(dp1/*dp15*/); // 正転
PwmOut RVS(dp2/*dp16*/); // 逆転
DigitalIn LMT_L(dp4/*dp24*/); // リミットL(L)
DigitalIn LMT_R(dp25); // リミットR(L)
DigitalOut RED(dp26); // 赤LED(H)
//DigitalIn TRIG(dp27); // トリガSW(H)
InterruptIn TRIG(dp27);
DigitalOut GRN(dp28); // 緑LED(H)

Ticker tick;

struct {
    int trig;
    State_Enum state;
    State_Enum lastState;
    int lastTrig; // 前回のトリガ値
    int cnt[4];
} g;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/
void DBG(uint8_t n)
{
    db0 = (n & 0x01);
    db1 = (n & 0x02) >> 1;
    db2 = (n & 0x04) >> 2;
}
void GoState(State_Enum state)
{
    g.lastState = g.state;
    g.state = state;
    
    if(state == Error){
        g.cnt[0] = 0;
        g.cnt[1] = 0;
        g.cnt[2] = 25; // 250msec RED timer
        GRN = 0;
    }
    if(state == MoveL || state == MoveR || state == Origin){
        g.cnt[0] = 500; // 3sec timer start
        g.cnt[1] = 25; // 250msec GRN timer
        g.cnt[2] = 0;
    }
    if(state == Idle){
        g.cnt[0] = 0;
        g.cnt[1] = 0;
        g.cnt[2] = 0;
        GRN = 1;
        RED = 0;
    }
}
/*10msec handler*/
void ticker_handler(void)
{
    // TIMEOUT
    if(g.cnt[0] > 0 && --g.cnt[0] <= 0) {
    }
    // GREEN LED
    if(g.cnt[1] > 0 && --g.cnt[1] <= 0) {
        GRN = !GRN; // モータ動作中
        g.cnt[1] = 25;
    }
    // RED LED
    if(g.cnt[2] > 0 && --g.cnt[2] <= 0) {
        RED = !RED; // エラー点滅
        g.cnt[2] = 25;
    }
}
/*TRIG Interrupt*/
void trig_handler(void)
{
    g.trig = 1;
}
/* Public functions ----------------------------------------------------------*/
/*----------------------------------------------------------------------------
UC MIRAI PROGRAM             by makabe
仕様
・トリガSWにてモーターを動かす。
・リミット到達したらモーターを停める。
・モーターは片方向動作。つまり[L->R]の次は[R->L]。これの繰り返し。
・動作異常時は赤LED点滅。（過負荷異常、タイムアウト）
・通常モーターはLかRどちらかに位置する。つまり一方のリミットセンサーが働いている状態。
・したがってモータが中途半端な位置にあった場合は原点復帰させる必要あり。
-----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------
LED仕様
・RED消灯：電源OFF
・RED点灯：
・RED点滅：エラー時
・GRN消灯：電源OFF
・GRN点灯：正常動作時
・GRN点滅：モータ移動時
-----------------------------------------------------------------------------*/
/*int main() {
    //DBG(0);
    while(1) {
        RED = LMT_R;
        GRN = LMT_L;
    }
}*/
#if 1
int main() 
{
    memset(&g, 0, sizeof(g));
    tick.attach(&ticker_handler, 0.01);
    TRIG.mode(PullDown);
    TRIG.rise(&trig_handler);
    LMT_L.mode(PullDown);
    LMT_R.mode(PullDown);
    OL.mode(PullDown);
    FWD.period(0.001); // PWM period 1ms (1kHz)
    RVS.period(0.001);
    RED = 0;
    GRN = 1;
    STOP();
    
    g.lastTrig = 1;
    GoState(Idle);
    while(1) {
        switch(g.state) {
        case Idle:
DBG(0);
            // トリガがあればモータを動かす
            // ただし、「トリガONで電源ON」の場合は動かしたくない。
            if(g.trig == 1/*g.lastTrig == 0 && TRIG == 1*/){
                g.trig = 0;
                if(LMT_L == 1 && LMT_R == 0) {
                    // モーター右にあれば左へ動かす
L001:
                    FORWARD();
                    GoState(MoveL);
                } else if(LMT_L == 0 && LMT_R == 1) {
                    // モーター左にあれば右へ動かす
                    REVERSE();
                    GoState(MoveR);
                } else if(LMT_L == 1 && LMT_R == 1) {
                    // モータ中途半端な位置にある場合、まず左原点復帰。その後右移動
                    REVERSE();
                    GoState(Origin);
                } else {
                    GoState(Error);
                }
            }
            break;
        case MoveL:
DBG(6);
            // 左移動中リミット到達したら停める
            if(LMT_L == 0){
                BRAKE();
                wait(0.5);
                STOP();
                GoState(Idle);
            } else if(IS_ERROR()){
                STOP();
                GoState(Error);
            }
            break;
        case MoveR:
DBG(5);/*ok*/
            // 右移動中リミット到達したら停める
            if(LMT_R == 0){
                BRAKE();
                wait(0.5);
                STOP();
                GoState(Idle);
            } else if(IS_ERROR()){
                STOP();
                GoState(Error);
            }
            break;
        case Origin:
DBG(4);
            // 原点復帰終えたら直ちに右移動
            if(LMT_R == 0){
                BRAKE();
                wait(0.5);
                STOP();
                goto L001;
            } else if(IS_ERROR()){
                BRAKE();
                wait(0.5);
                STOP();
                GoState(Error);
            }
            break;
        case Error:
DBG(3);
            // エラーに遷移するともう戻れない。赤LED点滅
            break;    
        default:
            break;
        }//end_switch        
        //g.lastTrig = TRIG;
    }//end_while
}
#endif
