/* 
 * BLE Current Time Service (subset)
 *
 * by ohneta/ Oct. 2015
 */
#ifndef __BLE_CURRENT_TIME_SERVICE_H__
#define __BLE_CURRENT_TIME_SERVICE_H__
 
#include "ble/BLE.h"
#include <time.h>
#include "ble_date_time.h"
#include "common.h"
//extern Serial  pc;
 
 
enum BLE_DayofWeek {
    notknown = 0,
    Monday = 1,
    Tuesday,
    Wednesday,
    Thursday,
    Friday,
    Saturday,
    Sunday
};
 
typedef struct : ble_date_time_t/*BLE_DateTime*/ {
    BLE_DayofWeek   dayOfWeek;
} BLE_DayDateTime;
 
typedef struct BLE_ExactTime256 : BLE_DayDateTime {
    uint8_t fractions256;
} BLE_ExactTime256;
 
typedef struct BLE_CurrentTime : BLE_ExactTime256 {
    uint8_t adjustReason;
} BLE_CurrentTime;
 
#define     BLE_CURRENT_TIME_CHAR_VALUE_SIZE      10
 
/**
 *
 */
class CurrentTimeService {
 
protected:
    Ticker  ticker; 
 
    /**
     * ticker callback.
     * interval = 1sec
     */
    void epochtimePeriodicCallback(void)
    {
        time_t tmpEpochTime = epochTimeByDateTimeBuffer();
        tmpEpochTime++;
        dataTimeBufferByEpochTime(&tmpEpochTime);
        
        counter++;
        tmCnt--;
    }
 
    void dataTimeBufferByEpochTime(time_t *epochTime)
    {
        struct tm *tmPtr = localtime(epochTime);
 
        *(uint16_t *)&valueBytes[0] = tmPtr->tm_year + 1900;
        valueBytes[2] = tmPtr->tm_mon + 1;
        valueBytes[3] = tmPtr->tm_mday;
        valueBytes[4] = tmPtr->tm_hour;
        valueBytes[5] = tmPtr->tm_min;
        valueBytes[6] = tmPtr->tm_sec;
        valueBytes[7] = (BLE_DayofWeek)((tmPtr->tm_wday == 0) ? 7 : tmPtr->tm_wday);
        valueBytes[8] = 0x00;
        valueBytes[9] = 0x00;
//#ifdef UART_DEBUG
//        pc.printf("WRITE ==> %d/%d/%d %02d:%02d:%02d \r\n", tmPtr->tm_year, tmPtr->tm_mon, tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min, tmPtr->tm_sec);
//#endif

        ble.gattServer().write(currentTimeCharacteristic.getValueHandle(), valueBytes, BLE_CURRENT_TIME_CHAR_VALUE_SIZE);
    }
    
    time_t epochTimeByDateTimeBuffer()
    {
        struct tm  timep;
        {
            timep.tm_year  = *(uint16_t *)&valueBytes[0] - 1900;
            timep.tm_mon   = valueBytes[2] - 1;
            timep.tm_mday  = valueBytes[3];
            timep.tm_hour  = valueBytes[4];
            timep.tm_min   = valueBytes[5];
            timep.tm_sec   = valueBytes[6];
            timep.tm_isdst = 0;
        }
//#ifdef UART_DEBUG
//        pc.printf("READ ==> %d/%d/%d %02d:%02d:%02d \r\n", timep.tm_year, timep.tm_mon, timep.tm_mday, timep.tm_hour, timep.tm_min, timep.tm_sec);
//#endif
        time_t epochTime = mktime(&timep);
    
        return epochTime;
    }
 
public:
    //------------------------------------------------------------------------------------
    /**
     *
     */
    CurrentTimeService(BLE &_ble, ble_date_time_t &initialDateTime) :
        ble(_ble),
        currentTimeCharacteristic(  GattCharacteristic::UUID_CURRENT_TIME_CHAR,
                                    valueBytes, BLE_CURRENT_TIME_CHAR_VALUE_SIZE, BLE_CURRENT_TIME_CHAR_VALUE_SIZE,
                                    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ
                                    | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY
                                    | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE
                                )                       
    {
        writeDateTime(initialDateTime);
        ticker.attach(this, &CurrentTimeService::epochtimePeriodicCallback, 1.0);
 
        GattCharacteristic *charsTable[] = {&currentTimeCharacteristic};
        GattService  currentTimeService(GattService::UUID_CURRENT_TIME_SERVICE, charsTable, sizeof(charsTable) / sizeof(GattCharacteristic *)   );
 
        ble.addService(currentTimeService);
        ble.onDataWritten(this, &CurrentTimeService::onDataWritten);
        
        counter = 0;
    }
 
    /**
     */
    void writeDateTime(ble_date_time_t &dateTime)
    {
        *(uint16_t *)&valueBytes[0] = dateTime.year;
        valueBytes[2] = dateTime.month;
        valueBytes[3] = dateTime.day;
        valueBytes[4] = dateTime.hours;
        valueBytes[5] = dateTime.minutes;
        valueBytes[6] = dateTime.seconds;
 
        // not support
        valueBytes[7] = 0x00;   // day of week
        valueBytes[8] = 0x00;   // Fractions256
        valueBytes[9] = 0x00;   // Adjust Reason
    }
 
    void writeEpochTime(time_t et)
    {
        dataTimeBufferByEpochTime(&et);
    }

    /**
     */
    void readDateTime(ble_date_time_t &dateTime)
    {
        dateTime.year     = *(uint16_t *)&valueBytes[0];
        dateTime.month    = valueBytes[2];
        dateTime.day      = valueBytes[3];
        dateTime.hours    = valueBytes[4];
        dateTime.minutes  = valueBytes[5];
        dateTime.seconds  = valueBytes[6];
    }
 
    time_t readEpochTime()
    {
        return epochTimeByDateTimeBuffer();
    }
 
 
    // for BLE GATT callback (optional)
    virtual void onDataWritten(const GattWriteCallbackParams *params)
    {
        if (params->handle == currentTimeCharacteristic.getValueHandle()) {
            memcpy((void *)&valueBytes, params->data, params->len);
        }
    }
 
    int getCounter() const { return counter; }
    void setCounter(int val) { counter = val; }
    int getTm() const { return tmCnt; }
    void setTm(int val) { tmCnt = val; }
    
protected:
    BLE &ble;
    uint8_t   valueBytes[BLE_CURRENT_TIME_CHAR_VALUE_SIZE];    
    GattCharacteristic  currentTimeCharacteristic;
private:
    int counter; /*timer counter*/
    int tmCnt; /*for timeout*/
};
 
#endif /* #ifndef __BLE_CURRENT_TIME_SERVICE_H__*/
 
