#include "mbed.h"
 
#define MCP24AA02_ADDR     (0xA0) // 24AA02 2K EEPROM using i2c Address
                                  // for the 22AA025 variant this can be 0xA0,0xA2,0xA4,0xA6,0xA8,0xAA,0xAC,0xAE
                                  // since LSB is used to to control a WRITE(0) or a READ(1) Operation
#define WRITE              (0x00) // 24AA02 2K EEPROM using i2c Write bit
#define READ               (0x01) // 24AA02 2K EEPROM using i2c Read bit
#define MCP24AA02_MID      (0xFA) // Manufacturer ID Address (Read Only 0x29==Microchip)
#define MCP24AA02_DID      (0xFB) // Device ID Adress (Read Only 0x41==4 is i2c family and 1 is 2K device)
                                  // Information from Microchip Datasheet DS20005202A Part #24AA02UID/24AA025UID
/* This is a very simple program to demonstrate the ease of use of the i2c bus utilising an EEPROM.
   Do not forget that Pull Up resistors are required on the SDA and SCL lines
   (Typically 2K ohms for 400KHz bus speeds)
   These are required since devices connected with the i2c will have 'open drain' (or open collector)
   circuitry to allow wire 'ORing' of their respective connections
   Used with a stm32-Nucleo-F401RE
*/

I2C i2c(I2C_SDA, I2C_SCL); //I2C Class Pin Assignments see I2C.h
 
Serial pc(SERIAL_TX, SERIAL_RX); //Serial Class Pin Assignments see Serial.h

void readID(int addy)
{
    char ucdata_write[1];
    char ucdata_read[1];
    ucdata_write[0] = addy; //MCP24AA02_DID, Address here for Device ID
    
    while (i2c.write((MCP24AA02_ADDR|WRITE), ucdata_write, 1, 0)){} //Wait for ACK if EEPROM is in 'write' cycle
    
    {
        i2c.read((MCP24AA02_ADDR|READ),ucdata_read,1,0);
        if (ucdata_read[0]==0x29)
        {
            pc.printf("Code=%#x (Microchip Technology ltd.(c))\n\r",ucdata_read[0]);return;
        }
        if (ucdata_read[0]==0x41)
        {
            pc.printf("Code=%#x (2K EEPROM Using i2c)\n\r",ucdata_read[0]);
        }
        else
        {
            pc.printf("0x%x 0x%X %c\n\r",addy,ucdata_read[0],(ucdata_read[0]));
        }    
     }
}

void readEE(int addy)
{
    char ucdata_write[1];
    char ucdata_read[1];
    ucdata_write[0] = addy; //Address here to read
    while (i2c.write((MCP24AA02_ADDR|WRITE), ucdata_write, 1, 0)){} //Wait for ACK if EEPROM is in 'write' cycle
    i2c.read((MCP24AA02_ADDR|READ),ucdata_read,1,0); //Note 'OR' Address with Read bit
    pc.printf("%#X %c",ucdata_read[0],ucdata_read[0]);
 }

void writeEE(int addy,int data)
{
    char ucdata_write[2];
    ucdata_write[0] = addy;
    ucdata_write[1] = data;
    while (i2c.write((MCP24AA02_ADDR|WRITE), ucdata_write, 1, 0)){} //Wait for ACK if EEPROM is in 'write' cycle
    i2c.write((MCP24AA02_ADDR|WRITE),ucdata_write,2,0); //Note 'OR' Adress with Write bit
    pc.printf("%04d %c\t",addy,data);
}

void eraseEE(void)
{
    for (int i=0;i<0xFA;i++) //0xFA to 0xFF are read only with Manufacture/Hardware ID and a Unique Serial Number
    {
        writeEE(i,0xFF);
    }
}

int main()
{
    unsigned int uibaudrate=115200;
    pc.baud(uibaudrate);
    
    unsigned int uifrequency=400000; //400KHz for i2c Max
    i2c.frequency (uifrequency);

    char ucdata_write[1];    
    if (!i2c.write((MCP24AA02_ADDR|WRITE), ucdata_write, 1, 0))// Check for ACK from EPPROM
    {
        pc.printf("\n\rHello World ");
        pc.printf("at %u BAUD and %uKHz i2c Frequency\n\r",uibaudrate,uifrequency/1000);
        pc.printf("Using mbed.org Martin\n\r");
        readID(MCP24AA02_MID);
        readID(MCP24AA02_DID);
        //  Uncomment the following 6 lines of code to write Data into the EEPROM
        /*
        writeEE(0x00,0x4D); // ASCII M
        writeEE(0x01,0x61); // ASCII a
        writeEE(0x02,0x72); // ASCII r
        writeEE(0x03,0x74); // ASCII t
        writeEE(0x04,0x69); // ASCII i
        writeEE(0x05,0x6E); // ASCII n
        pc.printf("\n\r");
        */

        //Uncomment the following line to Erase the EEPROM
        // eraseEE();
    
        for (int i=0;i<=0x5;i++) //Just going to read the first 6 (zero to five) locations in the EEPROM
        {
            readEE(i);pc.printf("\t");          //A Bit of formating going to tab into 8 columns
            if (!((i+1)%8)){pc.printf("\n\r");} //of data
        }
    }
    else
    {
        pc.printf("\n\rCannot get an ACK from the Device check connections!\n\r");
    }
}
  