////////////////////////////////////////////////////////////////////////////////
// Copyright Rottor SAS 2017
// All rigths reserved.
//
// File Name : State.h
// Authors : Martin Matignon
//
// If you find any bug or if you have any question please contact
// Martin Matignon <martin.matignon@rottor.fr>
// Nicolas Forestier <nicolas.forestier@rottor.fr>
//
////////////////////////////////////////////////////////////////////////////////

#ifndef __SM_STATE_H__
#define __SM_STATE_H__

#include "mbed.h"
#include "UserData.h"

/** Global preempt state flag
 */
extern bool preempt_flag;

/** A virtual class for executing a simple state into state machine
 */
class State {

public:

    /** Standard outcomes
     */
    static const char* SUCCEDED;
    static const char* ABORTED;
    static const char* PREEMPTED;

    /** State type definition
     */
    enum StateType_t{
        SIMPLE_STATE   = 1,
        STATE_MACHINE            
    };
    
    /** Create a State
     * 
     *  @param uuid The state uuid
     *  @param type The state type, default type is SIMPLE_STATE
     */
    State(const char* uuid, StateType_t type = SIMPLE_STATE);
    
    /** Remap state uuid
     * 
     *  @param uuid The new state uuid
     */
    void setUUID(const char* uuid);
    
    /** Provide state uuid
     * 
     *  @return uuid The state uuid
     */
    const char* getUUID();
    
    /** Remap state type
     * 
     *  @param type The new state type
     */
    void setType(StateType_t type);
    
    /** Provide state type
     * 
     *  @return type The state type
     */
    StateType_t getType();
    
    /** Virtual method called by state machine before onExecute 
     *  (need to redefine into inherited object)
     */
    virtual void onEntry() = 0;
    
    /** Virtual method called by state machine during execution 
     *  (need to redefine into inherited object)
     */
    virtual const char* onExecute() = 0;
    
    /** Virtual method called by state machine after execution 
     *  (need to redefine into inherited object)
     */
    virtual void onExit() = 0;
    
    /** This method call onExecute during thread execution
     *  (method reserved for parallel state machine)
     */
    void _onParallelExecute();
    
    /** Provide state outcome 
     *  (method reserved for parallel state machine)
     *
     * @return outcome The parallel state outcome
     */
    const char* getOutcome();
    
    /** Method called if preempt requested by parent state machine
     */
    void preempt();
    
    /** Provide preempt state
     * (Use this method in while(!isPreempted()){// do })
     *
     * @return bool The preempt state (true/false)
     */
    bool isPreempted();
    
private:
    const char*  m_uuid;    /* State uuid             */
    StateType_t  m_type;    /* State type             */
    bool         m_preempt; /* Preempt state          */
    const char*  m_outcome; /* Parallel outcome state */
};

#endif /* #ifndef __SM_STATE_H__*/