/*
   Diablo_Serial_4DLib.cpp - Library for 4D Systems Serial Environment.
 */

#include "Diablo_Serial_4DLib.h"

Diablo_Serial_4DLib::Diablo_Serial_4DLib(Serial *virtualPort, int timeout4D) { 
    _virtualPort = virtualPort;
    TimeLimit4D = timeout4D;
}

//*********************************************************************************************//
//**********************************Intrinsic 4D Routines**************************************//
//*********************************************************************************************//

void Diablo_Serial_4DLib::WriteChars(char * charsout)
{
  char wk ;
  do
  {
    wk = *charsout++ ;
    _virtualPort->putc(wk) ;
  } while (wk) ;
}

void Diablo_Serial_4DLib::WriteBytes(char * Source, int Size)
{
    char wk ;
    int i ;
    for (i = 0; i < Size; i++)
    {
        wk = *Source++ ;
        _virtualPort->putc(wk) ;
    }
}

void Diablo_Serial_4DLib::WriteWords(word * Source, int Size)
{
  word wk ;
  int i ;
  for (i = 0; i < Size; i++)
  {
    wk = *Source++ ;
    _virtualPort->putc(wk >> 8) ;
    _virtualPort->putc(wk) ;
  }
}

void Diablo_Serial_4DLib::getbytes(char * data, int size)
{
  //int read ;
  Timer timer;
  int sttime ;
  int readc ;
  readc  = 0 ;
  timer.reset();
  timer.start();
  sttime = timer.read_ms();
  while ((readc != size) && (timer.read_ms() - sttime < TimeLimit4D))
  {
    if(_virtualPort->readable()) 
    {
      data[readc++] = _virtualPort->getc() ;
    }
  }
  if (readc != size)
  {
    Error4D = GFX_Err4D_Timeout ;
    if (Callback4D != NULL)
      Callback4D(Error4D, Error4D_Inv) ;
  }
}

void Diablo_Serial_4DLib::GetAck(void)
{
  Timer timer;
  int read ;
  unsigned char readx ;
  int sttime ;
  Error4D = GFX_Err4D_OK ;
  timer.reset();
  timer.start();
  sttime  = timer.read_ms();
  read    = 0 ;
  while ((read != 1) && (timer.read_ms() - sttime < TimeLimit4D))
  {
    if(_virtualPort->readable() != 0) 
    {
      readx = (unsigned char)_virtualPort->getc() ;
      read = 1 ;
    }
  }
  if (read == 0)
  {
    Error4D = GFX_Err4D_Timeout ;
    if (Callback4D != NULL)
      Callback4D(Error4D, Error4D_Inv) ;
  }
  else if (readx != 6)
  {
    Error4D     = GFX_Err4D_NAK ;
    Error4D_Inv = readx ;
    if (Callback4D != NULL)
      Callback4D(Error4D, Error4D_Inv) ;
  }
}

word Diablo_Serial_4DLib::GetWord(void)
{
  Timer timer;
  unsigned char readx[2] ;
  int readc ;
  int sttime ;
  
  if (Error4D != GFX_Err4D_OK)
    return 0 ;
  timer.reset();
  timer.start();
  sttime  = timer.read_ms();
  readc   = 0 ;
  while ((readc != 2) && (timer.read_ms() - sttime < TimeLimit4D))
  {
    if(_virtualPort->readable()) 
    {
      readx[readc++] = (unsigned char)_virtualPort->getc();
    }
  }
  
  if (readc != 2)
  {
    Error4D  = GFX_Err4D_Timeout ;
    if (Callback4D != NULL)
      Callback4D(Error4D, Error4D_Inv) ;
  return 0 ;
  }
  else
    return readx[0] << 8 | readx[1] ;
}

void Diablo_Serial_4DLib::getString(char * outStr, int strLen)
{
  //unsigned char readx[2] ;
  Timer timer;
  int readc ;
  int sttime ;
  
  if (Error4D != GFX_Err4D_OK)
  {
    outStr[0] = 0 ;
    return ;
  }
  timer.reset();
  timer.start();
  sttime  = timer.read_ms();
  readc   = 0 ;
  while ((readc != strLen) && (timer.read_ms() - sttime < TimeLimit4D))
  {
    if(_virtualPort->readable()) 
    {
      outStr[readc++] = _virtualPort->getc() ;
    }
  }
  
  if (readc != strLen)
  {
    Error4D  = GFX_Err4D_Timeout ;
    if (Callback4D != NULL)
      Callback4D(Error4D, Error4D_Inv) ;
  }
  outStr[readc] = 0 ;
}

word Diablo_Serial_4DLib::GetAckResp(void)
{
    GetAck();
    return GetWord() ;
}

word Diablo_Serial_4DLib::GetAckRes2Words(word * word1, word * word2)
{
    int Result ;
    GetAck() ;
    Result = GetWord() ;
    *word1 = GetWord() ;
    *word2 = GetWord() ;
    return Result ;
}

void Diablo_Serial_4DLib::GetAck2Words(word * word1, word * word2)
{
    GetAck() ;
    *word1 = GetWord() ;
    *word2 = GetWord() ;
}

word Diablo_Serial_4DLib::GetAckResSector(GFX_t4DSector Sector)
{
    int Result ;
    GetAck() ;
    Result = GetWord() ;
    getbytes(Sector, 512) ;
    return Result ;
}

word Diablo_Serial_4DLib::GetAckResStr(char * OutStr)
{
    int Result ;
    GetAck() ;
    Result = GetWord() ;
    getString(OutStr, Result) ;
    return Result ;
}

word Diablo_Serial_4DLib::GetAckResData(GFX_t4DByteArray OutData, word size)
{
    int Result ;
    GetAck() ;
    Result = GetWord() ;
    getbytes(OutData, size) ;
    return Result ;
}

void Diablo_Serial_4DLib::SetThisBaudrate(int Newrate)
{
  int br ;
 // _virtualPort->flush() ;
//  _virtualPort->end() ;
  switch(Newrate)
  {
 /*   case BAUD_110    : br = 110 ;
      break ;
    case BAUD_300    : br = 300 ;
      break ;
    case BAUD_600    : br = 600 ;
      break ;
    case BAUD_1200   : br = 1200 ;
      break ;
    case BAUD_2400   : br = 2400 ;
      break ;
    case BAUD_4800   : br = 4800 ;
      break ;*/
    case GFX_BAUD_9600   : br = 9600 ;
      break ;
//   case BAUD_14400  : br = 14400 ;
//      break ;
    case GFX_BAUD_19200  : br = 19200 ;
      break ;
 /*   case BAUD_31250  : br = 31250 ;
      break ;
    case BAUD_38400  : br = 38400 ;
      break ;
    case BAUD_56000  : br = 56000 ;
      break ;
    case BAUD_57600  : br = 57600 ;
      break ;
    case BAUD_115200 : br = 115200 ;
      break ;
    case BAUD_128000 : br = 133928 ; // actual rate is not 128000 ;
      break ;
    case BAUD_256000 : br = 281250 ; // actual rate is not  256000 ;
      break ;
    case BAUD_300000 : br = 312500 ; // actual rate is not  300000 ;
      break ;
    case BAUD_375000 : br = 401785 ; // actual rate is not  375000 ;
      break ;
    case BAUD_500000 : br = 562500 ; // actual rate is not  500000 ;
      break ;
    case BAUD_600000 : br = 703125 ; // actual rate is not  600000 ;
      break ;*/
  }
//  _virtualPort->begin(br) ;
  wait(0.05) ; // Display sleeps for 100
  _virtualPort->baud(br) ;
}

//*********************************************************************************************//
//**********************************Compound 4D Routines***************************************//
//*********************************************************************************************//

word Diablo_Serial_4DLib::bus_Read8()
{
  _virtualPort->putc(GFX_F_bus_Read8 >> 8) ;
  _virtualPort->putc(GFX_F_bus_Read8) ;
  return GetAckResp() ;
}

void Diablo_Serial_4DLib::bus_Write8(word Bits)
{
  _virtualPort->putc(GFX_F_bus_Write8 >> 8) ;
  _virtualPort->putc(GFX_F_bus_Write8) ;
  _virtualPort->putc(Bits >> 8);
  _virtualPort->putc(Bits) ;
  GetAck() ;
}

word Diablo_Serial_4DLib::charheight(char  TestChar)
{
  _virtualPort->putc(GFX_F_charheight >> 8);
  _virtualPort->putc(GFX_F_charheight);
  _virtualPort->putc(TestChar);
  return GetAckResp();
}

word Diablo_Serial_4DLib::charwidth(char  TestChar)
{
  _virtualPort->putc(GFX_F_charwidth >> 8);
  _virtualPort->putc(GFX_F_charwidth);
  _virtualPort->putc(TestChar);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Close(word  Handle)
{
  _virtualPort->putc(GFX_F_file_Close >> 8);
  _virtualPort->putc(GFX_F_file_Close);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Count(char *  Filename)
{
  _virtualPort->putc(GFX_F_file_Count >> 8);
  _virtualPort->putc(GFX_F_file_Count);
  WriteChars(Filename);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Dir(char *  Filename)
{
  _virtualPort->putc(GFX_F_file_Dir >> 8);
  _virtualPort->putc(GFX_F_file_Dir);
  WriteChars(Filename);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Erase(char *  Filename)
{
  _virtualPort->putc(GFX_F_file_Erase >> 8);
  _virtualPort->putc(GFX_F_file_Erase);
  WriteChars(Filename);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Error()
{
  _virtualPort->putc(GFX_F_file_Error >> 8);
  _virtualPort->putc(GFX_F_file_Error);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Exec(char *  Filename, word  ArgCount, GFX_t4DWordArray  Args)
{
  _virtualPort->putc(GFX_F_file_Exec >> 8);
  _virtualPort->putc(GFX_F_file_Exec);
  WriteChars(Filename);
  _virtualPort->putc(ArgCount >> 8);
  _virtualPort->putc(ArgCount);
  WriteWords(Args, ArgCount);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Exists(char *  Filename)
{
  _virtualPort->putc(GFX_F_file_Exists >> 8);
  _virtualPort->putc(GFX_F_file_Exists);
  WriteChars(Filename);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_FindFirst(char *  Filename)
{
  _virtualPort->putc(GFX_F_file_FindFirst >> 8);
  _virtualPort->putc(GFX_F_file_FindFirst);
  WriteChars(Filename);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_FindNext()
{
  _virtualPort->putc(GFX_F_file_FindNext >> 8);
  _virtualPort->putc(GFX_F_file_FindNext);
  return GetAckResp();
}

char Diablo_Serial_4DLib::file_GetC(word  Handle)
{
  _virtualPort->putc(GFX_F_file_GetC >> 8);
  _virtualPort->putc(GFX_F_file_GetC);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_GetS(char *  StringIn, word  Size, word  Handle)
{
  _virtualPort->putc(GFX_F_file_GetS >> 8);
  _virtualPort->putc(GFX_F_file_GetS);
  _virtualPort->putc(Size >> 8);
  _virtualPort->putc(Size);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResStr(StringIn);
}

word Diablo_Serial_4DLib::file_GetW(word  Handle)
{
  _virtualPort->putc(GFX_F_file_GetW >> 8);
  _virtualPort->putc(GFX_F_file_GetW);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Image(word  X, word  Y, word  Handle)
{
  _virtualPort->putc(GFX_F_file_Image >> 8);
  _virtualPort->putc(GFX_F_file_Image);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Index(word  Handle, word  HiSize, word  LoSize, word  Recordnum)
{
  _virtualPort->putc(GFX_F_file_Index >> 8);
  _virtualPort->putc(GFX_F_file_Index);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(HiSize >> 8);
  _virtualPort->putc(HiSize);
  _virtualPort->putc(LoSize >> 8);
  _virtualPort->putc(LoSize);
  _virtualPort->putc(Recordnum >> 8);
  _virtualPort->putc(Recordnum);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_LoadFunction(char *  Filename)
{
  _virtualPort->putc(GFX_F_file_LoadFunction >> 8);
  _virtualPort->putc(GFX_F_file_LoadFunction);
  WriteChars(Filename);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_LoadImageControl(char *  Datname, char *  GCIName, word  Mode)
{
  _virtualPort->putc(GFX_F_file_LoadImageControl >> 8);
  _virtualPort->putc(GFX_F_file_LoadImageControl);
  WriteChars(Datname);
  WriteChars(GCIName);
  _virtualPort->putc(Mode >> 8);
  _virtualPort->putc(Mode);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Mount()
{
  _virtualPort->putc(GFX_F_file_Mount >> 8);
  _virtualPort->putc(GFX_F_file_Mount);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Open(char *  Filename, char  Mode)
{
  _virtualPort->putc(GFX_F_file_Open >> 8);
  _virtualPort->putc(GFX_F_file_Open);
  WriteChars(Filename);
  _virtualPort->putc(Mode);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_PlayWAV(char *  Filename)
{
  _virtualPort->putc(GFX_F_file_PlayWAV >> 8);
  _virtualPort->putc(GFX_F_file_PlayWAV);
  WriteChars(Filename);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_PutC(char  Character, word  Handle)
{
  _virtualPort->putc(GFX_F_file_PutC >> 8);
  _virtualPort->putc(GFX_F_file_PutC);
  _virtualPort->putc(Character);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_PutS(char *  StringOut, word  Handle)
{
  _virtualPort->putc(GFX_F_file_PutS >> 8);
  _virtualPort->putc(GFX_F_file_PutS);
  WriteChars(StringOut);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_PutW(word  Word, word  Handle)
{
  _virtualPort->putc(GFX_F_file_PutW >> 8);
  _virtualPort->putc(GFX_F_file_PutW);
  _virtualPort->putc(Word >> 8);
  _virtualPort->putc(Word);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Read(GFX_t4DByteArray  Data, word  Size, word  Handle)
{
  _virtualPort->putc(GFX_F_file_Read >> 8);
  _virtualPort->putc(GFX_F_file_Read);
  _virtualPort->putc(Size >> 8);
  _virtualPort->putc(Size);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResData(Data,Size);
}

word Diablo_Serial_4DLib::file_Rewind(word  Handle)
{
  _virtualPort->putc(GFX_F_file_Rewind >> 8);
  _virtualPort->putc(GFX_F_file_Rewind);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Run(char *  Filename, word  ArgCount, GFX_t4DWordArray  Args)
{
  _virtualPort->putc(GFX_F_file_Run >> 8);
  _virtualPort->putc(GFX_F_file_Run);
  WriteChars(Filename);
  _virtualPort->putc(ArgCount >> 8);
  _virtualPort->putc(ArgCount);
  WriteWords(Args, ArgCount);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_ScreenCapture(word  X, word  Y, word  Width, word  Height, word  Handle)
{
  _virtualPort->putc(GFX_F_file_ScreenCapture >> 8);
  _virtualPort->putc(GFX_F_file_ScreenCapture);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Width >> 8);
  _virtualPort->putc(Width);
  _virtualPort->putc(Height >> 8);
  _virtualPort->putc(Height);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Seek(word  Handle, word  HiWord, word  LoWord)
{
  _virtualPort->putc(GFX_F_file_Seek >> 8);
  _virtualPort->putc(GFX_F_file_Seek);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(HiWord >> 8);
  _virtualPort->putc(HiWord);
  _virtualPort->putc(LoWord >> 8);
  _virtualPort->putc(LoWord);
  return GetAckResp();
}

word Diablo_Serial_4DLib::file_Size(word  Handle, word *  HiWord, word *  LoWord)
{
  _virtualPort->putc(GFX_F_file_Size >> 8);
  _virtualPort->putc(GFX_F_file_Size);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckRes2Words(HiWord,LoWord);
}

word Diablo_Serial_4DLib::file_Tell(word  Handle, word *  HiWord, word *  LoWord)
{
  _virtualPort->putc(GFX_F_file_Tell >> 8);
  _virtualPort->putc(GFX_F_file_Tell);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckRes2Words(HiWord,LoWord);
}

void Diablo_Serial_4DLib::file_Unmount()
{
  _virtualPort->putc(GFX_F_file_Unmount >> 8);
  _virtualPort->putc(GFX_F_file_Unmount);
  GetAck();
}

word Diablo_Serial_4DLib::file_Write(word  Size, GFX_t4DByteArray  Source, word  Handle)
{
  _virtualPort->putc(GFX_F_file_Write >> 8);
  _virtualPort->putc(GFX_F_file_Write);
  _virtualPort->putc(Size >> 8);
  _virtualPort->putc(Size);
  WriteBytes(Source, Size);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::gfx_BevelShadow(word  Value)
{
  _virtualPort->putc(GFX_F_gfx_BevelShadow >> 8);
  _virtualPort->putc(GFX_F_gfx_BevelShadow);
  _virtualPort->putc(Value >> 8);
  _virtualPort->putc(Value);
  return GetAckResp();
}

word Diablo_Serial_4DLib::gfx_BevelWidth(word  Value)
{
  _virtualPort->putc(GFX_F_gfx_BevelWidth >> 8);
  _virtualPort->putc(GFX_F_gfx_BevelWidth);
  _virtualPort->putc(Value >> 8);
  _virtualPort->putc(Value);
  return GetAckResp();
}

word Diablo_Serial_4DLib::gfx_BGcolour(word  Color)
{
  _virtualPort->putc(GFX_F_gfx_BGcolour >> 8);
  _virtualPort->putc(GFX_F_gfx_BGcolour);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  return GetAckResp();
}

void Diablo_Serial_4DLib::gfx_Button(word  Up, word  x, word  y, word  buttonColour, word  txtColour, word  font, word  txtWidth, word  txtHeight, char *   text)
{
  _virtualPort->putc(GFX_F_gfx_Button >> 8);
  _virtualPort->putc(GFX_F_gfx_Button);
  _virtualPort->putc(Up >> 8);
  _virtualPort->putc(Up);
  _virtualPort->putc(x >> 8);
  _virtualPort->putc(x);
  _virtualPort->putc(y >> 8);
  _virtualPort->putc(y);
  _virtualPort->putc(buttonColour >> 8);
  _virtualPort->putc(buttonColour);
  _virtualPort->putc(txtColour >> 8);
  _virtualPort->putc(txtColour);
  _virtualPort->putc(font >> 8);
  _virtualPort->putc(font);
  _virtualPort->putc(txtWidth >> 8);
  _virtualPort->putc(txtWidth);
  _virtualPort->putc(txtHeight >> 8);
  _virtualPort->putc(txtHeight);
  WriteChars(text);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_ChangeColour(word  OldColor, word  NewColor)
{
  _virtualPort->putc(GFX_F_gfx_ChangeColour >> 8);
  _virtualPort->putc(GFX_F_gfx_ChangeColour);
  _virtualPort->putc(OldColor >> 8);
  _virtualPort->putc(OldColor);
  _virtualPort->putc(NewColor >> 8);
  _virtualPort->putc(NewColor);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_Circle(word  X, word  Y, word  Radius, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Circle >> 8);
  _virtualPort->putc(GFX_F_gfx_Circle);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Radius >> 8);
  _virtualPort->putc(Radius);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_CircleFilled(word  X, word  Y, word  Radius, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_CircleFilled >> 8);
  _virtualPort->putc(GFX_F_gfx_CircleFilled);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Radius >> 8);
  _virtualPort->putc(Radius);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_Clipping(word  OnOff)
{
  _virtualPort->putc(GFX_F_gfx_Clipping >> 8);
  _virtualPort->putc(GFX_F_gfx_Clipping);
  _virtualPort->putc(OnOff >> 8);
  _virtualPort->putc(OnOff);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_ClipWindow(word  X1, word  Y1, word  X2, word  Y2)
{
  _virtualPort->putc(GFX_F_gfx_ClipWindow >> 8);
  _virtualPort->putc(GFX_F_gfx_ClipWindow);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_Cls()
{
  _virtualPort->putc(GFX_F_gfx_Cls >> 8);
  _virtualPort->putc(GFX_F_gfx_Cls);
  GetAck();
}

word Diablo_Serial_4DLib::gfx_Contrast(word  Contrast)
{
  _virtualPort->putc(GFX_F_gfx_Contrast >> 8);
  _virtualPort->putc(GFX_F_gfx_Contrast);
  _virtualPort->putc(Contrast >> 8);
  _virtualPort->putc(Contrast);
  return GetAckResp();
}

void Diablo_Serial_4DLib::gfx_Ellipse(word  X, word  Y, word  Xrad, word  Yrad, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Ellipse >> 8);
  _virtualPort->putc(GFX_F_gfx_Ellipse);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Xrad >> 8);
  _virtualPort->putc(Xrad);
  _virtualPort->putc(Yrad >> 8);
  _virtualPort->putc(Yrad);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_EllipseFilled(word  X, word  Y, word  Xrad, word  Yrad, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_EllipseFilled >> 8);
  _virtualPort->putc(GFX_F_gfx_EllipseFilled);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Xrad >> 8);
  _virtualPort->putc(Xrad);
  _virtualPort->putc(Yrad >> 8);
  _virtualPort->putc(Yrad);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

word Diablo_Serial_4DLib::gfx_FrameDelay(word  Msec)
{
  _virtualPort->putc(GFX_F_gfx_FrameDelay >> 8);
  _virtualPort->putc(GFX_F_gfx_FrameDelay);
  _virtualPort->putc(Msec >> 8);
  _virtualPort->putc(Msec);
  return GetAckResp();
}

word Diablo_Serial_4DLib::gfx_Get(word  Mode)
{
  _virtualPort->putc(GFX_F_gfx_Get >> 8);
  _virtualPort->putc(GFX_F_gfx_Get);
  _virtualPort->putc(Mode >> 8);
  _virtualPort->putc(Mode);
  return GetAckResp();
}

word Diablo_Serial_4DLib::gfx_GetPixel(word  X, word  Y)
{
  _virtualPort->putc(GFX_F_gfx_GetPixel >> 8);
  _virtualPort->putc(GFX_F_gfx_GetPixel);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  return GetAckResp();
}

void Diablo_Serial_4DLib::gfx_Line(word  X1, word  Y1, word  X2, word  Y2, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Line >> 8);
  _virtualPort->putc(GFX_F_gfx_Line);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

word Diablo_Serial_4DLib::gfx_LinePattern(word  Pattern)
{
  _virtualPort->putc(GFX_F_gfx_LinePattern >> 8);
  _virtualPort->putc(GFX_F_gfx_LinePattern);
  _virtualPort->putc(Pattern >> 8);
  _virtualPort->putc(Pattern);
  return GetAckResp();
}

void Diablo_Serial_4DLib::gfx_LineTo(word  X, word  Y)
{
  _virtualPort->putc(GFX_F_gfx_LineTo >> 8);
  _virtualPort->putc(GFX_F_gfx_LineTo);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_MoveTo(word  X, word  Y)
{
  _virtualPort->putc(GFX_F_gfx_MoveTo >> 8);
  _virtualPort->putc(GFX_F_gfx_MoveTo);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  GetAck();
}

word Diablo_Serial_4DLib::gfx_Orbit(word  Angle, word  Distance, word *  Xdest, word *  Ydest)
{
  _virtualPort->putc(GFX_F_gfx_Orbit >> 8);
  _virtualPort->putc(GFX_F_gfx_Orbit);
  _virtualPort->putc(Angle >> 8);
  _virtualPort->putc(Angle);
  _virtualPort->putc(Distance >> 8);
  _virtualPort->putc(Distance);
  GetAck2Words(Xdest,Ydest);
  return 0 ;
}

word Diablo_Serial_4DLib::gfx_OutlineColour(word  Color)
{
  _virtualPort->putc(GFX_F_gfx_OutlineColour >> 8);
  _virtualPort->putc(GFX_F_gfx_OutlineColour);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  return GetAckResp();
}

void Diablo_Serial_4DLib::gfx_Panel(word  Raised, word  X, word  Y, word  Width, word  Height, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Panel >> 8);
  _virtualPort->putc(GFX_F_gfx_Panel);
  _virtualPort->putc(Raised >> 8);
  _virtualPort->putc(Raised);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Width >> 8);
  _virtualPort->putc(Width);
  _virtualPort->putc(Height >> 8);
  _virtualPort->putc(Height);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_Polygon(word  n, GFX_t4DWordArray  Xvalues, GFX_t4DWordArray  Yvalues, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Polygon >> 8);
  _virtualPort->putc(GFX_F_gfx_Polygon);
  _virtualPort->putc(n >> 8);
  _virtualPort->putc(n);
  WriteWords(Xvalues, n);
  WriteWords(Yvalues, n);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_PolygonFilled(word  n, GFX_t4DWordArray  Xvalues, GFX_t4DWordArray  Yvalues, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_PolygonFilled >> 8);
  _virtualPort->putc(GFX_F_gfx_PolygonFilled);
  _virtualPort->putc(n >> 8);
  _virtualPort->putc(n);
  WriteWords(Xvalues, n);
  WriteWords(Yvalues, n);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_Polyline(word  n, GFX_t4DWordArray  Xvalues, GFX_t4DWordArray  Yvalues, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Polyline >> 8);
  _virtualPort->putc(GFX_F_gfx_Polyline);
  _virtualPort->putc(n >> 8);
  _virtualPort->putc(n);
  WriteWords(Xvalues, n);
  WriteWords(Yvalues, n);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_PutPixel(word  X, word  Y, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_PutPixel >> 8);
  _virtualPort->putc(GFX_F_gfx_PutPixel);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_Rectangle(word  X1, word  Y1, word  X2, word  Y2, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Rectangle >> 8);
  _virtualPort->putc(GFX_F_gfx_Rectangle);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_RectangleFilled(word  X1, word  Y1, word  X2, word  Y2, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_RectangleFilled >> 8);
  _virtualPort->putc(GFX_F_gfx_RectangleFilled);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_ScreenCopyPaste(word  Xs, word  Ys, word  Xd, word  Yd, word  Width, word  Height)
{
  _virtualPort->putc(GFX_F_gfx_ScreenCopyPaste >> 8);
  _virtualPort->putc(GFX_F_gfx_ScreenCopyPaste);
  _virtualPort->putc(Xs >> 8);
  _virtualPort->putc(Xs);
  _virtualPort->putc(Ys >> 8);
  _virtualPort->putc(Ys);
  _virtualPort->putc(Xd >> 8);
  _virtualPort->putc(Xd);
  _virtualPort->putc(Yd >> 8);
  _virtualPort->putc(Yd);
  _virtualPort->putc(Width >> 8);
  _virtualPort->putc(Width);
  _virtualPort->putc(Height >> 8);
  _virtualPort->putc(Height);
  GetAck();
}

word Diablo_Serial_4DLib::gfx_ScreenMode(word  ScreenMode)
{
  _virtualPort->putc(GFX_F_gfx_ScreenMode >> 8);
  _virtualPort->putc(GFX_F_gfx_ScreenMode);
  _virtualPort->putc(ScreenMode >> 8);
  _virtualPort->putc(ScreenMode);
  return GetAckResp();
}

void Diablo_Serial_4DLib::gfx_Set(word  Func, word  Value)
{
  _virtualPort->putc(GFX_F_gfx_Set >> 8);
  _virtualPort->putc(GFX_F_gfx_Set);
  _virtualPort->putc(Func >> 8);
  _virtualPort->putc(Func);
  _virtualPort->putc(Value >> 8);
  _virtualPort->putc(Value);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_SetClipRegion()
{
  _virtualPort->putc(GFX_F_gfx_SetClipRegion >> 8);
  _virtualPort->putc(GFX_F_gfx_SetClipRegion);
  GetAck();
}

word Diablo_Serial_4DLib::gfx_Slider(word  Mode, word  X1, word  Y1, word  X2, word  Y2, word  Color, word  Scale, word  Value)
{
  _virtualPort->putc(GFX_F_gfx_Slider >> 8);
  _virtualPort->putc(GFX_F_gfx_Slider);
  _virtualPort->putc(Mode >> 8);
  _virtualPort->putc(Mode);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  _virtualPort->putc(Scale >> 8);
  _virtualPort->putc(Scale);
  _virtualPort->putc(Value >> 8);
  _virtualPort->putc(Value);
  return GetAckResp();
}

word Diablo_Serial_4DLib::gfx_Transparency(word  OnOff)
{
  _virtualPort->putc(GFX_F_gfx_Transparency >> 8);
  _virtualPort->putc(GFX_F_gfx_Transparency);
  _virtualPort->putc(OnOff >> 8);
  _virtualPort->putc(OnOff);
  return GetAckResp();
}

word Diablo_Serial_4DLib::gfx_TransparentColour(word  Color)
{
  _virtualPort->putc(GFX_F_gfx_TransparentColour >> 8);
  _virtualPort->putc(GFX_F_gfx_TransparentColour);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  return GetAckResp();
}

void Diablo_Serial_4DLib::gfx_Triangle(word  X1, word  Y1, word  X2, word  Y2, word  X3, word  Y3, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_Triangle >> 8);
  _virtualPort->putc(GFX_F_gfx_Triangle);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  _virtualPort->putc(X3 >> 8);
  _virtualPort->putc(X3);
  _virtualPort->putc(Y3 >> 8);
  _virtualPort->putc(Y3);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

void Diablo_Serial_4DLib::gfx_TriangleFilled(word  X1, word  Y1, word  X2, word  Y2, word  X3, word  Y3, word  Color)
{
  _virtualPort->putc(GFX_F_gfx_TriangleFilled >> 8);
  _virtualPort->putc(GFX_F_gfx_TriangleFilled);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  _virtualPort->putc(X3 >> 8);
  _virtualPort->putc(X3);
  _virtualPort->putc(Y3 >> 8);
  _virtualPort->putc(Y3);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  GetAck();
}

word Diablo_Serial_4DLib::img_ClearAttributes(word  Handle, word  Index, word  Value)
{
  _virtualPort->putc(GFX_F_img_ClearAttributes >> 8);
  _virtualPort->putc(GFX_F_img_ClearAttributes);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  _virtualPort->putc(Value >> 8);
  _virtualPort->putc(Value);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_Darken(word  Handle, word  Index)
{
  _virtualPort->putc(GFX_F_img_Darken >> 8);
  _virtualPort->putc(GFX_F_img_Darken);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_Disable(word  Handle, word  Index)
{
  _virtualPort->putc(GFX_F_img_Disable >> 8);
  _virtualPort->putc(GFX_F_img_Disable);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_Enable(word  Handle, word  Index)
{
  _virtualPort->putc(GFX_F_img_Enable >> 8);
  _virtualPort->putc(GFX_F_img_Enable);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_GetWord(word  Handle, word  Index, word  Offset )
{
  _virtualPort->putc(GFX_F_img_GetWord >> 8);
  _virtualPort->putc(GFX_F_img_GetWord);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  _virtualPort->putc(Offset  >> 8);
  _virtualPort->putc(Offset );
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_Lighten(word  Handle, word  Index)
{
  _virtualPort->putc(GFX_F_img_Lighten >> 8);
  _virtualPort->putc(GFX_F_img_Lighten);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_SetAttributes(word  Handle, word  Index, word  Value)
{
  _virtualPort->putc(GFX_F_img_SetAttributes >> 8);
  _virtualPort->putc(GFX_F_img_SetAttributes);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  _virtualPort->putc(Value >> 8);
  _virtualPort->putc(Value);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_SetPosition(word  Handle, word  Index, word  Xpos, word  Ypos)
{
  _virtualPort->putc(GFX_F_img_SetPosition >> 8);
  _virtualPort->putc(GFX_F_img_SetPosition);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  _virtualPort->putc(Xpos >> 8);
  _virtualPort->putc(Xpos);
  _virtualPort->putc(Ypos >> 8);
  _virtualPort->putc(Ypos);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_SetWord(word  Handle, word  Index, word  Offset , word  Word)
{
  _virtualPort->putc(GFX_F_img_SetWord >> 8);
  _virtualPort->putc(GFX_F_img_SetWord);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  _virtualPort->putc(Offset  >> 8);
  _virtualPort->putc(Offset );
  _virtualPort->putc(Word >> 8);
  _virtualPort->putc(Word);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_Show(word  Handle, word  Index)
{
  _virtualPort->putc(GFX_F_img_Show >> 8);
  _virtualPort->putc(GFX_F_img_Show);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  return GetAckResp();
}

word Diablo_Serial_4DLib::img_Touched(word  Handle, word  Index)
{
  _virtualPort->putc(GFX_F_img_Touched >> 8);
  _virtualPort->putc(GFX_F_img_Touched);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(Index >> 8);
  _virtualPort->putc(Index);
  return GetAckResp();
}

word Diablo_Serial_4DLib::media_Flush()
{
  _virtualPort->putc(GFX_F_media_Flush >> 8);
  _virtualPort->putc(GFX_F_media_Flush);
  return GetAckResp();
}

void Diablo_Serial_4DLib::media_Image(word  X, word  Y)
{
  _virtualPort->putc(GFX_F_media_Image >> 8);
  _virtualPort->putc(GFX_F_media_Image);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  GetAck();
}

word Diablo_Serial_4DLib::media_Init()
{
  _virtualPort->putc(GFX_F_media_Init >> 8);
  _virtualPort->putc(GFX_F_media_Init);
  return GetAckResp();
}

word Diablo_Serial_4DLib::media_RdSector(GFX_t4DSector  SectorIn)
{
  _virtualPort->putc(GFX_F_media_RdSector >> 8);
  _virtualPort->putc(GFX_F_media_RdSector);
  return GetAckResSector(SectorIn);
}

word Diablo_Serial_4DLib::media_ReadByte()
{
  _virtualPort->putc(GFX_F_media_ReadByte >> 8);
  _virtualPort->putc(GFX_F_media_ReadByte);
  return GetAckResp();
}

word Diablo_Serial_4DLib::media_ReadWord()
{
  _virtualPort->putc(GFX_F_media_ReadWord >> 8);
  _virtualPort->putc(GFX_F_media_ReadWord);
  return GetAckResp();
}

void Diablo_Serial_4DLib::media_SetAdd(word  HiWord, word  LoWord)
{
  _virtualPort->putc(GFX_F_media_SetAdd >> 8);
  _virtualPort->putc(GFX_F_media_SetAdd);
  _virtualPort->putc(HiWord >> 8);
  _virtualPort->putc(HiWord);
  _virtualPort->putc(LoWord >> 8);
  _virtualPort->putc(LoWord);
  GetAck();
}

void Diablo_Serial_4DLib::media_SetSector(word  HiWord, word  LoWord)
{
  _virtualPort->putc(GFX_F_media_SetSector >> 8);
  _virtualPort->putc(GFX_F_media_SetSector);
  _virtualPort->putc(HiWord >> 8);
  _virtualPort->putc(HiWord);
  _virtualPort->putc(LoWord >> 8);
  _virtualPort->putc(LoWord);
  GetAck();
}

void Diablo_Serial_4DLib::media_Video(word  X, word  Y)
{
  _virtualPort->putc(GFX_F_media_Video >> 8);
  _virtualPort->putc(GFX_F_media_Video);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  GetAck();
}

void Diablo_Serial_4DLib::media_VideoFrame(word  X, word  Y, word  Framenumber)
{
  _virtualPort->putc(GFX_F_media_VideoFrame >> 8);
  _virtualPort->putc(GFX_F_media_VideoFrame);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Framenumber >> 8);
  _virtualPort->putc(Framenumber);
  GetAck();
}

word Diablo_Serial_4DLib::media_WriteByte(word  Byte)
{
  _virtualPort->putc(GFX_F_media_WriteByte >> 8);
  _virtualPort->putc(GFX_F_media_WriteByte);
  _virtualPort->putc(Byte >> 8);
  _virtualPort->putc(Byte);
  return GetAckResp();
}

word Diablo_Serial_4DLib::media_WriteWord(word  Word)
{
  _virtualPort->putc(GFX_F_media_WriteWord >> 8);
  _virtualPort->putc(GFX_F_media_WriteWord);
  _virtualPort->putc(Word >> 8);
  _virtualPort->putc(Word);
  return GetAckResp();
}

word Diablo_Serial_4DLib::media_WrSector(GFX_t4DSector  SectorOut)
{
  _virtualPort->putc(GFX_F_media_WrSector >> 8);
  _virtualPort->putc(GFX_F_media_WrSector);
  WriteBytes(SectorOut, 512);
  return GetAckResp();
}

word Diablo_Serial_4DLib::mem_Free(word  Handle)
{
  _virtualPort->putc(GFX_F_mem_Free >> 8);
  _virtualPort->putc(GFX_F_mem_Free);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResp();
}

word Diablo_Serial_4DLib::mem_Heap()
{
  _virtualPort->putc(GFX_F_mem_Heap >> 8);
  _virtualPort->putc(GFX_F_mem_Heap);
  return GetAckResp();
}

word Diablo_Serial_4DLib::peekM(word  Address)
{
  _virtualPort->putc(GFX_F_peekM >> 8);
  _virtualPort->putc(GFX_F_peekM);
  _virtualPort->putc(Address >> 8);
  _virtualPort->putc(Address);
  return GetAckResp() ;
}

word Diablo_Serial_4DLib::pin_HI(word Pin)
{
  _virtualPort->putc(GFX_F_pin_HI >> 8);
  _virtualPort->putc(GFX_F_pin_HI);
  _virtualPort->putc(Pin >> 8);
  _virtualPort->putc(Pin);
  return GetAckResp() ;
}

word Diablo_Serial_4DLib::pin_LO(word Pin)
{
  _virtualPort->putc(GFX_F_pin_LO >> 8);
  _virtualPort->putc(GFX_F_pin_LO);
  _virtualPort->putc(Pin >> 8);
  _virtualPort->putc(Pin);
  return GetAckResp() ;
}

word Diablo_Serial_4DLib::pin_Read(word Pin)
{
  _virtualPort->putc(GFX_F_pin_Read >> 8);
  _virtualPort->putc(GFX_F_pin_Read);
  _virtualPort->putc(Pin >> 8);
  _virtualPort->putc(Pin);
  return GetAckResp() ;
}

word Diablo_Serial_4DLib::pin_Set(word Mode, word Pin)
{
  _virtualPort->putc(GFX_F_pin_Set >> 8);
  _virtualPort->putc(GFX_F_pin_Set);
  _virtualPort->putc(Mode >> 8);
  _virtualPort->putc(Mode);
  _virtualPort->putc(Pin >> 8);
  _virtualPort->putc(Pin);
  return GetAckResp() ;
}

void Diablo_Serial_4DLib::pokeM(word  Address, word  WordValue)
{
  _virtualPort->putc(GFX_F_pokeM >> 8);
  _virtualPort->putc(GFX_F_pokeM);
  _virtualPort->putc(Address >> 8);
  _virtualPort->putc(Address);
  _virtualPort->putc(WordValue >> 8);
  _virtualPort->putc(WordValue);
  GetAck() ;
}

void Diablo_Serial_4DLib::putCH(word  WordChar)
{
  _virtualPort->putc(GFX_F_putCH >> 8);
  _virtualPort->putc(GFX_F_putCH);
  _virtualPort->putc(WordChar >> 8);
  _virtualPort->putc(WordChar);
  GetAck() ;
}

word Diablo_Serial_4DLib::putstr(char *  InString)
{
  _virtualPort->putc(GFX_F_putstr >> 8);
  _virtualPort->putc(GFX_F_putstr);
  WriteChars(InString);
  return GetAckResp();
}

//-----------------print----------------------------------

void Diablo_Serial_4DLib::print(const string &s)
{
    int len = s.length();
    //char arr[len + 1];
    //s.toCharArray(arr,len + 1);
    //putstr(arr);
    for(int x=0; x<len; x++)
    {
        putCH(s[x]);
    }
}

void Diablo_Serial_4DLib::print(const char str[])
{
  int len = strlen(str);
  for(int x = 0; x<len; x++)
  {
    putCH(str[x]);
  }
}

void Diablo_Serial_4DLib::print(char c)
{
  putCH(c);
}

void Diablo_Serial_4DLib::print(unsigned char b, int base)
{
  print((unsigned long) b, base);
}

void Diablo_Serial_4DLib::print(int n, int base)
{
  print((long) n, base);
}

void Diablo_Serial_4DLib::print(unsigned int n, int base)
{
  print((unsigned long) n, base);
}

void Diablo_Serial_4DLib::print(long n, int base)
{

  if (base == 10) 
  {
    if (n < 0) 
    {
      //int t = print('-');
      putCH('-');
      n = -n;
      printNumber(n, 10);
    }
    else
    {
    printNumber(n, 10);
    }
  } 
  else 
  {
    printNumber(n, base);
  }

}

void Diablo_Serial_4DLib::print(unsigned long n, int base)
{
  printNumber(n, base);
}

void Diablo_Serial_4DLib::print(double n, int digits)
{
  printFloat(n, digits);
}


//println
/*
void Diablo_Serial_4DLib::println(const __FlashStringHelper *ifsh)
{
  print(ifsh);
  putCH('\n');
  putCH('\r');
}
*/
void Diablo_Serial_4DLib::println(const string &s)
{
  print(s);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(const char c[])
{
  print(c);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(char c)
{
 // print(string(c);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(unsigned char b, int base)
{
  print((unsigned char)b, base);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(int num, int base)
{
  print((int)num, base);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(unsigned int num, int base)
{
  print((unsigned int)num, base);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(long num, int base)
{
  print((long)num, base);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(unsigned long num, int base)
{
  print((unsigned long)num, base);
  putCH('\n');
  putCH('\r');
}

void Diablo_Serial_4DLib::println(double num, int digits)
{
  print((double)num, digits);
  putCH('\n');
  putCH('\r');
}


//-Private:

void Diablo_Serial_4DLib::printNumber(unsigned long n, uint8_t base) {
  char buf[8 * sizeof(long) + 1]; // Assumes 8-bit chars plus zero byte.
  int count = 0;
  do {
    unsigned long m = n;
    n /= base;
    char c = m - base * n;
    buf[count] = c < 10 ? c + '0' : c + 'A' - 10;
    count++;
  } while(n);
  
  for(int x = count - 1; x>=0; x--)
  {
    putCH(buf[x]);
  }
}

void Diablo_Serial_4DLib::printFloat(double number, uint8_t digits) 
{ 
  //size_t n = 0;
  if (isnan(number)) print("nan"); 
  else if (isinf(number)) print("inf"); 
  else if (number > 4294967040.0) print ("ovf");  // constant determined empirically
  else if (number <-4294967040.0) print ("ovf");  // constant determined empirically
  else{
  // Handle negative numbers
  if (number < 0.0)
  {
     putCH('-');
     number = -number;
  }

  // Round correctly so that print(1.999, 2) prints as "2.00"
  double rounding = 0.5;
  for (uint8_t i=0; i<digits; ++i)
    rounding /= 10.0;
  
  number += rounding;

  // Extract the integer part of the number and print it
  unsigned long int_part = (unsigned long)number;
  double remainder = number - (double)int_part;
  //print((unsigned long)int_part);
  printNumber(int_part, 10);

  // Print the decimal point, but only if there are digits beyond
  if (digits > 0) {
    putCH('.'); 
  }

  // Extract digits from the remainder one at a time
  while (digits-- > 0)
  {
    remainder *= 10.0;
    int toPrint = int(remainder);
    printNumber((unsigned long)toPrint, 10);
    remainder -= toPrint; 
  } 
  }
}


//--------------------------------------------------------

void Diablo_Serial_4DLib::snd_BufSize(word  Bufsize)
{
  _virtualPort->putc(GFX_F_snd_BufSize >> 8);
  _virtualPort->putc(GFX_F_snd_BufSize);
  _virtualPort->putc(Bufsize >> 8);
  _virtualPort->putc(Bufsize);
  GetAck();
}

void Diablo_Serial_4DLib::snd_Continue()
{
  _virtualPort->putc(GFX_F_snd_Continue >> 8);
  _virtualPort->putc(GFX_F_snd_Continue);
  GetAck();
}

void Diablo_Serial_4DLib::snd_Pause()
{
  _virtualPort->putc(GFX_F_snd_Pause >> 8);
  _virtualPort->putc(GFX_F_snd_Pause);
  GetAck();
}

word Diablo_Serial_4DLib::snd_Pitch(word  Pitch)
{
  _virtualPort->putc(GFX_F_snd_Pitch >> 8);
  _virtualPort->putc(GFX_F_snd_Pitch);
  _virtualPort->putc(Pitch >> 8);
  _virtualPort->putc(Pitch);
  return GetAckResp();
}

word Diablo_Serial_4DLib::snd_Playing()
{
  _virtualPort->putc(GFX_F_snd_Playing >> 8);
  _virtualPort->putc(GFX_F_snd_Playing);
  return GetAckResp();
}

void Diablo_Serial_4DLib::snd_Stop()
{
  _virtualPort->putc(GFX_F_snd_Stop >> 8);
  _virtualPort->putc(GFX_F_snd_Stop);
  GetAck();
}

void Diablo_Serial_4DLib::snd_Volume(word  Volume)
{
  _virtualPort->putc(GFX_F_snd_Volume >> 8);
  _virtualPort->putc(GFX_F_snd_Volume);
  _virtualPort->putc(Volume >> 8);
  _virtualPort->putc(Volume);
  GetAck();
}

word Diablo_Serial_4DLib::sys_Sleep(word  Units)
{
  _virtualPort->putc(GFX_F_sys_Sleep >> 8);
  _virtualPort->putc(GFX_F_sys_Sleep);
  _virtualPort->putc(Units >> 8);
  _virtualPort->putc(Units);
  return GetAckResp();
}

void Diablo_Serial_4DLib::touch_DetectRegion(word  X1, word  Y1, word  X2, word  Y2)
{
  _virtualPort->putc(GFX_F_touch_DetectRegion >> 8);
  _virtualPort->putc(GFX_F_touch_DetectRegion);
  _virtualPort->putc(X1 >> 8);
  _virtualPort->putc(X1);
  _virtualPort->putc(Y1 >> 8);
  _virtualPort->putc(Y1);
  _virtualPort->putc(X2 >> 8);
  _virtualPort->putc(X2);
  _virtualPort->putc(Y2 >> 8);
  _virtualPort->putc(Y2);
  GetAck();
}

word Diablo_Serial_4DLib::touch_Get(word  Mode)
{
  _virtualPort->putc(GFX_F_touch_Get >> 8);
  _virtualPort->putc(GFX_F_touch_Get);
  _virtualPort->putc(Mode >> 8);
  _virtualPort->putc(Mode);
  return GetAckResp();
}

void Diablo_Serial_4DLib::touch_Set(word  Mode)
{
  _virtualPort->putc(GFX_F_touch_Set >> 8);
  _virtualPort->putc(GFX_F_touch_Set);
  _virtualPort->putc(Mode >> 8);
  _virtualPort->putc(Mode);
  GetAck();
}

word Diablo_Serial_4DLib::txt_Attributes(word  Attribs)
{
  _virtualPort->putc(GFX_F_txt_Attributes >> 8);
  _virtualPort->putc(GFX_F_txt_Attributes);
  _virtualPort->putc(Attribs >> 8);
  _virtualPort->putc(Attribs);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_BGcolour(word  Color)
{
  _virtualPort->putc(GFX_F_txt_BGcolour >> 8);
  _virtualPort->putc(GFX_F_txt_BGcolour);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_Bold(word  Bold)
{
  _virtualPort->putc(GFX_F_txt_Bold >> 8);
  _virtualPort->putc(GFX_F_txt_Bold);
  _virtualPort->putc(Bold >> 8);
  _virtualPort->putc(Bold);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_FGcolour(word  Color)
{
  _virtualPort->putc(GFX_F_txt_FGcolour >> 8);
  _virtualPort->putc(GFX_F_txt_FGcolour);
  _virtualPort->putc(Color >> 8);
  _virtualPort->putc(Color);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_FontID(word  FontNumber)
{
  _virtualPort->putc(GFX_F_txt_FontID >> 8);
  _virtualPort->putc(GFX_F_txt_FontID);
  _virtualPort->putc(FontNumber >> 8);
  _virtualPort->putc(FontNumber);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_Height(word  Multiplier)
{
  _virtualPort->putc(GFX_F_txt_Height >> 8);
  _virtualPort->putc(GFX_F_txt_Height);
  _virtualPort->putc(Multiplier >> 8);
  _virtualPort->putc(Multiplier);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_Inverse(word  Inverse)
{
  _virtualPort->putc(GFX_F_txt_Inverse >> 8);
  _virtualPort->putc(GFX_F_txt_Inverse);
  _virtualPort->putc(Inverse >> 8);
  _virtualPort->putc(Inverse);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_Italic(word  Italic)
{
  _virtualPort->putc(GFX_F_txt_Italic >> 8);
  _virtualPort->putc(GFX_F_txt_Italic);
  _virtualPort->putc(Italic >> 8);
  _virtualPort->putc(Italic);
  return GetAckResp();
}

void Diablo_Serial_4DLib::txt_MoveCursor(word  Line, word  Column)
{
  _virtualPort->putc(GFX_F_txt_MoveCursor >> 8);
  _virtualPort->putc(GFX_F_txt_MoveCursor);
  _virtualPort->putc(Line >> 8);
  _virtualPort->putc(Line);
  _virtualPort->putc(Column >> 8);
  _virtualPort->putc(Column);
  GetAck();
}

word Diablo_Serial_4DLib::txt_Opacity(word  TransparentOpaque)
{
  _virtualPort->putc(GFX_F_txt_Opacity >> 8);
  _virtualPort->putc(GFX_F_txt_Opacity);
  _virtualPort->putc(TransparentOpaque >> 8);
  _virtualPort->putc(TransparentOpaque);
  return GetAckResp();
}

void Diablo_Serial_4DLib::txt_Set(word  Func, word  Value)
{
  _virtualPort->putc(GFX_F_txt_Set >> 8);
  _virtualPort->putc(GFX_F_txt_Set);
  _virtualPort->putc(Func >> 8);
  _virtualPort->putc(Func);
  _virtualPort->putc(Value >> 8);
  _virtualPort->putc(Value);
  GetAck();
}

word Diablo_Serial_4DLib::txt_Underline(word  Underline)
{
  _virtualPort->putc(GFX_F_txt_Underline >> 8);
  _virtualPort->putc(GFX_F_txt_Underline);
  _virtualPort->putc(Underline >> 8);
  _virtualPort->putc(Underline);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_Width(word  Multiplier)
{
  _virtualPort->putc(GFX_F_txt_Width >> 8);
  _virtualPort->putc(GFX_F_txt_Width);
  _virtualPort->putc(Multiplier >> 8);
  _virtualPort->putc(Multiplier);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_Wrap(word  Position)
{
  _virtualPort->putc(GFX_F_txt_Wrap >> 8);
  _virtualPort->putc(GFX_F_txt_Wrap);
  _virtualPort->putc(Position >> 8);
  _virtualPort->putc(Position);
  return GetAckResp();
}

word Diablo_Serial_4DLib::txt_Xgap(word  Pixels)
{
  _virtualPort->putc(GFX_F_txt_Xgap >> 8);
  _virtualPort->putc(GFX_F_txt_Xgap);
  _virtualPort->putc(Pixels >> 8);
  _virtualPort->putc(Pixels);
  return GetAckResp() ;
}

word Diablo_Serial_4DLib::txt_Ygap(word  Pixels)
{
  _virtualPort->putc(GFX_F_txt_Ygap >> 8);
  _virtualPort->putc(GFX_F_txt_Ygap);
  _virtualPort->putc(Pixels >> 8);
  _virtualPort->putc(Pixels);
  return GetAckResp() ;
}

word Diablo_Serial_4DLib::file_CallFunction(word  Handle, word  ArgCount, GFX_t4DWordArray  Args)
{
  _virtualPort->putc(GFX_F_file_CallFunction >> 8);
  _virtualPort->putc(GFX_F_file_CallFunction);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  _virtualPort->putc(ArgCount >> 8);
  _virtualPort->putc(ArgCount);
  WriteWords(Args, ArgCount);
  return GetAckResp();
}

word Diablo_Serial_4DLib::sys_GetModel(char *  ModelStr)
{
  _virtualPort->putc(GFX_F_sys_GetModel >> 8);
  _virtualPort->putc(GFX_F_sys_GetModel);
  return GetAckResStr(ModelStr);
}

word Diablo_Serial_4DLib::sys_GetVersion()
{
  _virtualPort->putc(GFX_F_sys_GetVersion >> 8);
  _virtualPort->putc(GFX_F_sys_GetVersion);
  return GetAckResp();
}

word Diablo_Serial_4DLib::sys_GetPmmC()
{
  _virtualPort->putc(GFX_F_sys_GetPmmC >> 8);
  _virtualPort->putc(GFX_F_sys_GetPmmC);
  return GetAckResp();
}

word Diablo_Serial_4DLib::writeString(word  Handle, char *  StringOut)
{
  _virtualPort->putc(GFX_F_writeString >> 8);
  _virtualPort->putc(GFX_F_writeString);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  WriteChars(StringOut) ;
  return GetAckResp() ;
}

word Diablo_Serial_4DLib::readString(word  Handle, char *  StringIn)
{
  _virtualPort->putc(GFX_F_readString >> 8);
  _virtualPort->putc(GFX_F_readString);
  _virtualPort->putc(Handle >> 8);
  _virtualPort->putc(Handle);
  return GetAckResStr(StringIn);
}

void Diablo_Serial_4DLib::blitComtoDisplay(word  X, word  Y, word  Width, word  Height, GFX_t4DByteArray  Pixels)
{
  _virtualPort->putc(GFX_F_blitComtoDisplay >> 8);
  _virtualPort->putc(GFX_F_blitComtoDisplay);
  _virtualPort->putc(X >> 8);
  _virtualPort->putc(X);
  _virtualPort->putc(Y >> 8);
  _virtualPort->putc(Y);
  _virtualPort->putc(Width >> 8);
  _virtualPort->putc(Width);
  _virtualPort->putc(Height >> 8);
  _virtualPort->putc(Height);
  WriteBytes(Pixels, Width*Height*2);
  GetAck() ;
}

word Diablo_Serial_4DLib::file_FindFirstRet(char *  Filename, char *  StringIn)
{
  _virtualPort->putc(GFX_F_file_FindFirstRet >> 8) ;
  _virtualPort->putc(GFX_F_file_FindFirstRet) ;
  WriteChars(Filename) ;
  return GetAckResStr(StringIn) ;
}

word Diablo_Serial_4DLib::file_FindNextRet(char *  StringIn)
{
  _virtualPort->putc(GFX_F_file_FindNextRet >> 8) ;
  _virtualPort->putc(GFX_F_file_FindNextRet) ;
  return GetAckResStr(StringIn) ;
}

void Diablo_Serial_4DLib::setbaudWait(word  Newrate)
{
  _virtualPort->putc(GFX_F_setbaudWait >> 8);
  _virtualPort->putc(GFX_F_setbaudWait);
  _virtualPort->putc(Newrate >> 8);
  _virtualPort->putc(Newrate);
  SetThisBaudrate(Newrate); // change this systems baud rate to match new display rate, ACK is 100ms away
  GetAck() ;
}
