/**
 This code tests the latest version of the library Ser25lcxxx, using 
 a 25LC256 eeprom (256KB)
 
 author: Mario Bambagini
*/

#include "mbed.h"

#include "Ser25lcxxx.h"


void printMemory (unsigned char *s, int len);

void waitKey();


#define PAGESIZE        64
#define PAGENUMBER      4096
#define EEPROM_SIZE     (PAGESIZE*PAGENUMBER)

#define BUFFER_SIZE     (PAGESIZE*4)


struct test_serialization {
    unsigned char data_1;
    unsigned int  data_2;
    unsigned long data_3;
};

int main()
{
    unsigned char buf[BUFFER_SIZE];
    memset(buf, 0x00, BUFFER_SIZE);

    printf("******* START *******\r\n\r\n");

    /**************************** INITIALIZATION ****************************/
    printf("INIT\r\n");
    Ser25LCxxx flash(p7, p5, p6, p9, PAGENUMBER, PAGESIZE);

    waitKey();

    /****************************** CONFIG TEST ******************************/
    printf("EEPROM fully writable: %s\r\n",
                                        (flash.isFullyWritable()?"YES":"NO"));
    printf("EEPROM half writable: %s\r\n",(flash.isHalfWritable()?"YES":"NO"));
    printf("EEPROM not writable: %s\r\n", (flash.isNotWritable()?"YES":"NO"));
    waitKey();

    /***************************** READING TEST *****************************/
    printf("READ 2 pages:\r\n");
    int num = flash.read(0, 2*PAGESIZE, buf);
    printMemory(buf,2*PAGESIZE);
    printf("%d bytes: %s\r\n", num, ((num==(2*PAGESIZE))?"OK":"NOT OK"));

    waitKey();

    /***************************** WRITING TEST *****************************/
    printf("WRITE first 20 bytes\r\n");
    memset(buf, 0x01, 20);
    num = flash.write(0, 20, buf);
    printf("%d bytes: %s\r\n\r\n", num, ((num==20)?"OK":"NOT OK"));

    printf("WRITE 10 bytes within a page (from addr 30)\r\n");
    memset(buf, 0x02, 10);
    num = flash.write(30, 10, buf);
    printf("%d bytes: %s\r\n\r\n", num, ((num==10)?"OK":"NOT OK"));

    printf("WRITE 30 bytes on two consecutive pages (from addr 50)\r\n");
    memset(buf, 0x03, 30);
    num = flash.write(50, 30, buf);
    printf("%d bytes: %s\r\n\r\n", num, ((num==30)?"OK":"NOT OK"));

    printf("READ 2 pages:\r\n");
    num = flash.read(0, 2*PAGESIZE, buf);
    printMemory(buf,2*PAGESIZE);
    printf("%d bytes: %s\r\n", num, ((num==(2*PAGESIZE))?"OK":"NOT OK"));

    waitKey();

    /*************************** ERASING PAGE TEST ***************************/
    printf("ERASE second page\r\n");
    num = flash.clearPage(1);
    printf("%d bytes: %s\r\n", num, ((num==PAGESIZE)?"OK":"NOT OK"));

    printf("READ 2 pages:\r\n");
    num = flash.read(0, 2*PAGESIZE, buf);
    printMemory(buf,2*PAGESIZE);
    printf("%d bytes: %s\r\n", num, ((num==(2*PAGESIZE))?"OK":"NOT OK"));

    waitKey();

    /************************** SERIALIZATION TEST **************************/
    struct test_serialization var_ser, var_deser;
    var_ser.data_1 = 0xAA;
    var_ser.data_2 = 0x7777;
    var_ser.data_3 = 0x33333333;
    printf("Serialize data at addr 20\r\n");
    num = flash.write(20, sizeof(test_serialization),
                                            (const unsigned char*)(&var_ser));
    printf("%d bytes: %s\r\n", num,
                            ((num==sizeof(test_serialization))?"OK":"NOT OK"));
    var_deser.data_1 = 0;
    var_deser.data_2 = 0;
    var_deser.data_3 = 0;
    printf("De-serialize data from addr 20\r\n");
    num = flash.read(20, sizeof(test_serialization),
                                                (unsigned char*)(&var_deser));
    printf("%d bytes: %s\r\n", num,
                            ((num==sizeof(test_serialization))?"OK":"NOT OK"));
    printf("Same data: %s\r\n", (((var_ser.data_1==var_deser.data_1) &&
                                  (var_ser.data_2==var_deser.data_2) &&
                                  (var_ser.data_3==var_deser.data_3))?
                                                               "OK":"NOT OK"));
    printf("READ 2 pages:\r\n");
    num = flash.read(0, 2*PAGESIZE, buf);
    printMemory(buf,2*PAGESIZE);
    printf("%d bytes: %s\r\n", num, ((num==(2*PAGESIZE))?"OK":"NOT OK"));

    waitKey();

    /************************** ERASING MEMORY TEST **************************/
    printf("ERASE memory\r\n");
    num = flash.clearMem();
    printf("%d bytes: %s\r\n", num, ((num==(EEPROM_SIZE))?"OK":"NOT OK"));

    printf("READ 2 pages:\r\n");
    num = flash.read(0, 2*PAGESIZE, buf);
    printMemory(buf,2*PAGESIZE);
    printf("%d bytes: %s\r\n", num, ((num==(2*PAGESIZE))?"OK":"NOT OK"));

    waitKey();

    /**************************** PROTECTION TEST ****************************/

    printf("Setting whole EEPROM as not writable\r\n");
    flash.setNotWritable();
    
    printf("EEPROM not writable: %s\r\n", (flash.isNotWritable()?"YES":"NO"));
    
    printf("WRITE first page\r\n");
    memset(buf, 0x00, PAGESIZE);
    num = flash.write(0, PAGESIZE, buf);
    printf("%d bytes: %s\r\n", num, ((num==(PAGESIZE))?"OK":"NOT OK"));

    printf("READ 2 pages:\r\n");
    num = flash.read(0, 2*PAGESIZE, buf);
    printMemory(buf,2*PAGESIZE);
    printf("%d bytes: %s\r\n", num, ((num==(2*PAGESIZE))?"OK":"NOT OK"));
    printf("DATA IS NOT CHANGED\r\n");

    printf("Setting whole EEPROM as writable\r\n");
    flash.setFullyWritable();

    printf("\r\nDONE\r\n");
    
    while(1);
}

void printMemory (unsigned char *s, int len)
{
    int i = 0;
    while (i<len) {
        printf("%02X ", s[i]);
        if (((++i)%32)==0)
            printf("\r\n");
    }
}

void waitKey ()
{
    char a;
    printf("Press a key to continue...\r\n\r\n");
    scanf("%c", &a); 
}
