/*
* Ser25lcxxx library
* Copyright (c) 2010 Hendrik Lipka
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#ifndef __SER25LCXXX_H__
#define __SER25LCXXX_H__

#include "mbed.h"


#define EEPROM_CMD_READ     0x03
#define EEPROM_CMD_WRITE    0x02
#define EEPROM_CMD_WRDI     0x04
#define EEPROM_CMD_WREN     0x06
#define EEPROM_CMD_RDSR     0x05
#define EEPROM_CMD_WRSR     0x01


#define EEPROM_SPI_SPEED    1000000

#define EEPROM_CLEAN_BYTE   0xff


/**
A class to read and write all 25* serial SPI eeprom devices from Microchip
(from 25xx010 to 25xx1024).

One needs to provide total size and page size, since this cannot be read from
the devices, and the page size differs even by constant size (look up the data
sheet for your part!)
*/
class Ser25LCxxx
{
    public:
    
        /**
            Create the handler class
            
            @param sck clock pin of the spi where the eeprom is connected
            @param si input pin of the spi where the eeprom is connected
            @param so output pin of the spi where the eeprom is connected
            @param enable the pin name for the port where /CS is connected
            @param pagenumber number of pages
            @param pagesize the size of each page
        */
        Ser25LCxxx(PinName sck, PinName si, PinName so, PinName enable,
                                                int pagenumber, int pagesize);
        
        /**
            Destroy the handler
        */
        ~Ser25LCxxx() {};
        
        /**
            Read from the eeprom memory
            
            @param startAdr the adress where to start reading
            @param len the number of bytes to read
            @param buf destination buffer
            @return number of read bytes
        */
        unsigned int read(unsigned int startAdr, unsigned int len,
                                                        unsigned char* buf);
        
        /**
            Writes the give buffer into the eeprom memory

            @param startAdr the eeprom adress where to start writing (it 
                                        doesn't have to match a page boundary)
            @param len number of bytes to write
            @param data data to write
            @return the number of written bytes
        */
        unsigned int write(unsigned int startAdr, unsigned int len,
                                                    const unsigned char* data);
        
        /**
            Fills the given page with EEPROM_CLEAN_BYTE

            @param pageNum the page to clear
            @return the number of written bytes (it should be the page size)
        */
        unsigned int clearPage(unsigned int pageNum);
        
        /**
            Fills the whole eeprom with EEPROM_CLEAN_BYTE
            
            @return the number of written bytes (it should be the eeprom size)
        */
        unsigned int clearMem();

        /**
            Return TRUE if a write is performing
        */
        bool isWriteInProgress() {
            return (readStatus()&0x01)!=0x00;
        }

        /**
            Return TRUE if the entire eeprom memory is writable
        */
        bool isFullyWritable() {
            return (readStatus()&0x0C)==0x00;
        }

        /**
            Return TRUE if only the first half of the eeprom memory is writable
        */
        bool isHalfWritable() {
            return (readStatus()&0x0C)==0x08;
        }

        /**
            Return TRUE if the eeprom memory is not writable
        */
        bool isNotWritable() {
            return (readStatus()&0x0C)==0x0C;
        }

        /**
            Set the eeprom memory not writable
        */
        void setNotWritable () {
            writeStatus(0x0C);
        }

        /**
            Set the eeprom memory fully writable
        */
        void setFullyWritable () {
            writeStatus(0x00);
        }
        
        /**
            Set only the first half of the eeprom memory as writable
        */
        void setHalfWritable() {
            writeStatus(0x08);
        }

    private:
    
        /**
            Write data within an eeprom page
            
            @param startAdr destination address of the eeprm memory
            @param len number of bytes to write
            @param data data to be written in the eeprom memory
            @return number of written bytes
        */
        unsigned int writePage(unsigned int startAdr, unsigned int len,
                                                    const unsigned char* data);
                                                    
        /**
            Read the status register of the eeprom memory
            
            @return status register
        */
        int readStatus();
        
        /**
            Write the status register
            
            @param val status register
        */
        void writeStatus(unsigned char val);

        /**
            Wait until a write procedure ends or an interval of 5ms is expired
            
            @return TRUE if the write procedure has successfully terminated,
                FALSE if the interval is expired
        */
        bool waitForWrite();
        
        /**
            Enable the write procedure
        */
        void enableWrite();

        SPI spi;
        DigitalOut cs;
        int size;
        int pageSize;
        int pageNumber;
};

#endif
