/*----------------------------------------------------------
A fictional IoT-enabled Ice cream machine dispenser arm state
The machine runs with an STM32F4 MCU; The Nucleo-F410RB is chosen for implementation, as it has serial communication, several digital outputs and 16 channel 12-bit ADC and more

Operation: Ice cream machine dispenser arm sensor provides 0-3.3 V (assume based on operating voltage 
    of MCU) to one ADC pin on STM32F4 MCU. Voltage is based on position of dispenser arm. 
- voltage > 1.65V implies the arm is pressed down 
- voltage < 1.65V implies the arm is released 
To prevent rapid switching, hysteresis is introduced with a counter function, if arm is pressed down for more than x seconds, the digital output will go high.

Requirements:
1. Read ADC value and print over serial port
2. Digital output pin high if voltage>1.65V
3. Digital output pin low if voltage <1.65V

Assumptions:
1. The code can run at a low speed as ice cream machines do not dispense ice cream quickly 
2. Low accuracy and low resolution due to nature of ice cream machine dispensing arms

Additional features
1. Temperature of ice cream mixture is important, assume temperature sensor can provide analogue 
    voltage to MCU
2. Level of ice cream mix in machine is important, assume a sensor can detect when the mixture drops 
    below a certain level, also be analogue voltage sensor output

ADC data on MCU: 
Value from 0.0 to 1.0, depending on input voltage between 0.0V and 
3.3V; ADC default settings
------------------------------------------------------------*/
#include "mbed.h"
Ticker readADC;     //Allows timed interval of reading data on ADC lines

//Serial pc(USBTX, USBRX); 
Serial device(D8, D2);  // tx, rx, default settings: 9600 8N1

AnalogIn DispArmIn(A0); //ADC input from dispensing arm
AnalogIn TempIn(A1);    //ADC input from temperature sensor
AnalogIn LevelIn(A2);   //ADC input from sensor detecting level

float DispArm_data;     //Dispenser arm data
float Temp_data;        //Temperature data
float Level_data;       //Mixture level data
bool ADC_ready;         //Indicate data has been read
int counter;            //Hysteresis for dispenser arm

DigitalOut dout(A4);    //Digital output to indicate if dispensing arm has crossed the halfway mark
DigitalOut led(LED2);   //Digital output for general use on LED

void ReadAnalog()
{
//Assume default AnalogIn read settings
    DispArm_data = DispArmIn;
    Temp_data  = TempIn;
    Level_data = LevelIn;
    
    if (DispArm_data>0.5f) //Use counter function to prevent rapid switching
    {//Trigger output pin; Add this section here as it is one of the critical requirements.
        counter++;
        if(counter>5)
        { 
            dout = 1;  //Arm is pressed down
            counter=0;
        }
    }
    else {
         dout=0;    //Arm is released
         counter=0;
    }
    ADC_ready=1;    //ADC have been read
}

int main() 
{
    //Initialise variables here
    ADC_ready=0;
    counter=0;    
    DispArm_data = 0.0;
    Temp_data  = 0.0;
    Level_data = 0.0;
    
    readADC.attach(&ReadAnalog, 0.5);   // the address of the function attached and the interval (0.5 seconds)
    device.printf("\nIce cream machine state:\n");
    
    while (1) {
        if (ADC_ready)
        {
           //pc.printf("%f \n\r",ADCdata);
            device.printf("Dispenser arm %f \n\r",DispArm_data); 
            if(Temp_data>0.7f)
            {   
                device.printf("Ice cream mixture not cooling: %f \n\r",Temp_data); 
            }
            if (Level_data<0.3f)
            {
                device.printf("Ice cream mixture level low: %f \n\r",Level_data); 
            }
            ADC_ready=0;
        } else {}
        wait (0.2f);
    }
}
