/* Copyright (c) 2012 Nordic Semiconductor. All Rights Reserved.
 *
 * The information contained herein is property of Nordic Semiconductor ASA.
 * Terms and conditions of usage are described in detail in NORDIC
 * SEMICONDUCTOR STANDARD SOFTWARE LICENSE AGREEMENT.
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 */

/** @file
 *
 * @defgroup ble_sdk_srv_bps Blood Pressure Service
 * @{
 * @ingroup ble_sdk_srv
 * @brief Blood Pressure Service module.
 *
 * @details This module implements the Blood Pressure Service.
 *
 *          If an event handler is supplied by the application, the Blood Pressure 
 *          Service will generate Blood Pressure Service events to the application.
 *
 * @note The application must propagate BLE stack events to the Blood Pressure Service
 *       module by calling ble_bps_on_ble_evt() from the from the @ref ble_stack_handler function.
 *
 * @note Attention! 
 *  To maintain compliance with Nordic Semiconductor ASA Bluetooth profile 
 *  qualification listings, this section of source code must not be modified.
 */

#ifndef BLE_BPS_H__
#define BLE_BPS_H__

#include <stdint.h>
#include <stdbool.h>
#include "ble.h"
#include "ble_srv_common.h"
#include "ble_date_time.h"

// Blood Pressure Feature bits
#define BLE_BPS_FEATURE_BODY_MOVEMENT_BIT               (0x01 << 0)         /**< Body Movement Detection Support bit. */
#define BLE_BPS_FEATURE_CUFF_FIT_BIT                    (0x01 << 1)         /**< Cuff Fit Detection Support bit. */
#define BLE_BPS_FEATURE_IRREGULAR_PULSE_BIT             (0x01 << 2)         /**< Irregular Pulse Detection Support bit. */
#define BLE_BPS_FEATURE_PULSE_RATE_RANGE_BIT            (0x01 << 3)         /**< Pulse Rate Range Detection Support bit. */
#define BLE_BPS_FEATURE_MEASUREMENT_POSITION_BIT        (0x01 << 4)         /**< Measurement Position Detection Support bit. */
#define BLE_BPS_FEATURE_MULTIPLE_BOND_BIT               (0x01 << 5)         /**< Multiple Bond Support bit. */

/**@brief Blood Pressure Service event type. */
typedef enum
{
    BLE_BPS_EVT_INDICATION_ENABLED,                                         /**< Blood Pressure value indication enabled event. */
    BLE_BPS_EVT_INDICATION_DISABLED,                                        /**< Blood Pressure value indication disabled event. */
    BLE_BPS_EVT_INDICATION_CONFIRMED                                        /**< Confirmation of a blood pressure measurement indication has been received. */
} ble_bps_evt_type_t;

/**@brief Blood Pressure Service event. */
typedef struct
{
    ble_bps_evt_type_t evt_type;                                            /**< Type of event. */
} ble_bps_evt_t;

// Forward declaration of the ble_bps_t type. 
typedef struct ble_bps_s ble_bps_t;

/**@brief Blood Pressure Service event handler type. */
typedef void (*ble_bps_evt_handler_t) (ble_bps_t * p_bps, ble_bps_evt_t * p_evt);

/**@brief SFLOAT format (IEEE-11073 16-bit FLOAT, defined as a 16-bit vlue with 12-bit mantissa and
 *        4-bit exponent. */
typedef struct
{
  int8_t  exponent;                                                         /**< Base 10 exponent, only 4 bits */
  int16_t mantissa;                                                         /**< Mantissa, only 12 bits */
} ieee_float16_t;

/**@brief Blood Pressure Service init structure. This contains all options and data
 *        needed for initialization of the service. */
typedef struct
{
    ble_bps_evt_handler_t        evt_handler;                               /**< Event handler to be called for handling events in the Blood Pressure Service. */
    ble_srv_cccd_security_mode_t bps_meas_attr_md;                          /**< Initial security level for blood pressure measurement attribute */
    ble_srv_security_mode_t      bps_feature_attr_md;                       /**< Initial security level for blood pressure feature attribute */
    uint16_t                     feature;                                   /**< Initial value for blood pressure feature */
} ble_bps_init_t;

/**@brief Blood Pressure Service structure. This contains various status information for
 *        the service. */
typedef struct ble_bps_s
{
    ble_bps_evt_handler_t        evt_handler;                               /**< Event handler to be called for handling events in the Blood Pressure Service. */
    uint16_t                     service_handle;                            /**< Handle of Blood Pressure Service (as provided by the BLE stack). */
    ble_gatts_char_handles_t     meas_handles;                              /**< Handles related to the Blood Pressure Measurement characteristic. */
    ble_gatts_char_handles_t     feature_handles;                           /**< Handles related to the Blood Pressure Feature characteristic. */
    uint16_t                     conn_handle;                               /**< Handle of the current connection (as provided by the BLE stack, is BLE_CONN_HANDLE_INVALID if not in a connection). */
    uint16_t                     feature;                                   /**< Value of Blood Pressure feature. */
} ble_bps_t;

/**@brief Blood Pressure Service measurement structure. This contains a Blood Pressure
 *        measurement. */
typedef struct ble_bps_meas_s
{
    bool                         blood_pressure_units_in_kpa;               /**< Blood Pressure Units Flag, 0=mmHg, 1=kPa */
    bool                         time_stamp_present;                        /**< Time Stamp Flag, 0=not present, 1=present. */
    bool                         pulse_rate_present;                        /**< Pulse Rate Flag, 0=not present, 1=present. */
    bool                         user_id_present;                           /**< User ID Flag, 0=not present, 1=present. */
    bool                         measurement_status_present;                /**< Measurement Status Flag, 0=not present, 1=present. */
    ieee_float16_t               blood_pressure_systolic;                   /**< Blood Pressure Measurement Compound Value - Systolic. */
    ieee_float16_t               blood_pressure_diastolic;                  /**< Blood Pressure Measurement Compound Value - Diastolic . */
    ieee_float16_t               mean_arterial_pressure;                    /**< Blood Pressure Measurement Compound Value - Mean Arterial Pressure. */
    ble_date_time_t              time_stamp;                                /**< Time Stamp. */
    ieee_float16_t               pulse_rate;                                /**< Pulse Rate. */
    uint8_t                      user_id;                                   /**< User ID. */
    uint16_t                     measurement_status;                        /**< Measurement Status. */
} ble_bps_meas_t;

/**@brief Function for initializing the Blood Pressure Service.
 *
 * @param[out]  p_bps       Blood Pressure Service structure. This structure will have to
 *                          be supplied by the application. It will be initialized by this function,
 *                          and will later be used to identify this particular service instance.
 * @param[in]   p_bps_init  Information needed to initialize the service.
 *
 * @return      NRF_SUCCESS on successful initialization of service, otherwise an error code.
 */
uint32_t ble_bps_init(ble_bps_t * p_bps, const ble_bps_init_t * p_bps_init);

/**@brief Function for handling the Application's BLE Stack events.
 *
 * @details Handles all events from the BLE stack of interest to the Blood Pressure Service.
 *
 * @param[in]   p_bps      Blood Pressure Service structure.
 * @param[in]   p_ble_evt  Event received from the BLE stack.
 */
void ble_bps_on_ble_evt(ble_bps_t * p_bps, ble_evt_t * p_ble_evt);

/**@brief Function for sending blood pressure measurement if indication has been enabled.
 *
 * @details The application calls this function after having performed a Blood Pressure
 *          measurement. If indication has been enabled, the measurement data is encoded and
 *          sent to the client.
 *
 * @param[in]   p_bps       Blood Pressure Service structure.
 * @param[in]   p_bps_meas  Pointer to new blood pressure measurement.
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
uint32_t ble_bps_measurement_send(ble_bps_t * p_bps, ble_bps_meas_t * p_bps_meas);

/**@brief Function for checking if indication of Blood Pressure Measurement is currently enabled.
 *
 * @param[in]   p_bps                  Blood Pressure Service structure.
 * @param[out]  p_indication_enabled   TRUE if indication is enabled, FALSE otherwise.
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
uint32_t ble_bps_is_indication_enabled(ble_bps_t * p_bps, bool * p_indication_enabled);

#endif // BLE_BPS_H__

/** @} */
