/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __ASYNCSERIAL_H__
#define __ASYNCSERIAL_H__

#include "mbed.h"
#include "FunctionPointer.h"
#include "AsyncSerial/FunctionPointerWithContext.h"

#define NDEBUG 1

#ifdef NDEBUG
#define DEBUG(...) /* nothing */
#else
#define DEBUG(...) { printf(__VA_ARGS__); }
#endif // NDEBUG

/*  Asynchronous Serial Driver

    This library can:
    Send a block of data (pointer + length) asynchronously
        * optional callback

    Receive a block of data asynchronously with timeout:
        * optional start condition (pointer + length)
        * optional stop condition (pointer + length)

    Call a function when a certain condition (pointer + length) is detected.
*/
class AsyncSerial : public SerialBase
{
public:
    /*  Receive Status:
        Found:    Stop condition caused the callback.
        Full:     The buffer is full.
        Timeout:  Time ran out before stop condition and buffer was filled.
    */
    typedef enum {
        RECEIVE_FOUND,
        RECEIVE_FULL,
        RECEIVE_TIMEOUT
    } receive_status_t;

    /*  Struct used in receive callback function to pass arguments.
    */
    typedef struct {
        uint8_t* buffer;
        uint16_t length;
        uint8_t status;
    } receive_params_t;

    //  Callback function type definitions.
    typedef void (*send_done_t)(void);
    typedef void (*receive_done_t)(receive_params_t* result);
    typedef void (*wait_done_t)(uint8_t status);

    /*  Constructor.

        Tx:   Transmit pin
        Rx:   Receive pin
        Rts:  Optional RTS flow control pin
        Cts:  Optional CTS flow control pin
    */
    AsyncSerial(PinName tx, PinName rx, PinName rts = NC, PinName cts = NC);
    virtual ~AsyncSerial();

    /*  Send block of data.

        send_done_t handler :  Callback function
        T* object/member    :  Callback object and member function

        const char* buffer  :  Pointer to block to be send.
        uint16_t    length  :  Length of block in bytes.
    */
    void send(send_done_t handler, const char* buffer, uint16_t length);

    template<typename T>
    void send(T* object, void (T::*member)(void),
              const char* buffer, uint16_t length)
    {
        sendHandler.attach(object, member);
        send(buffer, length);
    }

    template<int I>
    void send(send_done_t handler, const char (&buffer)[I])
    {
        send(handler, buffer, I);
    }

    template<typename T, int I>
    void send(T* object, void (T::*member)(void), const char (&buffer)[I])
    {
        send(object, member, buffer, I);
    }

    /*  Receive block of data.

        send_done_t handler      :  Callback function
        T* object/member         :  Callback object and member function

        const char* _receiveBuffer:  Pointer to receive buffer.
        uint16_t    _maxLength    :  Maximumm number of bytes to receive.

        const char* _conditionStartBuffer  :  Optional start condition for when to start
        uint16_t    _conditionStartLength  :  storing data in the receive buffer. Pass
                                              NULL pointer and 0 length to disable.

        const char* _conditionEndBuffer    :  Optional stop condition for when to stop
        uint16_t    _conditionEndLength    :  receiving data and signal callback function.
                                              Pass NULL pointer and 0 length to disable.

        uint32_t _timeoutMilli             :  Timeout in milliseconds.
    */
    void receive(receive_done_t handler,
                 uint8_t* _receiveBuffer, uint16_t _maxLength,
                 const char* _conditionStartBuffer, uint16_t _conditionStartLength,
                 const char* _conditionEndBuffer, uint16_t _conditionEndLength,
                 uint32_t _timeoutMilli);

    template<typename T>
    void receive(T* object, void (T::*member)(receive_params_t*),
                 uint8_t* _receiveBuffer, uint16_t _maxLength,
                 const char* _conditionStartBuffer, uint16_t _conditionStartLength,
                 const char* _conditionEndBuffer, uint16_t _conditionEndLength,
                 uint32_t _timeoutMilli)
    {
        receiveHandler.attach(object, member);

        /*  Signal callback function immediately if buffer and maxLength are invalid.
        */
        if ((_receiveBuffer == NULL) || (_maxLength == 0))
        {
            receiveResult.buffer = NULL;
            receiveResult.length = 0;
            receiveResult.status = AsyncSerial::RECEIVE_FULL;

            receiveHandler.call(&receiveResult);
        }
        else
        {
            receive(_receiveBuffer, _maxLength,
                    _conditionStartBuffer, _conditionStartLength,
                    _conditionEndBuffer, _conditionEndLength,
                    _timeoutMilli);
        }
    }


    /*  Register callback function to be called when a specific sequence is detected
        or the time has run out. No data is stored.

        send_done_t handler      :  Callback function
        T* object/member         :  Callback object and member function

        const char* _conditionEndBuffer    :  Sequence to be detected.
        uint16_t    _conditionEndLength    :

        uint32_t _timeoutMilli             :  Timeout in milliseconds.
    */
    void wait(wait_done_t handler,
              const char* _conditionEndBuffer, uint16_t _conditionEndLength,
              uint32_t _timeoutMilli);

    template<typename T>
    void wait(T* object, void (T::*member)(uint8_t),
              const char* _conditionEndBuffer, uint16_t _conditionEndLength,
              uint32_t _timeoutMilli)
    {
        waitHandler.attach(object, member);

        receive(NULL, 0,
                NULL, 0,
                _conditionEndBuffer, _conditionEndLength,
                _timeoutMilli);
    }

private:
    /*  Main send function.
    */
    void send(const char* buffer, uint16_t length);

    /*  Main receive function.

        All receive and wait calls are set up using this function.
    */
    void receive(uint8_t* _receiveBuffer, uint16_t _maxLength,
                 const char* _conditionStartBuffer, uint16_t _conditionStartLength,
                 const char* _conditionEndBuffer, uint16_t _conditionEndLength,
                 uint32_t _timeoutMilli);

    /*  Interrupt Service Routines for sending and receiving individual
        characters and timeout handling.
    */
    void putDone();
    void getReady();
    void receiveTimeout();

    /*  Common function for signaling receive and wait callback functions.
    */
    void getDone(uint8_t status);

    /*  Book keeping variables for sending data.
    */
    const char*       sendBuffer;
    uint16_t          sendLength;
    uint16_t          sendIndex;

    /*  Book keeping variables for receiving data.
    */
    uint8_t*          receiveBuffer;
    uint16_t          receiveMaxLength;
    uint16_t          receiveIndex;
    receive_status_t  receiveStatus;
    receive_params_t  receiveResult;
    Timeout           timeout;

    /*  Book keeping variables for detecting start and stop conditions.
    */
    bool              insideCondition;
    const char*       conditionStartBuffer;
    uint16_t          conditionStartLength;
    const char*       conditionEndBuffer;
    uint16_t          conditionEndLength;
    uint16_t          conditionIndex;

    /*  Objects storing callback function.
    */
    FunctionPointer                                 sendHandler;
    FunctionPointerWithContext<receive_params_t*>   receiveHandler;
    FunctionPointerWithContext<uint8_t>             waitHandler;
};

#endif // __ASYNCSERIAL_H__
