/*********************************************************************
This is an example sketch for our Monochrome SHARP Memory Displays

  Pick one up today in the adafruit shop!
  ------> http://www.adafruit.com/products/1393

These displays use SPI to communicate, 3 pins are required to  
interface

Adafruit invests time and resources providing this open source code, 
please support Adafruit and open-source hardware by purchasing 
products from Adafruit!

Written by Limor Fried/Ladyada  for Adafruit Industries.  
BSD license, check license.txt for more information
All text above, and the splash screen must be included in any redistribution
*********************************************************************/

 /*
  *  Modified by Marc PLOUHINEC 23/06/2015 for use in mbed
  */

#include <algorithm>
#include "mbed.h"
#include "Adafruit_SharpMem.h"

// any pins can be used
#define ENABLE p16
#define CS p15
#define MOSI p14
#define MISO p17
#define SCLK p13

Adafruit_SharpMem display(ENABLE, CS, MOSI, MISO, SCLK);

// Functions defined after main()
void testdrawchar(void);
void testdrawcircle(void);
void testfillrect(void);
void testdrawtriangle(void);
void testfilltriangle(void);
void testdrawroundrect(void);
void testfillroundrect(void); 
void testdrawrect(void);
void testdrawline();

int main() {
    // start & clear the display
    display.begin();
    display.setRotation(0);
    display.enableDisplay();
    display.clearDisplay();
    
    // draw a single pixel
    display.drawPixel(10, 10, BLACK);
    display.refresh();
    wait_ms(500);
    display.clearDisplay();
    
    // draw many lines
    testdrawline();
    wait_ms(500);
    display.clearDisplay();
    
    // draw rectangles
    testdrawrect();
    wait_ms(500);
    display.clearDisplay();
    
    // draw multiple rectangles
    testfillrect();
    display.refresh();
    wait_ms(500);
    display.clearDisplay();
    
    // draw a circle, 10 pixel radius
    display.fillCircle(display.width()/2, display.height()/2, 10, BLACK);
    display.refresh();
    wait_ms(500);
    display.clearDisplay();
    
    testdrawroundrect();
    display.refresh();  
    wait_ms(500);
    display.clearDisplay();
    
    testfillroundrect();
    display.refresh();
    wait_ms(500);
    display.clearDisplay();
    
    testdrawtriangle();
    display.refresh();
    wait_ms(500);
    display.clearDisplay();
    
    testfilltriangle();
    display.refresh();
    wait_ms(500);
    display.clearDisplay();
    
    // draw the first ~12 characters in the font
    testdrawchar();
    display.refresh();
    wait_ms(2000);
    display.clearDisplay();
    
    // text display tests
    display.setTextSize(1);
    display.setTextColor(BLACK);
    display.setCursor(0,0);
    display.printf("Hello, world!\n");
    display.setTextColor(WHITE, BLACK); // 'inverted' text
    display.printf("%d\n", 3.141592);
    display.setTextSize(2);
    display.setTextColor(BLACK);
    display.printf("0x%08x\n");
    display.refresh();
    wait_ms(2000);
    
    while(1) {
        // Screen must be refreshed at least once per second
        display.refresh();
        wait_ms(500);
    }
    
}

///

void testdrawchar(void) {
    display.setTextSize(1);
    display.setTextColor(BLACK);
    display.setCursor(0,0);
    
    for (uint8_t i=0; i < 168; i++) {
        if (i == '\n') continue;
        display.putc(i);
        //if ((i > 0) && (i % 14 == 0))
        //display.println();
    }    
    display.refresh();
}

void testdrawcircle(void) {
    for (uint8_t i=0; i<display.height(); i+=2) {
        display.drawCircle(display.width()/2-5, display.height()/2-5, i, BLACK);
        display.refresh();
    }
}

void testfillrect(void) {
    uint8_t color = 1;
    for (uint8_t i=0; i<display.height()/2; i+=3) {
        // alternate colors
        display.fillRect(i, i, display.width()-i*2, display.height()-i*2, color%2);
        display.refresh();
        color++;
    }
}

void testdrawtriangle(void) {
    for (uint16_t i=0; i<min(display.width(),display.height())/2; i+=5) {
        display.drawTriangle(display.width()/2, display.height()/2-i,
            display.width()/2-i, display.height()/2+i,
            display.width()/2+i, display.height()/2+i, BLACK);
        display.refresh();
    }
}

void testfilltriangle(void) {
    uint8_t color = BLACK;
    for (int16_t i=min(display.width(),display.height())/2; i>0; i-=5) {
        display.fillTriangle(display.width()/2, display.height()/2-i,
            display.width()/2-i, display.height()/2+i,
            display.width()/2+i, display.height()/2+i, color);
        if (color == WHITE) color = BLACK;
        else color = WHITE;
        display.refresh();
    }
}

void testdrawroundrect(void) {
    for (uint8_t i=0; i<display.height()/4; i+=2) {
        display.drawRoundRect(i, i, display.width()-2*i, display.height()-2*i, display.height()/4, BLACK);
        display.refresh();
    }
}

void testfillroundrect(void) {
    uint8_t color = BLACK;
    for (uint8_t i=0; i<display.height()/4; i+=2) {
        display.fillRoundRect(i, i, display.width()-2*i, display.height()-2*i, display.height()/4, color);
        if (color == WHITE) color = BLACK;
        else color = WHITE;
        display.refresh();
    }
}
   
void testdrawrect(void) {
    for (uint8_t i=0; i<display.height()/2; i+=2) {
        display.drawRect(i, i, display.width()-2*i, display.height()-2*i, BLACK);
        display.refresh();
    }
}

void testdrawline() {  
    for (uint8_t i=0; i<display.width(); i+=4) {
        display.drawLine(0, 0, i, display.height()-1, BLACK);
        display.refresh();
    }
    for (uint8_t i=0; i<display.height(); i+=4) {
        display.drawLine(0, 0, display.width()-1, i, BLACK);
        display.refresh();
    }
    wait_ms(250);
    
    display.clearDisplay();
    for (uint8_t i=0; i<display.width(); i+=4) {
        display.drawLine(0, display.height()-1, i, 0, BLACK);
        display.refresh();
    }
    for (int8_t i=display.height()-1; i>=0; i-=4) {
        display.drawLine(0, display.height()-1, display.width()-1, i, BLACK);
        display.refresh();
    }
    wait_ms(250);
    
    display.clearDisplay();
    for (int8_t i=display.width()-1; i>=0; i-=4) {
        display.drawLine(display.width()-1, display.height()-1, i, 0, BLACK);
        display.refresh();
    }
    for (int8_t i=display.height()-1; i>=0; i-=4) {
        display.drawLine(display.width()-1, display.height()-1, 0, i, BLACK);
        display.refresh();
    }
    wait_ms(250);
    
    display.clearDisplay();
    for (uint8_t i=0; i<display.height(); i+=4) {
        display.drawLine(display.width()-1, 0, 0, i, BLACK);
        display.refresh();
    }
    for (uint8_t i=0; i<display.width(); i+=4) {
        display.drawLine(display.width()-1, 0, i, display.height()-1, BLACK); 
        display.refresh();
    }
    wait_ms(250);
}