/*********************************************************************
This is an Arduino library for our Monochrome SHARP Memory Displays

  Pick one up today in the adafruit shop!
  ------> http://www.adafruit.com/products/1393

These displays use SPI to communicate, 3 pins are required to  
interface

Adafruit invests time and resources providing this open source code, 
please support Adafruit and open-source hardware by purchasing 
products from Adafruit!

Written by Limor Fried/Ladyada  for Adafruit Industries.  
BSD license, check license.txt for more information
All text above, and the splash screen must be included in any redistribution
*********************************************************************/

 /*
  *  Modified by Marc PLOUHINEC 23/06/2015 for use in mbed
  */
 
#include "Adafruit_SharpMem.h"

/**************************************************************************
    Sharp Memory Display Connector
    -----------------------------------------------------------------------
    Pin   Function        Notes
    ===   ==============  ===============================
      1   VIN             3.3-5.0V (into LDO supply)
      2   3V3             3.3V out
      3   GND
      4   SCLK            Serial Clock
      5   MOSI            Serial Data Input
      6   CS              Serial Chip Select
      9   EXTMODE         COM Inversion Select (Low = SW clock/serial)
      7   EXTCOMIN        External COM Inversion Signal
      8   DISP            Display On(High)/Off(Low)

 **************************************************************************/

#define SHARPMEM_BIT_WRITECMD   (0x80)
#define SHARPMEM_BIT_VCOM       (0x40)
#define SHARPMEM_BIT_CLEAR      (0x20)
#define TOGGLE_VCOM             do { _sharpmem_vcom = _sharpmem_vcom ? 0x00 : SHARPMEM_BIT_VCOM; } while(0);

static uint8_t sharpmem_buffer[(SHARPMEM_LCDWIDTH * SHARPMEM_LCDHEIGHT) / 8];

/* ************* */
/* CONSTRUCTORS  */
/* ************* */
Adafruit_SharpMem::Adafruit_SharpMem(PinName enable, PinName cs, PinName mosi, PinName miso_unused, PinName sclk, PinName _unused) :
    Adafruit_GFX(SHARPMEM_LCDWIDTH, SHARPMEM_LCDHEIGHT), spi(mosi, miso_unused, sclk, _unused), displayEnable(enable, 0), chipSelect(cs, 0), _sharpmem_vcom(SHARPMEM_BIT_VCOM) {
        
    spi.frequency(1000000);
    spi.format(8, 0);
}

void Adafruit_SharpMem::begin() {
    setRotation(2);
}

/* ************** */
/* PUBLIC METHODS */
/* ************** */

// 1<<n is a costly operation on AVR -- table usu. smaller & faster
static const uint8_t
  set[] = {  1,  2,  4,  8,  16,  32,  64,  128 },
  clr[] = { ~1, ~2, ~4, ~8, ~16, ~32, ~64, ~128 };

/**************************************************************************/
/*! 
    @brief Draws a single pixel in image buffer

    @param[in]  x
                The x position (0 based)
    @param[in]  y
                The y position (0 based)
*/
/**************************************************************************/
void Adafruit_SharpMem::drawPixel(int16_t x, int16_t y, uint16_t color) 
{
    if((x < 0) || (x >= _width) || (y < 0) || (y >= _height)) return;
    
    switch(rotation) {
        case 1:
            swap(x, y);
            x = _rawWidth  - 1 - x;
            break;
        case 2:
            x = _rawWidth  - 1 - x;
            y = _rawHeight - 1 - y;
            break;
        case 3:
            swap(x, y);
            y = _rawHeight - 1 - y;
            break;
    }
    
    if(color) {
        sharpmem_buffer[(y*SHARPMEM_LCDWIDTH + x) / 8] |= set[x & 7];
    } else {
        sharpmem_buffer[(y*SHARPMEM_LCDWIDTH + x) / 8] &= clr[x & 7];
    }
}

/**************************************************************************/
/*! 
    @brief Gets the value (1 or 0) of the specified pixel from the buffer

    @param[in]  x
                The x position (0 based)
    @param[in]  y
                The y position (0 based)

    @return     1 if the pixel is enabled, 0 if disabled
*/
/**************************************************************************/
uint8_t Adafruit_SharpMem::getPixel(uint16_t x, uint16_t y)
{
    if((x >= _width) || (y >= _height)) return 0; // <0 test not needed, unsigned
    
    switch(rotation) {
        case 1:
            swap(x, y);
            x = _rawWidth  - 1 - x;
            break;
        case 2:
            x = _rawWidth  - 1 - x;
            y = _rawHeight - 1 - y;
            break;
        case 3:
            swap(x, y);
            y = _rawHeight - 1 - y;
            break;
    }
    
    return sharpmem_buffer[(y*SHARPMEM_LCDWIDTH + x) / 8] & set[x & 7] ? 1 : 0;
}

/**************************************************************************/
/*! 
    @brief Clears the screen
*/
/**************************************************************************/
void Adafruit_SharpMem::clearDisplay() 
{
    memset(sharpmem_buffer, 0xff, (SHARPMEM_LCDWIDTH * SHARPMEM_LCDHEIGHT) / 8);
    // Send the clear screen command rather than doing a HW refresh (quicker)
    chipSelect = 1;
    spi.write(_sharpmem_vcom | SHARPMEM_BIT_CLEAR);
    spi.write(0x00);
    TOGGLE_VCOM;
    chipSelect = 0;
}

/**************************************************************************/
/*! 
    @brief Renders the contents of the pixel buffer on the LCD
*/
/**************************************************************************/
void Adafruit_SharpMem::refresh(void) 
{
    uint16_t i, totalbytes, currentline, oldline;  
    totalbytes = (SHARPMEM_LCDWIDTH * SHARPMEM_LCDHEIGHT) / 8;
    
    // Send the write command
    chipSelect = 1;
    spi.write(SHARPMEM_BIT_WRITECMD | _sharpmem_vcom);
    TOGGLE_VCOM;
    
    // Send the address for line 1
    oldline = currentline = 1;
    spi.write(bitReverse8(currentline));
    
    // Send image buffer
    for (i=0; i<totalbytes; i++)
    {
        spi.write(bitReverse8(sharpmem_buffer[i]));
        currentline = ((i+1)/(SHARPMEM_LCDWIDTH/8)) + 1;
        if(currentline != oldline)
        {
            // Send end of line and address bytes
            spi.write(0x00);
            if (currentline <= SHARPMEM_LCDHEIGHT)
            {
                spi.write(bitReverse8(currentline));
            }
            oldline = currentline;
        }
    }
    
    // Send another trailing 8 bits for the last line
    spi.write(0x00);
    chipSelect = 0;
}

/**************************************************************************/
/*! 
    @brief Turn on the screen
*/
/**************************************************************************/
void Adafruit_SharpMem::enableDisplay() {
    displayEnable = 1;
}

/**************************************************************************/
/*! 
    @brief Turn off the screen
*/
/**************************************************************************/
void Adafruit_SharpMem::disableDisplay() {
    displayEnable = 0;
}