//
// Filename: at42qt2120.cpp
//
// Atmel AT42QT2120 12 channel touch sensor.
//

#include "at42qt2120.h"

#include "log.h"

namespace HAL {

// This chip has a singe fixed I2C address.
const int ADDRESS =  0x1c;

// Chip Id hardcoded in the silicon.
const int AT42QT2120_ID = 0x3e;
    
AT42QT2120::AT42QT2120(I2C &i2cin)
: i2c(i2cin)
{
#ifdef VERBOSE
    if(Read((REG_AT42QT2120) QT_CHIP_ID) != AT42QT2120_ID)
        Log("AT42QT2120 bad chip id");
    else
        Log("AT42QT2120 chip id: Ok");
    
    char version[20];
    int ver = Read((REG_AT42QT2120) QT_FIRMWARE_VERSION);
    sprintf(version, "AT42QT2120 version: %u.%u", ver >> 4, ver & 0x0f);
    Log(version);
#endif

    for(uint8_t key = 0; key < 12; key++)
        SetKeyControl(key, false, 0, false, false);

    Calibrate();
}

bool AT42QT2120::ReadStatus(Status &status)
{
    char buffer[4];

    buffer[0] = Read(QT_DETECTION_STATUS);
    buffer[1] = Read(QT_KEY_STATUS);
    buffer[2] = Read(QT_KEY_STATUS2);
    buffer[3] = Read(QT_SLIDER_POSITION);

#ifdef VERBOSE
    char msg[32];
    sprintf(msg, "Touch: %x %x %x %x", buffer[0], buffer[1], buffer[2], buffer[3]);
#endif

    status.keyschanged = (buffer[0] & TDET) != 0;
    status.keys = buffer[2];
    status.keys = status.keys << 8;
    status.keys += buffer[1];

    status.sliderchanged = (buffer[0] & SDET) != 0;
    status.slider = buffer[3];

    return true;
}

bool AT42QT2120::Calibrate()
{
#ifdef VERBOSE
    Log("AT42QT2120 calibration start");
#endif

    bool status = Write(QT_CALIBRATE, 0xff);
    if(status)
    {
        while(Read(QT_DETECTION_STATUS) & 0x80);
    }

#ifdef VERBOSE
    if(status)
        Log("AT42QT2120 calibration finish");
    else
        Log("AT42QT2120 calibration fail");
#endif
    return status;
}

bool AT42QT2120::SetKeyControl(uint8_t key, bool guard, uint8_t group, bool gpo, bool enable)
{
    bool status = false;

    if(key < 12 && group < 4)
    {
        uint8_t config = 0;
        if(!enable)
            config |= CTRL_EN;
        if(gpo)
            config |= CTRL_GPO;
        if(group)
            config |= group << 2;
        if(guard)
            config |= CTRL_GUARD;

        uint8_t reg = key + 28; // 28 = QT_KEY0_CTRL
        status = Write((HAL::REG_AT42QT2120) reg, config);
    }

    return status;
}

void AT42QT2120::SetSliderOptions(SLIDERMODE_AT42QT2120 mode)
{
    switch(mode)
    {
        case SLIDER:
            Write(QT_SLIDER_OPTION, 0x80);
            break;

        case WHEEL:
            Write(QT_SLIDER_OPTION, 0xc0);
            break;

        default:
            Write(QT_SLIDER_OPTION, 0x0);
            break;
    }
}

int AT42QT2120::Read(REG_AT42QT2120 reg)
{
    if(i2c.write(ADDRESS << 1, (const char *) &reg, 1, true))
    {
        i2c.stop();
        return 0;
    }

    char data[2];
	i2c.read(ADDRESS << 1, data, 1);

    return data[0];
}

bool AT42QT2120::Write(REG_AT42QT2120 reg, uint8_t value)
{
    uint8_t regnum = reg;
    const char data[2] = { regnum, value };

    return i2c.write(ADDRESS << 1, data, 2) == 0;
}

} // End HAL namespace.

