//
// Filename: eink.h
//
// Flexbook eInk display driver definitions.
//

#ifndef EINK_H
#define EINK_H

#include "mbed.h"

/**
 * @brief Initialise the display hardware.
 * @param i2c The I2C to use for control data.
 * @param spi The SPI to use for data.
 * @param spics The SPI chip select line (the Epson driver manually sets it).
 * @param hven The display enable line.
 * @param rst The Epson controller rest line.
 * @param hvpok The power OK signal.
 */
void InitDisplay(I2C &i2c, SPI &spi, DigitalOut &spics, DigitalOut &hven, DigitalOut &rst, DigitalIn &hvpok);
void WriteImage(const int numimage);

struct pl_epdc;

/**
 * @brief Write a colour to the entire screen.
 * @param epdc The eInk device.
 * @param colour The colour to write.
 */
void WriteColour(pl_epdc &epdc, uint8_t colour);

/**
 * @brief Write an image to the screen.
 * @param file The file to write e.g. "/myfile.pgm".
 */
void WriteImage(const char *file);

/**
 * @brief Write a partial image to the display.
 * @param file The file to write e.g. "/myfile.pgm".
 * @param xd X display coordinate.
 * @param yd Y display coordinate.
 * @param xi X image coordinate.
 * @param yi Y image coordinate.
 * @param w Width of image.
 * @param h Height of image.
 */
void WritePartImage(const char *filename, int xd, int yd, int xi, int yi, int w, int h);

/**
 * @brief Update the eInk display.
 */
void UpdateDisplay();

/**
 * @brief Display hardware settings. Use the VERBOSE preprocessor define to see more information.
 */
void DisplayHardwareSettings();

/**
 * @ brief Get a handle to the display device information.
 * This information is only valid after InitDiaplay() has been called.
 * Use this to pass to WritePicture and WriteColour.
 * @return The eInk device information struct.
 */
pl_epdc &Get_epdc();

#endif // EINK_H

