//
// Filename: mcp23s17.h
//
// Flexbook Hardware Abstraction Layer.
//

#ifndef MCP23S17_H
#define MCP23S17_H

#include "mbed.h"

namespace HAL {

/**
 * @brief MCP23S17 register addresses in bank 0 mode.
 */
enum REG_MCP23S17
{
IODIRA   = 0x00,
IODIRB   = 0x01,
IPOLA    = 0x02,
IPOLB    = 0x03,
GPINTENA = 0x04,
GPINTENB = 0x05,
DEFVALA  = 0x06,
DEFVALB  = 0x07,
INTCONA  = 0x08,
INTCONB  = 0x09,
IOCON1   = 0x0a,
IOCON2   = 0x0b,
GPPUA    = 0x0c,
GPPUB    = 0x0d,
INTFA    = 0x0e,
INTFB    = 0x0f,
INTCAPA  = 0x10,
INTCAPB  = 0x11,
GPIOA    = 0x12,
GPIOB    = 0x13,
OLATA    = 0x14,
OLATB    = 0x15
};

/**
 * @brief Microchip MCP23S17 encapsulation.
 * This chip is an SPI connected I/O expander.
 *
 * Example usage:
 <pre>
    SPI spi(p5, p6, p7);
    DigitalOut cs(p19);
    MCP23S17 mcp23s17(0x00, spi, cs);
    ...
    mcp23s17.Write(GPIOB, 0xaa);
 </pre>
 *
 * The MCP23S17 uses pins 5, 6, 7 (SPI MOSI, MISO, SCK) and 19 (CS) and
 * the write command sets GPIOB to 0xaa. The chip address is 0x00.
 *
 * Note that this code example does not cover the setup of the chip
 * configuration registers.
 */
class MCP23S17
{
public:
    /**
     * @brief Constructor.
     * @param address The SPI address offset of the chip.
     * @param SPI the SPI to use for communication.
     * @param cs The chip select pin.
     */
    MCP23S17(int address, SPI &spi, DigitalOut &cs);

    /**
     * @brief Read a register.
     * @param address The SPI address of the MCP23S17.
     * @param reg The register to read.
     * @return The register value.
     */
    int Read(REG_MCP23S17 reg);

    /**
     * @brief Write a register.
     * @param address The SPI address of the MCP23S17.
     * @param reg The register to write.
     * @param value The value to write to the register.
     */
    void Write(REG_MCP23S17 reg, int value);

private:
    int address;
    SPI &spi;
    DigitalOut &cs;
};

} // End HAL namespace.

#endif // MCP23S17_H
