//
// Filename: hal.cpp
//
// Flexbook Hardware Abstraction Layer.
//

#include "hal.h"

#include "mcp23s17.h"

#include "mbed.h"

#include <stdio.h>

// Initialisation values for MCP23S17 registers.
uint8_t MCP23S17value[0x16] =
{
0x00, // IODIRA
0x00, // IODIRB
0xff, // IPOLA
0x00, // IPOLB
0x00, // GPINTENA
0x00, // GPINTENB
0x00, // DEFVALA
0x00, // DEFVALB
0x00, // INTCONA
0x00, // INTCONB
0x00, // IOCON1
0x00, // IOCON2
0x00, // GPPUA
0x00, // GPPUB
0x00, // INTFA
0x00, // INTFB
0x00, // INTCAPA
0x00, // INTCAPB
0x00, // GPIOA
0x00, // GPIOB
0x00, // OLATA
0x00, // OLATB
};

namespace HAL {

MCP23S17 &GetMCP23S17()
{
    // The MCP23S17 uses pins 5, 6, 7 (SPI MOSI, MISO, SCK) and 19 (CS).
    static SPI spi(p5, p6, p7);
    static DigitalOut cs(p19);
    static MCP23S17 mcp23s17(0x00, spi, cs);

    return mcp23s17;
}

// Initialise the HAL.
void Initialise()
{
    // Setup MCP23S17 I/O expander.
    for(unsigned int reg = 0; reg < sizeof(MCP23S17value); reg++)
        GetMCP23S17().Write((REG_MCP23S17) reg, MCP23S17value[reg]);
    
    GetMCP23S17().Write(GPIOA, 0);
    GetMCP23S17().Write(GPIOB, 0);
}

void EnablePowerToPages(Flexbook::PageType pages)
{
    // Power on the board(s).
    GetMCP23S17().Write(GPIOB, pages);

    // Wait for the power to stabilize.
    wait(0.1);
    
    // Enable the I2C connection.
    GetMCP23S17().Write(GPIOA, pages);   
}

Flexbook::PageType GetPageType()
{
    using namespace Flexbook;

    static PageType lastype = Flexbook::PageType_PageNull;   

    //printf("%x\n", GetMCP23S17().Read(GPIOA));

    switch(GetMCP23S17().Read(GPIOA))
    {
        case 0:
            break;
            
        //case PageType_PageSensor:    
        case 0xFE:
            lastype = PageType_PageSensor;
            break;

        case 0xF7:
            lastype = PageType_PageDice;
            break;

        case PageType_PageOLED:
            lastype = PageType_PageOLED;
            break;

        //case PageType_PageTouch:
        case 0xFD:
            lastype = PageType_PageTouch;
            break;

        case PageType_PageFlexEnable:
            lastype = PageType_PageFlexEnable;
            break;

        case PageType_PageNFC:
            lastype = PageType_PageNFC;
            break;

        default:
            break;
    }
    
    return lastype;
}

} // End HAL namespace.




