//
// Filename: at42qt2120.h
//
// Atmel AT42QT2120 12 channel touch sensor.
//

#ifndef AT42QT2120_H
#define AT42QT2120_H

#include "mbed.h"

namespace HAL {

/**
 * @brief AT42QT2120 registers.
 */
enum REG_AT42QT2120
{
QT_CHIP_ID = 0,
QT_FIRMWARE_VERSION,
QT_DETECTION_STATUS,
QT_KEY_STATUS,
QT_KEY_STATUS2,
QT_SLIDER_POSITION = 5,
QT_CALIBRATE,
QT_RESET,
QT_LP,
QT_TTD,
QT_ATD,
QT_DI,
QT_TRD,
QT_DHT,
QT_SLIDER_OPTION,
QT_CHARDE_TIME,
QT_KEY0_DTHR,
QT_KEY1_DTHR,
QT_KEY2_DTHR,
QT_KEY3_DTHR,
QT_KEY4_DTHR,
QT_KEY5_DTHR,
QT_KEY6_DTHR,
QT_KEY7_DTHR,
QT_KEY8_DTHR,
QT_KEY9_DTHR,
QT_KEY10_DTHR,
QT_KEY11_DTHR,	
QT_KEY0_CTRL,
QT_KEY1_CTRL,
QT_KEY2_CTRL,
QT_KEY3_CTRL,
QT_KEY4_CTRL,
QT_KEY5_CTRL,
QT_KEY6_CTRL,
QT_KEY7_CTRL,
QT_KEY8_CTRL,
QT_KEY9_CTRL,
QT_KEY10_CTRL,
QT_KEY11_CTRL,	
QT_KEY0_PULSE_SCALE,
QT_KEY1_PULSE_SCALE,
QT_KEY2_PULSE_SCALE,
QT_KEY3_PULSE_SCALE,
QT_KEY4_PULSE_SCALE,
QT_KEY5_PULSE_SCALE,
QT_KEY6_PULSE_SCALE,
QT_KEY7_PULSE_SCALE,
QT_KEY8_PULSE_SCALE,
QT_KEY9_PULSE_SCALE,
QT_KEY10_PULSE_SCALE,
QT_KEY11_PULSE_SCALE,
QT_KEY0_SIGNAL,
QT_KEY1_SIGNAL = 54,
QT_KEY2_SIGNAL = 56,
QT_KEY3_SIGNAL = 58,
QT_KEY4_SIGNAL = 60,
QT_KEY5_SIGNAL = 62,
QT_KEY6_SIGNAL = 64,
QT_KEY7_SIGNAL = 66,
QT_KEY8_SIGNAL = 68,
QT_KEY9_SIGNAL = 70,
QT_KEY10_SIGNAL = 72,
QT_KEY11_SIGNAL = 74,	
QT_KEY0_REFERENCE = 76,
QT_KEY1_REFERENCE = 78,
QT_KEY2_REFERENCE = 80,
QT_KEY3_REFERENCE = 82,
QT_KEY4_REFERENCE = 84,
QT_KEY5_REFERENCE = 86,
QT_KEY6_REFERENCE = 88,
QT_KEY7_REFERENCE = 90,
QT_KEY8_REFERENCE = 92,
QT_KEY9_REFERENCE = 94,
QT_KEY10_REFERENCE = 96,
QT_KEY11_REFERENCE = 98,
};

enum SLIDERMODE_AT42QT2120
{
NONE,
SLIDER,
WHEEL
};

// Control register bit positions.
const uint8_t CTRL_EN = (1 << 0);
const uint8_t CTRL_GPO = (1 << 1);
const uint8_t CTRL_GUARD = (1 << 4);
    
const int KEY0  = (1 << 0);
const int KEY1  = (1 << 1);
const int KEY2  = (1 << 2);
const int KEY3  = (1 << 3);
const int KEY4  = (1 << 4);
const int KEY5  = (1 << 5);
const int KEY6  = (1 << 6);
const int KEY7  = (1 << 7);
const int KEY8  = (1 << 8);
const int KEY9  = (1 << 9);
const int KEY10 = (1 << 10);
const int KEY11 = (1 << 11);

const uint8_t TDET = (1 << 0);
const uint8_t SDET = (1 << 1);

/**
 * @brief Atmel AT42QT2120 encapsulation.
 * This chip is a 12 channel touch sensor.
 *
 * Example usage:
 <pre>
    I2C i2c(p28, p27);
    AT42QT2120 at42qt2120(i2c);
    ...
    at42qt2120.Write(QT_RESET, 0);
 </pre>
 *
 * The AT42QT2120 uses pins 28 and 27 (SDA and SCL) for communication
 * and the write command resets the chip.
 *
 * The chip address of the AT42QT2120 is fixed at 0x1c.
 *
 * Note that this code example does not cover the setup of the chip
 * configuration registers.
 */
class AT42QT2120
{
public:
    /**
     * @brief Constructor.
     * @param I2C the I2C to use for communication.
     */
    AT42QT2120(I2C &i2c);

    /**
     * @brief Calibrate any enabled touch sensors.
     * @return True if calibration was successfull.
     */
    bool Calibrate();

    /**
     * @brief Set the key control bits.
     * See section 5.17 of the datasheet for more information.
     * All keys are initially disabled in the constructor so this
     * function must be called to configure the keys before use.
     * After calling this function you should call Calibrate().
     * @param key The key to configure.
     * @param guard Set to true if this is a guard channel.
     * @param group The key group (AKS) number.
     * @param gpu Set to true for high output.
     * @param enable Enable this key. Unused keys should be disabled.
     */
    bool SetKeyControl(uint8_t key, bool guard, uint8_t group, bool gpo, bool enable);

    /**
     * @brief Set the slider options.
     * @param Set to SLIDER, WHEEL or NONE to set the required mode.
     */
    void SetSliderOptions(SLIDERMODE_AT42QT2120 mode);

    /**
     * @brief Touch sensor status.
     */
    struct Status
    {
        /**
         * @brief True if the keys have changed.
         */
        bool     keyschanged;

        /**
         * @brief The key change bit mask.
         * Use KEYn to extract the individual key status.
         */
        uint16_t keys;

        /**
         * @brief True if the slider has changed.
         */
        bool     sliderchanged;

        /**
         * @brief The slider position.
         */
        bool     slider;
    };

    /**
     * @brief Read the status of the touch sensor.
     * @param status Pointer to hold the status information.
     * @return True if the transfer completes successfully.
     */
    bool ReadStatus(Status &status);

    /**
     * @brief Read a register.
     * @param reg The register to read.
     * @return The register value.
     */
    int Read(REG_AT42QT2120 reg);

    /**
     * @brief Write a register.
     * @param reg The register to write.
     * @param value The value to write to the register.
     * @return True on success.
     */
    bool Write(REG_AT42QT2120 reg, uint8_t value);

private:
    I2C &i2c;
    Status status;
};

} // End HAL namespace.

#endif // MCP23S17_H
